from paysys.sre.tools.monitorings.lib.util.helpers import (
    solomon_check,
    solomon_equal_selector,
    join_solomon_program_code,
)
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import (
    OutputMixin,
    ProgramMixin,
    AlertMixin,
    compose_program,
)


TOO_MANY_RETRIES_SUM = 'too_many_retries_sum'
FAIL_SUM = 'fail_sum'

STATUSES = {'too_many_retries', 'fail'}
SELECTOR_NAMES = {TOO_MANY_RETRIES_SUM, FAIL_SUM}


def check(
    alert_name,
    alert_annotation,
    selector_conf,
    alert_conf,
    notification,
    window_secs,
    no_points_policy="NO_POINTS_MANUAL",
    resolved_empty_policy="RESOLVED_EMPTY_OK",
):
    class QueueStatusSelector(OutputMixin, ProgramMixin):
        def __init__(self, cluster, service, queue, classname="*"):
            self.cluster = cluster
            self.service = service
            self.classname = classname
            self.queue = queue
            self.statuses = STATUSES

        @property
        def output_names(self):  # type: () -> set[str]
            return SELECTOR_NAMES

        def program(self):  # type: () -> str
            program = []
            for status in self.statuses:
                program.append(
                    'let {status}_data = series_sum({selector}+constant_line(0))'.format(
                        status=status,
                        selector=solomon_equal_selector(
                            cluster=self.cluster,
                            host='cluster',
                            sensor='export.status',
                            service=self.service,
                            classname=self.classname,
                            queue=self.queue,
                            status=status,
                        ),
                    )
                )
                program.append(
                    'let {status}_sum = sum({status}_data)'.format(status=status)
                )
            return join_solomon_program_code(*program)

    class QueueStatusAlertBlock(AlertMixin):
        def __init__(self, alarm_size, **kwargs):
            super(QueueStatusAlertBlock, self).__init__(**kwargs)
            self.alarm_size = alarm_size

        @property
        def expected_inputs(self):
            return SELECTOR_NAMES

        def ok_if(self):  # type: () -> str
            return "{too_many_retries_sum} < {alarm_size} && {fail_sum} < {alarm_size}".format(
                too_many_retries_sum=self.in_var(TOO_MANY_RETRIES_SUM),
                fail_sum=self.in_var(FAIL_SUM),
                alarm_size=self.alarm_size,
            )

        def alarm_if(self):  # type: () -> str
            return "{too_many_retries_sum} >= {alarm_size} || {fail_sum} >= {alarm_size}".format(
                too_many_retries_sum=self.in_var(TOO_MANY_RETRIES_SUM),
                fail_sum=self.in_var(FAIL_SUM),
                alarm_size=self.alarm_size,
            )

    selector = QueueStatusSelector(**selector_conf)
    alert_block = QueueStatusAlertBlock(
        too_many_retries_sum=selector.too_many_retries_sum,
        fail_sum=selector.fail_sum,
        **alert_conf
    )

    solomon_project = "balance"

    expr = solomon_expression_custom(
        project_id=solomon_project,
        program_str=compose_program(selector, alert_block),
        annotations=alert_annotation,
        resolved_empty_policy=resolved_empty_policy,
        no_points_policy=no_points_policy,
        window_secs=window_secs
    )

    return solomon_check(
        alert_name,
        expr,
        notification,
        {"aggregator_kwargs": None},
    )
