from paysys.sre.tools.monitorings.lib.util.helpers import solomon_check
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import (
    OutputMixin,
    ProgramMixin,
    AlertMixin,
    compose_program,
    TimeModifier,
)


solomon_project = "balance"

common_annotation = {
    "description": "Current value: {{expression.count_500}}",
}


def status_500(alert_name, selector_conf, time_modifier_conf, alert_conf, notification):
    class Count500Selector(OutputMixin, ProgramMixin):
        def __init__(self, cluster, service, include_code=None, exclude_code=None):
            self.cluster = cluster
            self.service = service
            self.exclude_code = exclude_code if exclude_code is not None else []
            self.include_code = include_code if include_code is not None else []

        @property
        def output_names(self):
            return {"count_500"}

        def program(self):
            included = ", ".join(
                'status_code="{}"'.format(code) for code in self.include_code
            )
            excluded = ", ".join(
                'status_code!="{}"'.format(code) for code in self.exclude_code
            )
            query = (
                '{{cluster="{cluster}", service="{service}", host="cluster", '
                'method_name="ALL_METHODS", sensor="method.status_code", '
                "{status_codes} }}".format(
                    cluster=self.cluster,
                    service=self.service,
                    status_codes=", ".join(filter(bool, [included, excluded])),
                )
            )
            return "\n".join(
                [
                    "let data = series_sum({query});".format(query=query),
                    "let count_500 = sum(data);",
                ]
            )

    class Count500AlertBlock(AlertMixin):
        def __init__(self, base_limit, **kwargs):
            super(Count500AlertBlock, self).__init__(**kwargs)
            self.base_limit = base_limit

        @property
        def expected_inputs(self):
            return {"count_500", "modifier"}

        def ok_if(self):  # type: () -> str
            return "{data} == 0".format(data=self.in_var("count_500"))

        def alarm_if(self):  # type: () -> str
            return "{data} > {base} + {modifier}".format(
                data=self.in_var("count_500"),
                modifier=self.in_var("modifier"),
                base=self.base_limit,
            )

    selector = Count500Selector(**selector_conf)
    limit_modifier = TimeModifier(**time_modifier_conf)
    alert_block = Count500AlertBlock(
        count_500=selector.count_500,
        modifier=limit_modifier.time_modifier,
        **alert_conf
    )
    expr = solomon_expression_custom(
        project_id=solomon_project,
        program_str=compose_program(selector, limit_modifier, alert_block),
        annotations=common_annotation,
    )

    return solomon_check(
        alert_name,
        expr,
        notification,
        {"aggregator_kwargs": None},
    )
