from paysys.sre.tools.monitorings.lib.util.helpers import solomon_check
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import (
    InputMixin,
    OutputMixin,
    ProgramMixin,
    TimeModifier,
    QueueAlertBlock,
    compose_program,
)

solomon_project = "balance"


def max_notify_delay(cluster, alert_name, default_limit, notification):
    class TonSelector(OutputMixin, ProgramMixin):
        def __init__(self, metric_cluster):
            self.cluster = metric_cluster

        @property
        def output_names(self):  # type: () -> set[str]
            return {"min_delay"}

        def program(self):  # type: () -> str
            query = (
                '{{cluster="{cluster}", service="simple_brother", '
                'metric_group="KPI_SLA", name="max_notif_delay"}}'
            ).format(cluster=self.cluster)
            return "\n".join(
                [
                    "let delay = {query};".format(
                        query=query,
                    ),
                    "let min_delay = trunc(min(delay) * 100) / 100;",
                ]
            )

    class DynamicLimit(InputMixin, OutputMixin, ProgramMixin):
        def __init__(self, default_limit, **kwargs):
            super(DynamicLimit, self).__init__(**kwargs)
            self.default_limit = default_limit

        @property
        def expected_inputs(self):  # type: () -> set[str]
            return {"modifier"}

        @property
        def output_names(self):  # type: () -> set[str]
            return {"dyn_limit"}

        def program(self):  # type: () -> str
            return "let dyn_limit = {default_limit} + {modifier};".format(
                default_limit=self.default_limit,
                modifier=self.in_var("modifier"),
            )

    selector = TonSelector(cluster)
    modifier = TimeModifier(
        utc_start=2,
        utc_end=23,
        in_range_modifier=4,
        out_range_modifier=0,
    )
    dynamic_limit = DynamicLimit(
        default_limit=default_limit,
        modifier=modifier.time_modifier,
    )
    alert_block = QueueAlertBlock(
        alarm_size=dynamic_limit.dyn_limit,
        queue_size=selector.min_delay,
    )

    annotations = {
        "description": "TON max notify delay: {notify_delay}, limit: {dyn_limit}".format(
            notify_delay=selector.expr_min_delay,
            dyn_limit=dynamic_limit.expr_dyn_limit,
        )
    }

    expr = solomon_expression_custom(
        project_id=solomon_project,
        program_str=compose_program(selector, modifier, dynamic_limit, alert_block),
        annotations=annotations,
    )

    return solomon_check(
        alert_name,
        expr,
        notification,
        {"aggregator_kwargs": None},
    )


def queue_len(cluster, alert_name, alarm_size, notification):
    class TonQueueLenSelector(OutputMixin, ProgramMixin):
        def __init__(self, metric_cluster):
            self.cluster = metric_cluster

        @property
        def output_names(self):  # type: () -> set[str]
            return {"ton_queue"}

        def program(self):  # type: () -> str
            query = (
                '{{cluster="{cluster}", service="simple_brother", '
                'metric_group="TON", host="greedaggr_prod", schema="BO"}}'
            ).format(cluster=self.cluster)
            return "\n".join(
                [
                    "let ton_queue = min(series_sum({query}));".format(
                        query=query,
                    ),
                ]
            )

    selector = TonQueueLenSelector(cluster)
    alert_block = QueueAlertBlock(alarm_size=alarm_size, queue_size=selector.ton_queue)

    annotations = {
        "description": "TON queue: {queue_len}".format(
            queue_len=selector.expr_ton_queue
        )
    }

    expr = solomon_expression_custom(
        project_id=solomon_project,
        program_str=compose_program(selector, alert_block),
        annotations=annotations,
    )

    return solomon_check(
        alert_name,
        expr,
        notification,
        {"aggregator_kwargs": None},
    )
