from paysys.sre.tools.monitorings.lib.util.helpers import solomon_check
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import (
    OutputMixin,
    ProgramMixin,
    AlertMixin,
    compose_program,
    TimeModifier,
)

solomon_project = "balance"

common_annotation = {
    "description": "Idle percent is {{expression.fixed_percent}}",
    "idle_percent": "{{expression.fixed_percent}}",
}


def worker_idle(
    alert_name, selector_conf, time_modifier_conf, alert_conf, notification
):
    class IdlePercentSelector(OutputMixin, ProgramMixin):
        def __init__(self, cluster, service):
            self.service = service
            self.cluster = cluster

        @property
        def output_names(self):  # type: () -> set[str]
            return {"idle_percent"}

        def program(self):  # type: () -> str
            query = (
                '{{cluster="{cluster}", service="{service}", '
                'host!="cluster", sensor="worker.idle_percent"}}'.format(
                    service=self.service,
                    cluster=self.cluster,
                )
            )
            return "\n".join(
                [
                    "let data = series_avg({query});".format(query=query),
                    "let idle_percent = avg(data);",
                    "let fixed_percent = to_fixed(idle_percent, 2);",
                ]
            )

    class IdlePercentAlertBlock(AlertMixin):
        def __init__(self, alarm_base, warn_base, **kwargs):
            super(IdlePercentAlertBlock, self).__init__(**kwargs)
            self.alarm_base = alarm_base
            self.warn_base = warn_base

        @property
        def expected_inputs(self):
            return {"idle_percent", "modifier"}

        def cond(self, base_limit):  # type: (int) -> str
            return "{idle_percent} < {base_limit} - {modifier}".format(
                idle_percent=self.in_var("idle_percent"),
                modifier=self.in_var("modifier"),
                base_limit=base_limit,
            )

        def alarm_if(self):  # type: () -> str
            return self.cond(self.alarm_base)

        def warn_if(self):  # type: () -> str
            return self.cond(self.warn_base)

    selector = IdlePercentSelector(**selector_conf)
    limit_modifier = TimeModifier(**time_modifier_conf)
    alert_block = IdlePercentAlertBlock(
        idle_percent=selector.idle_percent,
        modifier=limit_modifier.time_modifier,
        **alert_conf
    )
    expr = solomon_expression_custom(
        project_id=solomon_project,
        program_str=compose_program(selector, limit_modifier, alert_block),
        annotations=common_annotation,
        resolved_empty_policy="RESOLVED_EMPTY_ALARM",
        no_points_policy="NO_POINTS_ALARM",
    )

    return solomon_check(
        alert_name,
        expr,
        notification,
        {"aggregator_kwargs": None},
    )
