import juggler_sdk

from paysys.sre.tools.monitorings.lib.checks.active.http import https, http
from paysys.sre.tools.monitorings.lib.checks.awacs import l7_monitoring
from paysys.sre.tools.monitorings.lib.checks.base import unreachable
from paysys.sre.tools.monitorings.lib.util.helpers import merge, check, gen_children_deploy, flaps, solomon_check
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import OutputMixin, ProgramMixin, \
    UpperThresholdAlertBlock, compose_program


def get_checks(children_api, children_tasks, children_sequencer, children_lbexporter, balancers):
    return merge(
        l7_monitoring(balancers),
        check('certs'),
        check('unispace'),
        unreachable,
        https('api', 443, ok_codes=[200, 301, 302], crit=0, headers={"Host": "localhost"}),
        check('api', gen_children_deploy(children_api, 'api')),
        http('tasks', 8080, ok_codes=[200, 301, 302], crit=0, headers={"Host": "localhost"}),
        check('tasks', gen_children_deploy(children_tasks, 'tasks')),
        http('lbexporter', 8080, ok_codes=[200, 301, 302], crit=0, headers={"Host": "localhost"}),
        check('lbexporter', gen_children_deploy(children_lbexporter, 'lbexporter')),
        http('sequencer', 8080, ok_codes=[200, 301, 302], crit=0, headers={"Host": "localhost"}),
        check('sequencer', gen_children_deploy(children_sequencer, 'sequencer')),
        *[
            check(
                x, gen_children_deploy(children_api + children_tasks + children_sequencer + children_lbexporter, x),
                flaps(180, 600),
            )
            for x in [
                'UNREACHABLE',
                'certs',
                'unispace',
            ]
        ]
    )


def lbexporter_db_deadlock(project_id, env, warn_threshold_seconds, crit_threshold_seconds, *args):
    class DBDeadlockSelector(OutputMixin, ProgramMixin):
        @property
        def output_names(self):  # type: () -> set[str]
            return {'read_lag'}

        def program(self):  # type: () -> str
            query = (
                'series_max({{cluster="{env}", service="lbexporter", '.format(env=env) +
                'host!="cluster", sensor="batch.read.timestamp.last"})'
            )
            return "\n".join([
                "let last_db_read = {query};".format(query=query),
                "let read_lag = time_interval_end() - max(last_db_read);"
            ])

    selector = DBDeadlockSelector()
    alert_block = UpperThresholdAlertBlock(
        alarm_limit=crit_threshold_seconds,
        warn_limit=warn_threshold_seconds,
        decision_value=selector.read_lag,
    )

    check_name = 'lbexporter-db-deadlock'

    link = "grafana link: https://grafana.yandex-team.ru/d/A1JqLIQGk/accounter?viewPanel=610&var-env={env}".format(env=env)
    alarm_descr = "\n".join([
        "lbexporter haven't checked shard {{labels.shard}} for {{expression.read_lag}} seconds",
        "action: try to kill lbexporter instances to unblock",
        link
    ])
    ok_descr = "\n".join([
        "lbexporter is succesfully exporting shard {{labels.shard}}",
        link
    ])
    solomon_alert_ids = {
        "prod": "billing30-accounts-{}".format(check_name),
        "test": "billing30-accounts-test-{}".format(check_name),
    }
    expr = solomon_expression_custom(
        project_id=project_id,
        program_str=compose_program(selector, alert_block),
        annotations={
            "description": "{{{{#isAlarm}}}}{alarm_descr}{{{{/isAlarm}}}}\n".format(alarm_descr=alarm_descr) +
                           "{{{{#isWarn}}}}{alarm_descr}{{{{/isWarn}}}}\n".format(alarm_descr=alarm_descr) +
                           "{{{{#isOk}}}}{ok_descr}{{{{/isOk}}}}\n".format(ok_descr=ok_descr),
        },
        window_secs=5 * 60,
        group_by_labels=['shard'],
        juggler_children={"replace": True, "children": [
            juggler_sdk.Child(solomon_alert_ids[env], "all", "all", "MONITORING_MULTIALERT")]}
    )

    return solomon_check(
        check_name,
        expr,
        *args
    )
