# coding=utf-8
from collections import namedtuple

from paysys.sre.tools.monitorings.configs.billing30.base import annotations_present, annotations_timing
from paysys.sre.tools.monitorings.lib.checks.active.http import http, https
from paysys.sre.tools.monitorings.lib.checks.awacs import l7_monitoring
from paysys.sre.tools.monitorings.lib.checks.doc import doc_link, doc
from paysys.sre.tools.monitorings.lib.util.aggregators import logic_or
from paysys.sre.tools.monitorings.lib.util.helpers import flaps, merge, check, gen_children, gen_children_deploy, ttl
from paysys.sre.tools.monitorings.lib.util.solomon import (
    solomon_presence_expression_monitoring, solomon_timing_expression_monitoring,
    solomon_threshold_expression_monitoring, selectors_to_string, solomon_expression_custom
)

# ID проекта в соломоне.
# See: https://solomon.yandex-team.ru/?project=newbilling-tarification
SOLOMON_TARIFICATION_PROJECT_ID = "newbilling-tarification"

# Ссылка на текущего дежурного в доке Diod`a.
SWAT_DUTY = 'https://wiki.yandex-team.ru/balance/swat/#anchor-duty'

# ID проекта в соломоне, откуда можно взять метрики для YDB для Diod.
# See: https://solomon.yandex-team.ru/?project=kikimr
SOLOMON_KIKIMR_PROJECT_ID = 'kikimr'

Topic = namedtuple('Topic', field_names=['name', 'topic', 'consumer', 'warn', 'crit'])


def quorum_problem(name=None):
    """
    Использовать модуль агрегации timed_more_than_limit_is_problem
    See: https://docs.yandex-team.ru/juggler/aggregates/aggregators#timed_more_than_limit_is_problem
    """
    aggregator_kwargs = {
        'nodata_mode': 'force_warn',
        'limits': [
            {
                'day_start': 1, 'day_end': 7,
                'time_start': 0, 'time_end': 23,
                'warn': '33%', 'crit': '50%'
            }
        ],
    }

    if name:
        aggregator_kwargs['unreach_mode'] = 'skip'
        aggregator_kwargs['unreach_service'] = [{'check': ':unreachable-%s' % name}]

    return {
        'aggregator': 'timed_more_than_limit_is_problem',
        'aggregator_kwargs': aggregator_kwargs
    }


def get_http_checks(children):
    return [
        # mediator
        http(
            'mediator-http-check', 8080, ok_codes=[200],
            headers={"Host": "localhost"},
            warn='34%', crit='50%', no_unreach=True
        ),
        check(
            'mediator-http-check', gen_children_deploy(children['mediator'], 'mediator'),
            quorum_problem('mediator')
        ),

        # mediator-copier
        http(
            'mediator-copier-http-check', 8080, ok_codes=[200],
            headers={"Host": "localhost"},
            warn='34%', crit='50%', no_unreach=True
        ),
        check(
            'mediator-copier-http-check', gen_children_deploy(children['mediator-copier'], 'mediator-copier'),
            quorum_problem('mediator-copier')
        ),

        # processor
        https(
            'processor-https-check', 443, ok_codes=[200],
            headers={"Host": "processor.billing.yandex.net"},
            warn='34%', crit='50%', no_unreach=True
        ),
        check(
            'processor-https-check', gen_children_deploy(children['processor'], 'processor'),
            quorum_problem('processor')
        ),

        # diod
        http(
            name='diod-http-check',
            port=9000,
            ok_codes=[200],
            headers={'Host': 'localhost'},
            warn='34%',
            crit='50%',
            no_unreach=True
        ),
        check(
            'diod-http-check',
            gen_children_deploy(children['diod'], 'diod'),
            quorum_problem('diod')
        ),
    ]


def get_mediator_checks(env, topics):
    checks = [
        {
            "mediator-fatal-errors": {
                "solomon": solomon_presence_expression_monitoring(
                    project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                    annotations=annotations_present,
                    selectors={
                        "cluster": env,
                        "service": "mediator",
                        "resolution": "error:fatal",
                        "sensor": "messages:count",
                        "host": "cluster"
                    },
                    warn_threshold=5,
                    crit_threshold=10,
                    window_secs=5 * 60,
                    description="Fatal errors",
                    fn='sum',
                    total_fn='avg',
                ),
                'aggregator_kwargs': {'nodata_mode': 'force_warn'}
            },
            "mediator-goroutine-num-{env}".format(env=env): {
                "solomon": solomon_presence_expression_monitoring(
                    project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                    selectors={
                        "cluster": env,
                        "host!": "cluster",
                        "service": "mediator",
                        "sensor": "go.goroutine.num",
                    },
                    warn_threshold=500,
                    crit_threshold=1000,
                    window_secs=60,
                    fn="max",
                    total_fn="avg",
                    description="Goroutine count",
                ),
                'aggregator_kwargs': {'nodata_mode': 'force_warn'}
            },
            "mediator-fd-num-{env}".format(env=env): {
                "solomon": solomon_presence_expression_monitoring(
                    project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                    selectors={
                        "cluster": env,
                        "host!": "cluster",
                        "service": "mediator",
                        "sensor": "proc.fd",
                    },
                    warn_threshold=50,
                    crit_threshold=100,
                    window_secs=60,
                    fn="max",
                    total_fn="avg",
                    description="File descriptor count",
                ),
                'aggregator_kwargs': {'nodata_mode': 'force_warn'}
            }
        }
    ]

    for item in topics:
        checks.append({
            ("mediator-lag-%s" % item.name): {
                "solomon": solomon_presence_expression_monitoring(
                    project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                    annotations=annotations_present,
                    selectors={
                        "cluster": 'lbkx',
                        "project": "kikimr",
                        "Account": item.topic.strip('/').split('/')[0],
                        "sensor": "MessageLagByCommitted",
                        "service": "pqtabletAggregatedCounters",
                        "TopicPath": item.topic,
                        "ConsumerPath": item.consumer
                    },
                    warn_threshold=item.warn,
                    crit_threshold=item.crit,
                    window_secs=90,
                    description="Unread message count",
                    fn='max'
                ),
                'aggregator_kwargs': {'nodata_mode': 'force_warn'}
            }
        })

    return checks


def _processor_yt(env):
    result = {}

    for yt_cluster in ['arnold', 'hahn']:
        result["processor-yt-timing-{env}-{yt_cluster}".format(env=env, yt_cluster=yt_cluster)] = {
            "solomon": solomon_timing_expression_monitoring(
                project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                annotations=annotations_timing,
                selectors={
                    "cluster": env,
                    "host": "cluster",
                    "service": "processor",
                    "sensor": "readRows:duration",
                    "yt_cluster": yt_cluster
                },
                percentile=95,
                warn_timing_threshold=0.2,
                crit_timing_threshold=1,
                window_secs=60,
                fn="median",
                description="YT %s timings" % yt_cluster,
            ),
            'aggregator_kwargs': {'nodata_mode': 'force_warn'}
        }

        yt_selectors = {
            "project": "yt",
            "service": "accounts",
            "account": "balance",
            "medium": "ssd_blobs",
            "cluster": yt_cluster,
        }

        result["processor-yt-ssd-blobs-quota-usage-{yt_cluster}".format(yt_cluster=yt_cluster)] = {
            "solomon": solomon_threshold_expression_monitoring(
                project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                selectors=dict(
                    {"sensor": "disk_space_in_gb"},
                    **yt_selectors
                ),
                limit_selectors=dict(
                    {"sensor": "disk_space_limit_in_gb"},
                    **yt_selectors
                ),
                warn_percent_threshold=0.7,
                crit_percent_threshold=0.9,
                description="YT %s ssd_blobs quota usage" % yt_cluster,
            ),
            'aggregator_kwargs': {'nodata_mode': 'force_warn'}
        }

    return result


def _processor_environment(env):
    result = {}

    result["processor-goroutine-num-{env}".format(env=env)] = {
        "solomon": solomon_presence_expression_monitoring(
            project_id=SOLOMON_TARIFICATION_PROJECT_ID,
            selectors={
                "cluster": env,
                "host!": "cluster",
                "service": "processor",
                "sensor": "go.goroutine.num",
            },
            warn_threshold=3000,
            crit_threshold=5000,
            window_secs=60,
            fn="max",
            total_fn="avg",
            description="Goroutine count",
        ),
        'aggregator_kwargs': {'nodata_mode': 'force_warn'}
    }

    result["processor-fd-num-{env}".format(env=env)] = {
        "solomon": solomon_presence_expression_monitoring(
            project_id=SOLOMON_TARIFICATION_PROJECT_ID,
            selectors={
                "cluster": env,
                "host!": "cluster",
                "service": "processor",
                "sensor": "proc.fd",
            },
            warn_threshold=1000,
            crit_threshold=3000,
            window_secs=60,
            fn="max",
            total_fn="avg",
            description="File descriptor count",
        ),
        'aggregator_kwargs': {'nodata_mode': 'force_warn'}
    }

    return result


def _processor_calculator_timings(env):
    result = {}
    instances = {
        'taxi': ['cashless', 'fuel', 'payout', 'revenue', 'subvention']
    }

    for instance, endpoints in instances.items():
        for name in endpoints:
            selectors = {
                "cluster": env,
                "host": "cluster",
                "service": "processor",
                "name": name,
                "error": "-||url.timeout",
                "sensor": "client_calculator-%s.interaction_response_time" % instance,
            }

            result["proc-calc-timing-{}-{}-{}".format(env, instance, name)] = {
                "solomon": solomon_timing_expression_monitoring(
                    project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                    annotations=annotations_timing,
                    selectors=selectors,
                    percentile=90,
                    warn_timing_threshold=0.25,
                    crit_timing_threshold=0.5,
                    window_secs=60,
                    fn="median",
                    description="Taxi calculator {} timings",
                ),
                'aggregator_kwargs': {'nodata_mode': 'force_warn'}
            }

    return result


def get_processor_checks(env):
    return [
        _processor_environment(env),
        _processor_yt(env),
        _processor_calculator_timings(env)
    ]


def _diod_environment(env):
    return {
        'diod-goroutine-num-{env}'.format(env=env): {
            'solomon': solomon_presence_expression_monitoring(
                project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                selectors={
                    'cluster': 'diod-{env}'.format(env=env),
                    'host!': 'cluster',
                    'service': 'diod',
                    'sensor': 'go.goroutine.num'
                },
                warn_threshold=3000,
                crit_threshold=5000,
                window_secs=60,
                fn='max',
                total_fn='avg',
                description='Goroutine count',
            ),
            'aggregator_kwargs': {'nodata_mode': 'force_warn'}
        },

        'diod-fd-num{env}'.format(env=env): {
            'solomon': solomon_presence_expression_monitoring(
                project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                selectors={
                    'cluster': 'diod-{env}'.format(env=env),
                    'host!': 'cluster',
                    'service': 'diod',
                    'sensor': 'proc.fd'
                },
                warn_threshold=1000,
                crit_threshold=3000,
                window_secs=60,
                fn='max',
                total_fn='avg',
                description='File discriptor count',
            ),
            'aggregator_kwargs': {'nodata_mode': 'force_warn'}
        }
    }


def _ydb_enviroment(env, ydb_cluster, ydb_database):
    elapsed_cpu_selectors = {
        'cluster': ydb_cluster,
        'host': 'cluster',
        'service': 'utils',
        'database': ydb_database,
        'sensor': 'ElapsedMicrosec',
        'slot': 'static',
        'execpool': 'User',
        'project': SOLOMON_KIKIMR_PROJECT_ID
    }

    total_cpu_selectors = {
        'cluster': ydb_cluster,
        'host': 'cluster',
        'service': 'utils',
        'database': ydb_database,
        'sensor': 'ParkedMicrosec',
        'slot': 'static',
        'execpool': 'User',
        'project': SOLOMON_KIKIMR_PROJECT_ID
    }

    limit_storage_selectors = {
        'cluster': ydb_cluster,
        'host': 'cluster',
        'service': 'ydb',
        'database': ydb_database,
        'name': 'resources.storage.limit_bytes',
        'project': SOLOMON_KIKIMR_PROJECT_ID,
    }

    used_storage_selectors = {
        'cluster': ydb_cluster,
        'host': 'cluster',
        'service': 'ydb',
        'database': ydb_database,
        'name': 'resources.storage.used_bytes',
        'project': SOLOMON_KIKIMR_PROJECT_ID,
    }

    return {
        # Метрика времени обработки транзакций
        # See: https://ydb.yandex-team.ru/docs/troubleshooting/monitoring#transactions
        'diod-ydb-transactions-total-duration': {
            'solomon': solomon_timing_expression_monitoring(
                project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                selectors={
                    'project': SOLOMON_KIKIMR_PROJECT_ID,
                    'cluster': ydb_cluster,
                    'host': 'cluster',
                    'service': 'kqp',
                    'database': ydb_database,
                    'sensor': 'Transactions/Pure/TotalDurationMs'
                },
                percentile=95,
                warn_timing_threshold=1.2,
                crit_timing_threshold=3,
                window_secs=60,
                fn='median',
                description='YDB transaction execution duration'
            ),
            'aggregator_kwargs': {},
        },
        # Метрика использованного CPU
        'diod-ydb-used-cpu': {
            'solomon': solomon_expression_custom(
                project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                program_str='''
                let elapsed = {elapsed_selectors};

                let total = elapsed + {total_selectors};

                let utilization = last(elapsed/total);

                alarm_if(utilization >= {crit_percent_threshold});
                warn_if(utilization >= {warn_percent_threshold});
                '''.format(
                    elapsed_selectors=selectors_to_string(elapsed_cpu_selectors),
                    total_selectors=selectors_to_string(total_cpu_selectors),
                    crit_percent_threshold=0.9,
                    warn_percent_threshold=0.7
                )
            ),
            'aggregator_kwargs': {}
        },

        # Метрика использованной памяти
        'diod-ydb-used-memory': {
            'solomon': solomon_expression_custom(
                project_id=SOLOMON_TARIFICATION_PROJECT_ID,
                program_str='''
                let total = {limit_storage_selectors};

                let used = {used_storage_selectors};

                let utilization = last(used / total);

                alarm_if(utilization >= {crit_percent_threshold});
                warn_if(utilization >= {warn_percent_threshold});
                '''.format(
                    limit_storage_selectors=selectors_to_string(limit_storage_selectors),
                    used_storage_selectors=selectors_to_string(used_storage_selectors),
                    crit_percent_threshold=0.9,
                    warn_percent_threshold=0.7,
                )
            ),
            'aggregator_kwargs': {}
        }
    }


def get_diod_checks(env, ydb):
    return (
        [_diod_environment(env)]
        + [_ydb_enviroment(env, item['cluster'], item['database']) for item in ydb]
    )


def _dt_check(env, name, yt_cluster, resource_id):
    return check(
        'dt-row-lag-{env}-{name}-{yt_cluster}'.format(env=env, name=name, yt_cluster=yt_cluster),
        gen_children(resource_id, 'dta_04_row_max_lag_constant', group_type='HOST'),
        {'aggregator_kwargs': {'nodata_mode': 'force_warn'}}
    )


def get_dt_checks(env):
    return [
        _dt_check(env, name, yt_cluster, resource_id) for resource_id, name, yt_cluster in [
            ('dttaav8ihqvsocccr87k', 'product', 'arnold'),
            ('dttnh8tv7jg7fs86bipq', 'product', 'hahn'),
            ('dtt202rg4a2eeappvg13', 'iso-currency-rate', 'arnold'),
            ('dtt77rsh8541rgkspusn', 'iso-currency-rate', 'hahn'),
            ('dttckq0l2n2cnfemiged', 'personal-account', 'arnold'),
            ('dtthcbf4pg9p4sqtbpbi', 'personal-account', 'hahn'),
            ('dtt942gt0s70c6ktvedc', 'contract', 'arnold'),
            ('dttvipga57dnfa39ib3d', 'contract', 'hahn'),
        ]
    ]


def get_checks(children, balancers, env, topics, ydb):
    args = []

    for name, child in children.items():
        args.append(
            check(
                # check name
                'unreachable-%s' % name,
                # check endpoint
                gen_children_deploy(child, 'unreachable'), quorum_problem(),
                # type of the active check
                {"active": "icmpping"},
                # Документация
                doc_link("common"),
                doc(SWAT_DUTY, 'Кого нужно пингануть'),
            )
        )
        args.append(check('unispace-%s' % name, gen_children_deploy(child, 'unispace'), quorum_problem(name)))

        # Do not add pushclient_check for diod, processor and faas children.
        if name == 'calculator-taxi':
            args.append(
                check(
                    'pushclient_check-%s' % name, gen_children_deploy(child, 'pushclient_check'),
                    ttl(900, 300), flaps(180, 900), quorum_problem(name)
                )
            )

    args += get_http_checks(children)
    args += get_mediator_checks(env, topics)
    args += get_processor_checks(env)
    args += get_diod_checks(env, ydb)
    args += get_dt_checks(env)

    args += [
        l7_monitoring(balancers),
        check(
            'certs',
            gen_children_deploy([
                children['processor'],
                children['diod']
            ], 'certs'),
            logic_or,
            {'aggregator_kwargs': {'nodata_mode': 'force_warn'}}
        ),
    ]

    return merge(*args)
