# coding=utf-8
import paysys.sre.tools.monitorings.configs.billing30.base.accounts as accounts
from paysys.sre.tools.monitorings.configs.billing30.base import (annotations_absent, annotations_percent,
                                                                 annotations_present,
                                                                 annotations_timing)
from paysys.sre.tools.monitorings.lib.checks.active.http import https
from paysys.sre.tools.monitorings.lib.checks.nirvana import Checks as NirvanaChecks
from paysys.sre.tools.monitorings.lib.checks.reactor import Checks as ReactorChecks
from paysys.sre.tools.monitorings.lib.notifications import Notifications
from paysys.sre.tools.monitorings.lib.util.aggregators import timed_more_than_limit_is_problem
from paysys.sre.tools.monitorings.lib.util.helpers import (
    empty_children, merge, flaps, check, gen_children_deploy, solomon_check
)
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_timing_expression_monitoring, \
    solomon_absence_expression_monitoring, \
    solomon_presence_expression_monitoring, \
    solomon_expression_custom

import itertools

notifications_telegram = (
    Notifications()
        .set_telegram(group=["hot_billing_accounter_notifications"], delay=30)
        .telegram
)

iron_woman_telegram_notifications = (
    Notifications()
        .set_telegram(group=["hot_billing_accounter_notifications"], delay=30)
        .set_iron_woman(logins=['@svc_billing-core:duty-none', 'kositsyn-pa', 'lightrevan'], delay=1200)
)

notifications_telegram_and_call = (
    iron_woman_telegram_notifications.iron_woman_and_telegram
)
notifications_telegram_and_call_daytime = (
    iron_woman_telegram_notifications.iron_woman_daytime_and_telegram
)


defaults = {'namespace': 'newbilling-accounter'}

host = "billing30.accounts"
project_id = "newbilling-accounter"

children = empty_children
children_api = ['billing-accounts@stage=billing-accounts-prod-stage;deploy_unit=api']
children_tasks = ['billing-accounts@stage=billing-accounts-prod-stage;deploy_unit=tasks']
children_lbexporter = ['billing-accounts@stage=billing-accounts-prod-stage;deploy_unit=lbexporter']
children_sequencer = ['billing-accounts@stage=billing-accounts-prod-stage;deploy_unit=sequencer']

l7_balancers = [
    {
        "namespace": "accounts.billing.yandex.net",
        "datacenters": ["man", "sas", "vla"],
        "host": "accounts.billing.yandex.net",
        "http_ports": [80],
        "https_ports": [443],
        "services": [
            "rtc_balancer_accounts_billing_yandex_net_man",
            "rtc_balancer_accounts_billing_yandex_net_sas",
            "rtc_balancer_accounts_billing_yandex_net_vla",
        ],
        "checks": {
            "cpu_usage": {"warn": 60, "crit": 80},
            "cpu_wait": {"warn": 0.5, "crit": 1},
            "codes_5xx": {"warn": 20, "crit": 100, "flaps": {"stable": 300, "critical": 600}},
            "attempts_backend_errors": {"warn": 20, "crit": 100, "flaps": {"stable": 300, "critical": 600}},
        },
    },
]


def checks():
    base_checks = accounts.get_checks(
        children_api,
        children_tasks,
        children_sequencer,
        children_lbexporter,
        l7_balancers,
    )
    base_checks = merge(
        base_checks,
        https(
            'accounts-https-check', 443, ok_codes=[200], crit=0,
            headers={"Host": "accounts.billing.yandex.net"}
        ),
        check('accounts-https-check', gen_children_deploy(children_api, 'api')),
    )
    reactor_project_id = 11220499
    reactor = ReactorChecks("newbillingaccountupdater", reactor_project_id, project_id, notifications_telegram_and_call_daytime)
    nirvana = NirvanaChecks("billing30_hot_accounts", project_id, notifications_telegram_and_call_daytime)

    solomon_checks = merge(
        _jobs_timings(),
        _lbexporter_timings(),
        _sequencer_timings(),
        _api_2xx(),
        _lbexporter_success(),
        _sequencer_success(),
        _jobs_success(),
        _jobs_fails(),
        _sequencer_fails(),
        _lbexporter_fails(),
        accounts.lbexporter_db_deadlock(project_id, 'prod', 600, 3600, notifications_telegram_and_call_daytime),
        reactor.reaction_bundle(),
        reactor.reaction_instances_created_velocity(),
        reactor.running_reaction_instances_number(),
        reactor.reaction_instances_total(),
        nirvana.bundle(),
        *itertools.chain(
            [_api_5xx_percent(path) for path in (
                '/v1/accounts/balance', '/v1/batch/read', '/v1/batch/write',
                '/v1/locks/init', '/v1/locks/ping', '/v1/locks/state', '/v1/locks/remove',
            )],
            [_api_4xx_percent(path) for path in (
                '/v1/accounts/balance', '/v1/batch/read', '/v1/batch/write',
                '/v1/locks/init', '/v1/locks/ping', '/v1/locks/state', '/v1/locks/remove',
            )],
            [_api_timings(path) for path in (
                '/v1/accounts/balance', '/v1/locks/init', '/v1/locks/ping', '/v1/locks/state',
            )]
        )
    )

    for key, val in solomon_checks.items():
        if any(key.startswith(prefix) for prefix in ["api-5xx-percent-prod", "api-4xx-percent-prod", "api-timing-prod"]):
            val["aggregator_kwargs"] = merge(val.get("aggregator_kwargs", {}), {"nodata_mode": "force_ok"})
            continue
        val["aggregator_kwargs"] = merge(val.get("aggregator_kwargs", {}), {"nodata_mode": "force_crit"})

    return merge(base_checks, solomon_checks)


def _lbexporter_fails():
    name = "lbexporter-fail-prod"
    selectors = {
        "cluster": "prod",
        "service": "lbexporter",
        "component": "handleExportQueue",
        "sensor": "write:fail",
    }
    expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_present,
        selectors=selectors,
        description="Failed writes lbexporter",
    )
    return {
        name: merge(
            {
                "solomon": expr,
            },
            flaps(600, 1800),  # 10min and at least 1 fail every 2 minutes is actually bad.
            notifications_telegram_and_call_daytime
        )
    }


def _sequencer_fails():
    name = "sequencer-fail-prod"
    selectors = {
        "cluster": "prod",
        "host": "cluster",
        "service": "sequencer",
        "sensor": "process:fail",
    }
    expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_present,
        selectors=selectors,
        description="Failed attemps to set seq_id",
    )
    return {
        name: merge(
            {
                "solomon": expr,
            },
            flaps(600, 1800),  # 10min and at least 1 update fails every 2 minutes is actually bad.
            notifications_telegram_and_call
        )
    }


def _jobs_fails():
    expr = solomon_expression_custom(
        program_str=(
            'let data = group_lines("sum", {cluster="prod", host="cluster", sensor="jobs:fail", service="tasks"});\n'
            'let total = sum(data);\n'
            'warn_if(total > 0);\n'),
        project_id=project_id,
        annotations=annotations_present,
        description="Failed jobs",
        window_secs=15 * 60,
    )
    return {
        "jobs-fail-prod": merge(
            {
                "solomon": expr,
            },
            flaps(1800, 3600),  # only 3x in a row (delta 10min) triggers alert.
            notifications_telegram
        )
    }


def _jobs_success():
    # Мы ищем только обновления конкретно шардов
    selectors = {
        "cluster": "prod",
        "service": "tasks",
        "sensor": "jobs:success",
        "job": "UpdateRollupShard",
        "host": "cluster",
    }
    expr = solomon_absence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_absent,
        selectors=selectors,
        warn_threshold=0.15,
        crit_threshold=0.05,
        window_secs=60 * 30,
        description="Successful jobs",
    )
    return {
        "jobs-success-prod": merge(
            {
                "solomon": expr,
            },
            notifications_telegram_and_call,
        )
    }


def _sequencer_success():
    selectors = {
        "cluster": "prod",
        "service": "sequencer",
        "sensor": "process:success",
    }
    expr = solomon_absence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_absent,
        selectors=selectors,
        warn_threshold=0.2,
        crit_threshold=0.1,
        description="Successful seq_id updates",
    )
    return solomon_check("sequencer-process-success-prod", expr)


def _lbexporter_success():
    selectors = {
        "cluster": "prod",
        "service": "lbexporter",
        "sensor": "write:success",
    }
    expr = solomon_absence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_absent,
        selectors=selectors,
        warn_threshold=0.2,
        crit_threshold=0.1,
        window_secs=60 * 5,
        description="Successful updates lbexporter",
    )
    return {
        "lbexporter-write-success-prod": merge(
            {
                "solomon": expr,
            },
            timed_more_than_limit_is_problem(limits=[
                # time_end: 8 значит до 8:59. Добро пожаловать в juggler
                {"time_start": 10, "time_end": 8, "day_start": 1, "day_end": 7, "warn": 0, "crit": 0},
                # с 9 до 10 у такси технологическое окно когда они не шлют событий. Поставил большой crit для игнора
                # time_end: 9 значит до 9:59. Добро пожаловать в juggler
                {"time_start": 9, "time_end": 9, "day_start": 1, "day_end": 7, "warn": 0, "crit": 5},
            ])
        )
    }


def _api_2xx():
    selectors = {
        "cluster": "prod",
        "service": "api",
        "host": "cluster",
        "status": "2*",
        "sensor": "requests",
    }
    expr = solomon_absence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_absent,
        selectors=selectors,
        warn_threshold=20,
        crit_threshold=10,
        description="Successful API 2xx",
    )
    return {
        "api-2xx-prod": merge(
            {
                "solomon": expr,
            },
            notifications_telegram_and_call
        )
    }


def _lbexporter_timings():
    selectors = {
        "cluster": "prod",
        "host": "cluster",
        "service": "lbexporter",
        "sensor": "duration",
    }
    expr = solomon_timing_expression_monitoring(
        project_id=project_id,
        annotations=annotations_timing,
        selectors=selectors,
        percentile=95,
        warn_timing_threshold=60,
        crit_timing_threshold=100,
        description="Lbexporter timings",
    )
    return solomon_check("lbexporter-timing-prod", expr)


def _sequencer_timings():
    selectors = {
        "cluster": "prod",
        "host": "cluster",
        "service": "sequencer",
        "sensor": "duration",
    }
    expr = solomon_timing_expression_monitoring(
        project_id=project_id,
        annotations=annotations_timing,
        selectors=selectors,
        percentile=95,
        warn_timing_threshold=1,
        crit_timing_threshold=2,
        description="Sequencer timings",
    )
    return solomon_check("sequencer-timing-prod", expr)


def _jobs_timings():
    selectors = {
        "cluster": "prod",
        "service": "tasks",
        "job": "UpdateRollupShard",
        "sensor": "duration",
        "host": "cluster",
    }
    expr = solomon_timing_expression_monitoring(
        project_id=project_id,
        annotations=annotations_timing,
        selectors=selectors,
        percentile=95,
        warn_timing_threshold=25,
        crit_timing_threshold=40,
        description="Jobs timings",
        window_secs=60 * 30,
    )
    return solomon_check("jobs-timing-prod", expr)


def _api_timings(path):
    selectors = {
        "cluster": "prod",
        "service": "api",
        "host": "cluster",
        "path": path,
        "sensor": "duration",
    }
    expr = solomon_timing_expression_monitoring(
        project_id=project_id,
        annotations=annotations_timing,
        selectors=selectors,
        percentile=95,
        warn_timing_threshold=0.5,
        crit_timing_threshold=2,
        description="API timings"
    )
    return solomon_check("api-timing-prod-{}".format(_normalize_path(path)), expr)


def _api_5xx_percent(path):
    expr = solomon_expression_custom(
        program_str=(
            'let used = series_sum({{status="5*", service="api", cluster="prod", host="cluster", path="{}", sensor="requests"}});\n'.format(path) +
            'let limit = series_sum({{status="*", service="api", cluster="prod", host="cluster", path="{}", sensor="requests"}});\n'.format(path) +
            'let percent = avg(used / limit);\n' +
            'let summary_limit = avg(limit);\n' +
            'alarm_if((percent >= 0.05) && (summary_limit > 5));\n' +
            'warn_if((percent >= 0.005) && (summary_limit > 5));\n'
        ),
        project_id=project_id,
        annotations=annotations_percent,
        description="API 5xx percent rate",
    )
    return {
        "api-5xx-percent-prod-{}".format(_normalize_path(path)): merge(
            {
                "solomon": expr,
            },
            flaps(180, 600),
            notifications_telegram_and_call
        )
    }


def _api_4xx_percent(path):
    expr = solomon_expression_custom(
        program_str=(
            'let used = series_sum({{status="4*", service="api", cluster="prod", host="cluster", path="{}", sensor="requests"}});\n'.format(path) +
            'let limit = series_sum({{status="*", service="api", cluster="prod", host="cluster", path="{}", sensor="requests"}});\n'.format(path) +
            'let percent = avg(used / limit);\n' +
            'let summary_limit = avg(limit);\n' +
            'alarm_if((percent >= 0.8) && (summary_limit > 10));\n' +
            'warn_if((percent >= 0.1) && (summary_limit > 10));\n'
        ),
        project_id=project_id,
        annotations=annotations_percent,
        description="API 4xx percent rate",
    )
    return {
        "api-4xx-percent-prod-{}".format(_normalize_path(path)): merge(
            {
                "solomon": expr,
            },
            flaps(180, 600),
            notifications_telegram_and_call
        )
    }


def _get_selector_for_status(path, status):
    return {
        "cluster": "prod",
        "host": "cluster",
        "service": "api",
        "sensor": "requests",
        "path": path,
        "status": status
    }


def _normalize_path(path):
    return path[1:].replace("/", ".")
