# coding=utf-8

import itertools

import paysys.sre.tools.monitorings.configs.billing30.base.payout as payout
from paysys.sre.tools.monitorings.lib.checks.active.http import https
from paysys.sre.tools.monitorings.lib.util.solomon import (solomon_threshold_expression_monitoring,
                                                           solomon_timing_expression_monitoring,
                                                           solomon_presence_expression_monitoring,
                                                           solomon_absence_expression_monitoring)
from paysys.sre.tools.monitorings.lib.util.helpers import empty_children, merge, solomon_check, check, gen_children, \
    gen_children_deploy
from paysys.sre.tools.monitorings.lib.notifications import Notifications, \
    FIN_TOOLS_ESCALATION_USERS, FIN_TOOLS_TELEGRAM_GROUP
from paysys.sre.tools.monitorings.lib.checks.postgres import postgres

notifications = Notifications()
notifications._telegram_default_options["status"] = ["CRIT", "WARN", "OK"]
notifications.set_iron_woman(logins=FIN_TOOLS_ESCALATION_USERS, delay=1200) \
    .set_telegram(group=FIN_TOOLS_TELEGRAM_GROUP)

defaults = {'namespace': 'billing30.payout.stable'}

host = "billing30.payout"
project_id = "newbilling-payouts"

annotations_timing = {
    "description": "{{#isAlarm}}Timings {{expression.percentile}} are above critical threshold!{{/isAlarm}}\n"
                   "{{#isWarn}}Timings {{expression.percentile}} are above warn threshold{{/isWarn}}"}
annotations_present = {
    "description": "{{#isAlarm}}Total value: {{expression.total}} is above critical threshold!{{/isAlarm}}\n"
                   "{{#isWarn}}Total value: {{expression.total}} is above warn threshold{{/isWarn}}"}
annotations_percent = {
    "description": "{{#isAlarm}}Percent {{expression.percent}} above critical threshold!{{/isAlarm}}\n"
                   "{{#isWarn}}Percent {{expression.percent}} above warn threshold{{/isWarn}}"}
annotations_absent = {
    "description": "{{#isAlarm}}Total value: {{expression.total}} is below critical threshold!{{/isAlarm}}\n"
                   "{{#isWarn}}Total value: {{expression.total}} is below warn threshold{{/isWarn}}"}

children = empty_children
children_api = ['billing-payout@stage=billing-payout-prod-stage;deploy_unit=api-gate']
children_tasks = ['billing-payout@stage=billing-payout-prod-stage;deploy_unit=tasks']
children_gates = ['billing-payout@stage=billing-payout-prod-stage;deploy_unit=gates']

l7_balancers = [
    {
        "namespace": "payout.billing.yandex.net",
        "datacenters": ["man", "sas", "vla"],
        "host": "payout.billing.yandex.net",
        "http_ports": [80],
        "https_ports": [443],
        "services": [
            "rtc_balancer_payout_billing_yandex_net_man",
            "rtc_balancer_payout_billing_yandex_net_sas",
            "rtc_balancer_payout_billing_yandex_net_vla",
        ],
        "checks": {
            "cpu_usage": {"warn": 60, "crit": 80},
            "cpu_wait": {"warn": 0.5, "crit": 1},
            "codes_5xx": {"warn": 0.5, "crit": 1},
            "attempts_backend_errors": {"warn": 1, "crit": 2},
        },
    },
]


def checks():
    base = payout.get_checks(
        children_api,
        children_tasks,
        children_gates,
        l7_balancers,
    )

    payouts = _get_payouts()
    requests = _get_requests()
    cpf = _get_cpf()
    dt_export = _get_dt_exporter()
    errors = merge(
        _get_error_messages_count(),
        _get_oebs_errors_count(),
    )
    solomons = merge(
        payouts,
        requests,
        cpf,
        dt_export,
        errors,
        _api_timings("POST", 0.5, 2),
        _api_timings("GET", 2, 4),
        *itertools.chain(
            [_api_5xx_percent(path, "POST") for path in (
                '/api/v1/check', '/api/v1/payout', '/api/v1/payout-by-client',
            )],
            [_api_5xx_percent(path, "GET") for path in (
                '/api/v1/payout',
            )],
            [_api_4xx_percent(path, "POST") for path in (
                '/api/v1/check', '/api/v1/payout', '/api/v1/payout-by-client',
            )],
            [_api_4xx_percent(path, "GET") for path in (
                '/api/v1/payout',
            )],
        )
    )
    for key, val in solomons.items():
        if key.startswith("5xx-percent-prod") or key.startswith("4xx-percent-prod"):
            val["aggregator_kwargs"] = merge(val.get("aggregator_kwargs", {}), {"nodata_mode": "force_ok"})
        else:
            val["aggregator_kwargs"] = merge(val.get("aggregator_kwargs", {}), {"nodata_mode": "force_warn"})

    return merge(
        base,
        solomons,
        postgres("mdbrg678bi14u1uf4n1e", "payoutdb",
                 conn_limit=20,  # подключений может быть много (default=10)
                 transaction_time_limit=3000,  # ms, могут быть длинные транзакции (default=1k)
                 query_time_limit=40,  # запросы могут быть длинными (default=20)
                 ),
        _disaster(),
        _check_tls_certificate()
    )


def _disaster():
    result = check("billing30_payout_disaster",
                   gen_children("billing30.payout", [
                       "5xx-percent-prod-api.v1.check-POST",
                       "payout.billing.yandex.net_codes_5xx_vla",
                       "payout.billing.yandex.net_codes_5xx_sas",
                   ], group_type="HOST"))
    result['billing30_payout_disaster']['tags'] = [
        'warden_alert_create_spi',
        'is_disaster',
        'disaster',
        'marty',
        'createspi',
    ]
    result['billing30_payout_disaster']['meta'] = {
        'urls': [
            {
                'title': 'Описание алерта',
                'url': 'https://wiki.yandex-team.ru/balance/monitorings/disaster',
                'type': 'wiki'
            },
        ],
    }
    result['billing30_payout_disaster']['aggregator'] = 'logic_and'
    return result


def _check_tls_certificate():
    return merge(
        https(
            'payout-https-check', 443, ok_codes=[200], crit=0,
            headers={"Host": "payout.billing.yandex.net"}
        ),
        check(
            'payout-https-check',
            gen_children_deploy(children_api, 'api')
        )
    )


def _get_payouts():
    """
        Формирует все необходимые проверки связанные с БЛ для выплат.
        Сейчас только лаг репликации для прода
        :return:
        """
    return merge(
        _get_last_failed("payouts", "rejected"),
        _get_retardation("payouts", "new", 60, 2 * 60),  # 1h, 2h
        _get_retardation("payouts", "pending", 2 * 60, 4 * 60),  # 2h, 4h
        _get_more_new("payouts", 150 * 1000, 300 * 1000),
        _get_last_approve_required("payouts"),
        _get_partitions_count("payouts")
    )


def _get_requests():
    """
        Формирует все необходимые проверки связанные с БЛ для заявок.
        :return:
        """
    return merge(
        _get_last_failed("requests", "error"),
        _get_retardation("requests", "new", 60, 2 * 60),  # 1h, 2h
        _get_more_new("requests", 50 * 1000, 100 * 1000),
    )


def _get_cpf():
    """
        Формирует все необходимые проверки, связанные с CPF
    """
    return merge(
        _get_partitions_count("cpf")
    )


def _get_dt_exporter():
    """
    Формирует все необходимые проверки связанные с Data Transfer.
    Сейчас только лаг репликации для прода
    :return:
    """
    return merge(
        _get_dt_exporter_cluster("hahn", "dttto2d7l4lhokqqohec"),
        _get_dt_exporter_cluster("arnold", "dtthjalmiloeqjgclgh3"),
    )


def _get_dt_exporter_cluster(cluster, resource_id):
    """
    Формирует проверку на репликацию Data transfer из pg в YT
    :param cluster: название кластера, для названия
    :param resource_id: id трансфера для мониторинга
    :return: словарь с проверкой с ключами для соломона
    """
    export_cluster_prod_lag = check("export-{cluster}-prod-lag".format(cluster=cluster),
                                    gen_children([resource_id], ["dta_04_row_max_lag_constant", ],
                                                 group_type="HOST"))
    export_checks = merge(
        export_cluster_prod_lag,
    )
    for key, val in export_checks.items():
        val["aggregator_kwargs"] = merge(val.get("aggregator_kwargs", {}), {"nodata_mode": "force_crit"})

    return export_checks


def _api_timings(method, warn, crit):
    selectors = {
        "cluster": "prod",
        "service": "api",
        "host": "cluster",
        "sensor": "duration",
        "status": "_ALL_",
        "method": method
    }
    expr = solomon_timing_expression_monitoring(
        project_id=project_id,
        annotations=annotations_timing,
        selectors=selectors,
        percentile=95,
        warn_timing_threshold=warn,
        crit_timing_threshold=crit,
        description="API timings"
    )
    return solomon_check("api-{method}-timing-prod".format(method=method), expr)


def _get_retardation(service, status, warn, crit):
    selectors = {
        "cluster": "prod",
        "service": service,
        "sensor": "retardation",
        "status": status
    }
    expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_present,
        selectors=selectors,
        warn_threshold=warn,
        crit_threshold=crit,
        description="{service} retardation".format(service=service),
        fn="max",
        window_secs=5 * 60,  # 5 min
    )
    return solomon_check("{service}-prod-retardation-{status}".format(service=service, status=status), expr)


def _get_last_failed(service, failed):
    selectors = {
        "cluster": "prod",
        "service": service,
        "sensor": "last",
        "status": failed
    }
    expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_present,
        selectors=selectors,
        warn_threshold=0,
        crit_threshold=0,
        description="Failed {service} last 5 min".format(service=service),
        fn="max",
        window_secs=60,
    )
    return solomon_check("{service}-prod-last-failed".format(service=service), expr)


def _get_last_approve_required(service):
    selectors = {
        "cluster": "prod",
        "service": service,
        "sensor": "last",
        "approve_req": "true",
    }
    expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_present,
        selectors=selectors,
        warn_threshold=0,
        crit_threshold=0,
        description="Approve required payouts found in {service} last 5 min".format(service=service),
        fn="sum",
        window_secs=60,
    )
    return solomon_check("{service}-prod-last-approve_req".format(service=service), expr)


def _get_more_new(service, warn, crit):
    selectors = {
        "cluster": "prod",
        "service": service,
        "sensor": "total",
        "status": "new"
    }
    expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_present,
        selectors=selectors,
        warn_threshold=warn,
        crit_threshold=crit,
        description="A lot of new {service} (hour avg)".format(service=service),
        fn="avg",
        window_secs=30 * 60,  # 30m
    )
    return solomon_check("{service}-prod-more-new".format(service=service), expr)


def _get_selector_for_status(path, status, method):
    return {
        "cluster": "prod",
        "host": "cluster",
        "service": "api",
        "sensor": "count",
        "status": status,
        "path": path,
        "method": method
    }


def _normalize_path(path):
    return path[1:].replace("/", ".")


def _api_5xx_percent(path, method):
    expr = solomon_threshold_expression_monitoring(
        project_id=project_id,
        warn_percent_threshold=0.005,
        crit_percent_threshold=0.05,
        selectors=_get_selector_for_status(path, "5*", method),
        limit_selectors=_get_selector_for_status(path, "*", method),
        description="API 5xx percent rate",
        annotations=annotations_percent,
        fn="series_sum",
    )
    return solomon_check("5xx-percent-prod-{}-{}".format(_normalize_path(path), method), expr)


def _api_4xx_percent(path, method):
    expr = solomon_threshold_expression_monitoring(
        project_id=project_id,
        warn_percent_threshold=0.1,
        crit_percent_threshold=0.3,
        selectors=_get_selector_for_status(path, "4*", method),
        limit_selectors=_get_selector_for_status(path, "*", method),
        description="API 4xx percent rate",
        annotations=annotations_percent,
        fn="series_sum",
    )
    return solomon_check("4xx-percent-prod-{}-{}".format(_normalize_path(path), method), expr)


def _get_error_messages_count():
    """
        Формирует проверку на кол-во сообщений в топике errors по всем namespace
    """
    selectors = {
        "project": "kikimr", "cluster": "lbkx", "service": "pqproxy_writeSession",
        "Account": "billing-payout", "OriginDC": "cluster", "host": "cluster",
        "sensor": "MessagesWrittenOriginal",
        "TopicPath": "billing-payout/prod/notices/errors-*"
    }
    msgs_count_expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        selectors=selectors,
        warn_threshold=0,
        crit_threshold=0,
        annotations=annotations_present,
        description="Error messages count for all namespaces"
    )
    return solomon_check("error-topic-count-all-namespaces", msgs_count_expr)


def _get_partitions_count(service):
    """
        Формирует проверку на кол-во непрерывно созданных партиций переданной таблицы в будущее
    """
    selectors = {
        "cluster": "prod",
        "service": service,
        "sensor": "partition_count",
        "host": "cluster"
    }
    expr = solomon_absence_expression_monitoring(
        project_id=project_id,
        annotations=annotations_absent,
        selectors=selectors,
        warn_threshold=10,
        crit_threshold=4,
        description="{service} partition count".format(service=service),
        window_secs=3 * 60,
    )
    return solomon_check("{service}-prod-partition-count".format(service=service), expr)


def _get_oebs_errors_count():
    """
         Формирует проверку на количество сообщений в топике oebs-errors
    """
    selectors = {
        "project": "kikimr", "cluster": "lbkx", "service": "pqproxy_writeSession",
        "Account": "billing-payout", "OriginDC": "cluster", "host": "cluster",
        "sensor": "MessagesWrittenOriginal",
        "TopicPath": "billing-payout/prod/oebs-errors"
    }
    msgs_count_expr = solomon_presence_expression_monitoring(
        project_id=project_id,
        selectors=selectors,
        warn_threshold=0,
        crit_threshold=0,
        annotations=annotations_present,
        description="OEBS errors count",
        window_secs=15 * 60,
    )
    return solomon_check("oebs-errors-topic-count", msgs_count_expr)
