
import itertools
import re

from paysys.sre.tools.monitorings.lib.checks.graphite import (
    graphite_check,
    summarize,
    movingMin,
)
from paysys.sre.tools.monitorings.lib.util.helpers import (
    merge,
    ttl,
)


host = "paysys-dashboard.balance"

telegram = "Balance_Monitorings"
children = []

defaults = merge(
    ttl(620, 60),
    {'namespace': 'psadmin.stable.dashboards'},
)

window_re = re.compile(r"(\d+)(\w+)$")


def _summarize_check(name, metric, window, crit, aggr='min', less=False):
    window_duration, window_unit = window_re.match(window).groups()

    return graphite_check(
        name,
        summarize(metric, window, aggr),
        crit=crit,
        window='{0}{1}'.format(-2 * int(window_duration), window_unit),
        less=less,
    )


def response_timings():
    prefix = "summarize(keepLastValue(averageSeries(one_min.greedaggrprod_yandex_ru.simplebrother_balance_bo." \
             "BO.CALL_STAT_2_PERC80.greed[1-9][a-z].{0}), 2), '30min', true)"
    create_transfer_multiple = prefix.format('Balance.CreateTransferMultiple')
    ctl_issue_invoice = prefix.format('ctl_issue_invoice')
    get_payment_preview = prefix.format('get_payment_preview')

    return [
        graphite_check('create_transfer_multiple', create_transfer_multiple, crit=100),
        graphite_check('ctl_issue_invoice', ctl_issue_invoice, crit=100),
        graphite_check('get_payment_preview', get_payment_preview, crit=100)
    ]


def response_timing_by_dc():
    dcs = ['h', 'f', 'v', 'e']
    result = []
    for dc in dcs:
        prefix = 'aliasByNode(keepLastValue(one_min.greedaggrprod_yandex_ru.' \
                 'simplebrother_balance_bo.BO.CALL_STAT_2_PERC80.greed[1-9]{1}.Balance.{0}, 1), 5)'
        create_invoice = prefix.format('CreateInvoice', dc)
        get_request = prefix.format('GetRequestChoices', dc)
        create_request = prefix.format('CreateRequest', dc)
        find_client = prefix.format('FindClient', dc)

        avg = "groupByNode(aliasSub(groupByNode(group({0}, {1}, {2}, {3}), 0, 'avg'), '[0-9]+', '.'), 1, 'avg')".format(
            create_invoice,
            create_request,
            get_request,
            find_client
        )
        result.append(
            graphite_check('response_time_'+dc, movingMin(avg, '10min'), crit=2.5)
        )
    return result


def invoices_count():
    res = []

    prefix = "sumSeries(keepLastValue(one_min.greedaggrprod_yandex_ru.simplebrother_balance_bo.BO.{0}.{1}, 20))"
    service2params = {
        'total': ('*', '30min'),
        'direct': ('PPC', '30min'),
        'market': ('market', '2h'),
    }

    for alias, (key, window) in service2params.items():
        res.append(
            _summarize_check(
                'invoices_count_{0}'.format(alias),
                prefix.format('INVOICES', key),
                window,
                crit=1,
                aggr='sum',
                less=True
            ),
        )
        res.append(
            _summarize_check(
                'requests_count_{0}'.format(alias),
                prefix.format('REQUESTS', key),
                window,
                crit=1,
                aggr='sum',
                less=True
            ),
        )

    return res


def unpaid_instant_invoices_pct():
    res = []

    prefix = "keepLastValue(one_min.greedaggrprod_yandex_ru.simplebrother_balance_bo.BO." \
             "UNPAID_INSTANT_INVOICES_PCT.{0}, 20)"
    service2params = {
        'total': ('ALL_SERVICES', '30min', 30),
        'direct': ('PPC', '30min', 35),
        'market': ('market', '30min', 35),
        'auto_ru': ('auto_ru', '45min', 35),
        'cloud': ('cloud', '45min', 35),
    }

    for alias, (key, window, crit) in service2params.items():
        res.append(
            _summarize_check(
                'unpaid_instant_invoices_{0}'.format(alias),
                prefix.format(key),
                window,
                crit
            ),
        )

    return res


def unfinished_requests_pct():
    res = []

    prefix = "keepLastValue(one_min.greedaggrprod_yandex_ru.simplebrother_balance_bo.BO." \
             "UNFINISHED_REQUESTS_PCT.{0}, 20)"
    service2params = {
        'direct': ('PPC', '30min', 15),
        # 'auto_ru': ('auto_ru', '30min', 15),
        'cloud': ('cloud', '30min', 15),
        'market': ('market', '1h', 15),
    }

    for alias, (key, window, crit) in service2params.items():
        res.append(
            _summarize_check(
                'unfinished_requests_{0}'.format(alias),
                prefix.format(key),
                window,
                crit
            ),
        )

    return res


def queues():
    prefix = 'keepLastValue(offset(sumSeries(one_min.greedaggrprod_yandex_ru.' \
             'simplebrother_balance_bo.BO.EXPORT_STATE_0.CURRENT.{0}), 0), 10)'
    process_payments = prefix.format('PROCESS_PAYMENTS')
    oebs = prefix.format('OEBS')
    oebs_api = prefix.format('OEBS_API')
    process_cache = prefix.format('PROCESS_CACHE')
    thirdparty_trans = prefix.format('THIRDPARTY_TRANS')
    # process_taxi = prefix.format('PROCESS_TAXI')
    process_completion = prefix.format('PROCESS_COMPLETION')

    notifications = 'keepLastValue(sumSeries(one_min.greedaggrprod_yandex_ru.simplebrother_balance_bo.BO.TON.*), 10)'

    return [
        _summarize_check('process_payments_queue', process_payments, '2h', 5000),
        _summarize_check('oebs_queue', oebs, '1h', 100000),
        _summarize_check('oebs_api_queue', oebs_api, '1h', 100000),
        _summarize_check('process_cache_queue', process_cache, '1h', 1000),
        _summarize_check('thirdparty_trans_queue', thirdparty_trans, '1h', 250000),
        # This alert is flaky. Will be hidden for 1 month (till 12/09/2020). This queue should be optimized.
        # _summarize_check('process_taxi_queue', process_taxi, '1h', 250000),
        _summarize_check('process_completion_queue', process_completion, '30min', 1000),
        _summarize_check('notifications_queue', notifications, '30min', 20000),
    ]


def delays():
    prefix = "maxSeries(one_min.greedaggrprod_yandex_ru.simplebrother_balance_bo.BO.KPI_SLA.{0})"

    max_shipm_delay = prefix.format('max_shipm_delay')

    return [
        _summarize_check('max_shipm_delay', max_shipm_delay, '30min', 40),
    ]


def checks():
    return merge(*itertools.chain(
        response_timings(),
        response_timing_by_dc(),
        invoices_count(),
        unpaid_instant_invoices_pct(),
        unfinished_requests_pct(),
        queues(),
        delays(),
    ))
