from paysys.sre.tools.monitorings.lib.checks.active.http import https_cert
from paysys.sre.tools.monitorings.lib.checks.base import unreachable
from paysys.sre.tools.monitorings.lib.util.aggregators import empty_kwargs
from paysys.sre.tools.monitorings.lib.util.helpers import (
    merge,
    gen_children_from_tuples,
    gen_children_deploy,
    make_aggregated_check,
    check,
    flaps
)
from paysys.sre.tools.monitorings.lib.util.yasm import yasm_deploy_threshold_alert

MAX_ALERT_NAME_LENGTH = 128


def l3(service, geos=[]):
    children = []
    for geo in geos:
        children.append({'service': service, 'host': geo})
    hosts = [(geo, service, 'HOST') for geo in geos]
    result = check('L3', gen_children_from_tuples(hosts), empty_kwargs)
    return result


def _get_juggler_check_name(prj, ctype, itype, geo, check_name):
    items = [prj]
    if ctype != 'none':
        items.append(ctype)
    if itype != 'deploy':
        items.append(itype)
    items.append(check_name)
    items.append(geo)

    name = "_".join(items)

    if len(name) > MAX_ALERT_NAME_LENGTH:
        raise ValueError("too long check name {}".format(name))

    return name


def _gen_yasm_check(prj, ctype, itype, geo, signal, check_name, warn, crit, stable=None, critical=None):
    if geo in ["myt", "iva"]:
        geo = "msk"
    yasm_check = {
        _get_juggler_check_name(prj, ctype, itype, geo, check_name): merge(
            yasm_deploy_threshold_alert(
                signal_name=signal,
                namespace=prj,
                warn_threshold=warn,
                crit_threshold=crit,
                geo=geo,
                ctype='none'),
            flaps(stable, critical)
        )
    }
    return yasm_check


def nginx_monitorings(geos, prj, ctype, itype):
    assert isinstance(prj, str)
    assert isinstance(ctype, str)
    assert isinstance(itype, str)
    assert isinstance(geos, list) and all(isinstance(geo, str) for geo in geos)

    result = list()
    for geo in geos:
        common_args = {
            "prj": prj,
            "ctype": ctype,
            "itype": itype,
            "geo": geo,
        }

        result.extend([
            _gen_yasm_check(
                signal="""perc(sum(
                       diehard-nginx-request_500_count_deee,
                       diehard-nginx-request_502_count_deee,
                       diehard-nginx-request_504_count_deee),
                       diehard-nginx-request_count_deee)
                       """,
                check_name='5xx_percent',
                warn=5,
                crit=10,
                stable=20,
                critical=100,
                **common_args
            ),
            _gen_yasm_check(
                signal="""perc(sum(
                   diehard-nginx-request_400_count_deee,
                   diehard-nginx-request_401_count_deee,
                   diehard-nginx-request_404_count_deee,
                   diehard-nginx-request_499_count_deee),
                   diehard-nginx-request_count_deee)
                   """,
                check_name='4xx_percent',
                warn=5,
                crit=10,
                stable=20,
                critical=100,
                **common_args
            ),
            ]
        )
    r = merge(*result)
    return r


def cert(name, host, port=443):
    return merge(
        https_cert(name, port=port),
        {name: empty_kwargs},
        {name: gen_children_from_tuples([(host, name, "HOST")])},
    )


def get_checks(children):
    return merge(
        make_aggregated_check(check=unreachable, percent=40),
        check('UNREACHABLE', gen_children_deploy(children, 'UNREACHABLE')),
    )
