from itertools import chain

from paysys.sre.tools.monitorings.configs.diehard.base import common
from paysys.sre.tools.monitorings.lib.checks import services
from paysys.sre.tools.monitorings.lib.checks.base import cpu_usage
from paysys.sre.tools.monitorings.lib.checks.doc import doc
from paysys.sre.tools.monitorings.lib.util import conductor
from paysys.sre.tools.monitorings.lib.checks.graphite import graphite_check
from paysys.sre.tools.monitorings.lib.util.helpers import (
    check,
    flaps,
    gen_children_from_tuples,
    merge,
    ttl,
)

nginx_status_doc = doc(
    "https://nginx.org/ru/docs/http/ngx_http_stub_status_module.html#data",
    "Nginx server status variables",
)


def nginx_status(variable, children, warn=50, crit=100, window="-20min", less=False):
    # See description of server status variables at
    # https://nginx.org/ru/docs/http/ngx_http_stub_status_module.html#data

    check_name = "nginx-metric_{}".format(variable)
    metric_tmpl = "one_min.{host}.nginx.{var}"
    hosts = [
        h.fqdn for h in chain(*[conductor.get_hosts_in_group(c) for c in children])
    ]

    subchecks = {}
    for host in hosts:
        metric = metric_tmpl.format(host=host.replace(".", "_"), var=variable)
        subchecks[host] = merge(
            graphite_check(check_name, metric, crit, warn, window, less=less),
            check(check_name, flaps(60, 120), {"children": []}, nginx_status_doc),
        )

    return check(
        check_name,
        flaps(60, 120),
        {"tags": ["subchecks"]},
        {"subchecks": subchecks},
        nginx_status_doc,
        gen_children_from_tuples([host, check_name, "HOST"] for host in hosts),
    )


def get_checks(children, exclude_common):
    return merge(
        common.get_checks(children, exclude_common),
        cpu_usage(children),
        services.nginx_alive,
        services.yandex_passport_tvmtool,
        services.yandex_duckgo,
        {"cp-cardform": merge(ttl(300, 120), flaps(60, 120))},
        {"cp-pyconfpatch": merge(ttl(300, 120), flaps(60, 120))},
        {"cp-pykeyapi": merge(ttl(300, 120), flaps(60, 120))},
        {"cp-pykeykeeper2": merge(ttl(300, 120), flaps(60, 120))},
        {"cp-pytokenizer": merge(ttl(300, 120), flaps(60, 120))},
        {"cp-pytokenizer-internal": merge(ttl(300, 120), flaps(60, 120))},
        {"zookeeper": ttl(300, 120)},
        {"yandex-duckgo-secrets": ttl(7200, 3600)},
        {"mastercard-keys": ttl(7200, 3600)}
    )
