from itertools import chain

from paysys.sre.tools.monitorings.lib.checks.graphite import graphite_check
from paysys.sre.tools.monitorings.configs.diehard.base import common
from paysys.sre.tools.monitorings.lib.checks import galera
from paysys.sre.tools.monitorings.lib.checks.doc import doc
from paysys.sre.tools.monitorings.lib.util import conductor
from paysys.sre.tools.monitorings.lib.util.helpers import (
    check,
    flaps,
    gen_children_from_tuples,
    merge,
    ttl,
)


mysql_status_doc = doc(
    "https://dev.mysql.com/doc/refman/5.7/en/server-status-variables.html",
    "MySQL server status variables",
)


def mysql_status(variable, children, warn=50, crit=100, window="-20min"):
    # See description of server status variables at
    # https://dev.mysql.com/doc/refman/5.7/en/server-status-variables.html

    check_name = "mysql-metric_{}".format(variable)
    metric_tmpl = "perSecond(one_min.{host}.mysql.common.{var})"
    hosts = [
        h.fqdn for h in chain(*[conductor.get_hosts_in_group(c) for c in children])
    ]

    subchecks = {}
    for host in hosts:
        metric = metric_tmpl.format(host=host.replace(".", "_"), var=variable)
        subchecks[host] = merge(
            graphite_check(check_name, metric, crit, warn, window),
            check(check_name, flaps(120, 300), {"children": []}, mysql_status_doc),
        )

    return check(
        check_name,
        flaps(120, 300),
        {"tags": ["subchecks"]},
        {"subchecks": subchecks},
        mysql_status_doc,
        gen_children_from_tuples([host, check_name, "HOST"] for host in hosts),
    )


def mysql_free_connections(children, warn=0.7, crit=0.8, window="-20min"):
    check_name = "mysql-connections-free"
    metric_tmpl = "divideSeries(one_min.{host}.mysql.common.Threads_connected,one_min.{host}.mysql.common.max_connections)"
    hosts = [
        h.fqdn for h in chain(*[conductor.get_hosts_in_group(c) for c in children])
    ]

    subchecks = {}
    for host in hosts:
        metric = metric_tmpl.format(host=host.replace(".", "_"))
        subchecks[host] = merge(
            graphite_check(check_name, metric, crit, warn, window),
            check(check_name, {"children": []}),
        )

    return check(
        check_name,
        {"tags": ["subchecks"]},
        {"subchecks": subchecks},
        gen_children_from_tuples([host, check_name, "HOST"] for host in hosts),
    )


def get_checks(children, exclude_common):
    return merge(
        common.get_checks(children, exclude_common),
        galera.cluster(ttl(300, 120)),
    )
