# coding=utf-8
import juggler_sdk
import solomon_client
from paysys.sre.tools.monitorings.lib.util.aggregators import logic_and, downtime_skip
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.checks.base import unreachable
from paysys.sre.tools.monitorings.lib.notifications import (
    FIN_TOOLS_ESCALATION_USERS,
    FIN_TOOLS_TELEGRAM_GROUP,
    Notifications
)
from paysys.sre.tools.monitorings.lib.util.helpers import (
    solomon_check,
    check, ttl,
    unreach_skip,
    merge,
    flaps,
    update_checks,
    empty_kwargs,
    empty_children,
    gen_children,
    gen_children_deploy,
)

from paysys.sre.tools.monitorings.configs.dwh.base import dwh
from paysys.sre.tools.monitorings.configs.mdh.common import Production, PgCluster

annotations_yt_recency = {
    "description": "{{#isAlarm}}Mview \"{{labels.mview}}\" has not been updated in yt for {{expression.hours}} hours. {{/isAlarm}}\n"
                   "{{#isWarn}}Mview \"{{labels.mview}}\" has not been updated in yt for {{expression.hours}} hours. {{/isWarn}}\n"
                   "{{#isOk}}Mview \"{{labels.mview}}\" has been updated in yt recently.{{/isOk}}\n"
}
annotations_oracle_recency = {
    "description": "{{#isAlarm}}Mview \"{{labels.mview}}\" has not been updated in oracle for {{expression.minutes}} minutes. {{/isAlarm}}\n"
                   "{{#isWarn}}Mview \"{{labels.mview}}\" has not been updated in oracle for {{expression.minutes}} minutes. {{/isWarn}}\n"
                   "{{#isOk}}Mview \"{{labels.mview}}\" has been updated in oracle recently.{{/isOk}}\n"
}

annotations_row_count = {
    "description": "{{#isAlarm}}Row count for mview \"{{labels.mview}}\": {{expression.diff}} is above critical threshold {{expression.alarm_limit}}!{{/isAlarm}}\n"
                   "{{#isWarn}}Row count for mview \"{{labels.mview}}\": {{expression.diff}} is above warn threshold {{expression.warn_limit}}!{{/isWarn}}\n"
                   "{{#isOk}}Row count for mview \"{{labels.mview}}\" is OK.{{/isOk}}\n"
}

crossvalidation_alert_desc_too_big_too_small = """{{#isAlarm}}
Total acts with too big or too small amount in {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}": {{expression.last_point}} is above critical threshold {{expression.alarm_limit}}!
{{/isAlarm}}
{{#isWarn}}
Total acts with too big or too small amount in {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}": {{expression.last_point}} is above warn threshold {{expression.warn_limit}}
{{/isWarn}}
{{#isOk}}
Total acts with too big or too small amount in {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}": {{expression.last_point}} is OK
{{/isOk}}
{{#isNoData}}
Data for total acts with too big or too small amount in {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}" is missing
{{/isNoData}}
Alert status dashboard: https://solomon.yandex-team.ru/admin/projects/dwh/alerts/dwh-prod-crossvalidation_too_big_too_small/subAlerts?view=tiles
Alert settings: https://solomon.yandex-team.ru/admin/projects/dwh/alerts/dwh-prod-crossvalidation_too_big_too_small
"""
crossvalidation_alert_desc_sums_divergence = \
    '{{#isAlarm}}\n' \
    'In path {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}" sums in subdir "{{labels.dir1}}" and "{{labels.dir2}}" (sliced by {{labels.slice}}) diverged critically! ' \
    '({{expression.last_point}} > {{expression.alarm_limit}})\n' \
    '{{/isAlarm}}\n' \
    '{{#isWarn}}\n' \
    'In path {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}" sums in subdir "{{labels.dir1}}" and "{{labels.dir2}}" (sliced by {{labels.slice}}) diverged ' \
    '({{expression.last_point}} > {{expression.warn_limit}})\n' \
    """{{/isWarn}}
{{#isOk}}
In path {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}" sums in subdir "{{labels.dir1}}" and "{{labels.dir2}}" (sliced by {{labels.slice}}) are OK ({{expression.last_point}})
{{/isOk}}
{{#isNoData}}
Data for diverged sums in path {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}" in subdirs "{{labels.dir1}}" and "{{labels.dir2}}" (sliced by {{labels.slice}}) is missing
{{/isNoData}}
Alert status dashboard: https://solomon.yandex-team.ru/admin/projects/dwh/alerts/dwh-prod-crossvalidation_sums_divergence/subAlerts?view=tiles
Alert settings: https://solomon.yandex-team.ru/admin/projects/dwh/alerts/dwh-prod-crossvalidation_sums_divergence
"""
crossvalidation_alerts_lost_acts_count = """{{#isAlarm}}
There are {{expression.last_point}} acts in "//home/balance/prod/{{labels.dir2}}" which are not in "//home/balance/prod/{{labels.dir1}}" on cluster {{labels.yt_cluster}}!
{{/isAlarm}}
{{#isWarn}}
There are {{expression.last_point}} acts in "//home/balance/prod/{{labels.dir2}}" which are not in "//home/balance/prod/{{labels.dir1}}" on cluster {{labels.yt_cluster}}!
{{/isWarn}}
{{#isOk}}
There are {{expression.last_point}} acts in "//home/balance/prod/{{labels.dir2}}" which are not in "//home/balance/prod/{{labels.dir1}}" on cluster {{labels.yt_cluster}}.
{{/isOk}}
{{#isNoData}}
Data for missing acts at {{labels.yt_cluster}}."//home/balance/prod/{{labels.dir1}}" is missing
{{/isNoData}}
Alert status dashboard: https://solomon.yandex-team.ru/admin/projects/dwh/alerts/dwh-prod-crossvalidation_lost_acts_count/subAlerts?view=tiles
Alert settings: https://solomon.yandex-team.ru/admin/projects/dwh/alerts/dwh-prod-crossvalidation_lost_acts_count
"""

annotations_healthchecker = """{{#isAlarm}}
Healthchecker for ch_balance on {{expression.cluster}}: 0 working instances instead of {{expression.warn_threshold}}
{{/isAlarm}}
{{#isWarn}}
Healthchecker for ch_balance on {{expression.cluster}}: {{expression.healthcheck}} working instances instead of {{expression.warn_threshold}}
{{/isWarn}}
{{#isOk}}
Healthchecker for ch_balance on {{expression.cluster}}: {{expression.healthcheck}} working instances
{{/isOk}}
"""

annotations_db_structure = {
    "description":
        "{{#isAlarm}}Found structure difference!{{/isAlarm}}"
        "{{#isOk}}Structure is OK.{{/isOk}}",
    "details":
        "{{#expression.is_empty}}ok{{/expression.is_empty}}"
        "{{^expression.is_empty}}{{expression.failed_tables_labels}}{{/expression.is_empty}}"
}

annotations_dwh_work_duration = {
    "description":
        "{{#isAlarm}}Work {{labels.type}} {{labels.params}} duration is {{expression.minutes}} minutes!{{/isAlarm}}"
        "{{#isWarn}}Work {{labels.type}} {{labels.params}} duration is {{expression.minutes}} minutes!{{/isWarn}}"
        "{{#isOk}}All works are OK.{{/isOk}}",
}
crossvalidation_alerts_nans_count = """
{{#isAlarm}}In cluster {{labels.yt_cluster}}, "//home/balance/prod/{{labels.dir1}}" of prev month has {{expression.last_point}} NaNs{{/isAlarm}}
{{#isWarn}}In cluster {{labels.yt_cluster}}, "//home/balance/prod/{{labels.dir1}}" of prev month has {{expression.last_point}} NaNs{{/isWarn}}
{{#isOk}}No NaNs in cluster {{labels.yt_cluster}}, "//home/balance/prod/{{labels.dir1}}" of prev month.{{/isOk}}
"""

cfg = dwh.DwhConfig(env=Production(
    notify_iron=FIN_TOOLS_ESCALATION_USERS,
    notify_telegram=FIN_TOOLS_TELEGRAM_GROUP,
))

host = cfg.host_alias
namespace = cfg.project
children = empty_children
children_common = cfg.children_all

reports = "balance-reports"

notifications = Notifications()
notifications._telegram_default_options["status"] = ["CRIT", "WARN", "OK"]
notifications.set_iron_woman(logins=FIN_TOOLS_ESCALATION_USERS, delay=1200) \
    .set_telegram(group=FIN_TOOLS_TELEGRAM_GROUP)


def checks_common():
    yt_recency = _get_yt_recency()
    oracle_recency = _get_oracle_recency()
    solomons = merge(
        cfg.get_checks(
            pg_cluster=PgCluster('mdbmqatu39ljeahkeb62', size='nano'),
        ),
        yt_recency,
        oracle_recency,
        _get_mapnodes_curr_month_row_count_diff("yt"),
        _get_mapnodes_prev_months_row_count_diff("yt"),
        _get_tables_rows_count_diff('yt', '1d', '2d'),
        _get_tables_rows_count_diff('yt', '1d', '4w'),
        _get_clique_healthchecker("hahn"),
        _get_clique_healthchecker("arnold"),
        _get_db_structure(),
        _get_dwh_work_duration(),
        update_checks(
            merge(
                _get_crossvalidation_multialert('sums_divergence', ['yt_cluster', 'dir1', 'dir2', 'slice'],
                                                warn_threshold=1000,
                                                crit_threshold=10000,
                                                annotations_desc=crossvalidation_alert_desc_sums_divergence),
                _get_crossvalidation_multialert('too_big_too_small', ['yt_cluster', 'dir1'],
                                                warn_threshold=100,
                                                crit_threshold=1000,
                                                annotations_desc=crossvalidation_alert_desc_too_big_too_small),
                _get_crossvalidation_multialert('lost_acts_count', ['yt_cluster', 'dir1', 'dir2'],
                                                warn_threshold=0,
                                                crit_threshold=0,
                                                annotations_desc=crossvalidation_alerts_lost_acts_count),
                _get_crossvalidation_multialert('nans_count', ['yt_cluster', 'dir1'],
                                                warn_threshold=0,
                                                crit_threshold=0,
                                                annotations_desc=crossvalidation_alerts_nans_count)
            ),
            merge(
                empty_kwargs,
                notifications.telegram
            )
        )
    )
    for val in solomons.values():
        val["aggregator_kwargs"] = merge(val.get("aggregator_kwargs", {}), {"nodata_mode": "force_ok"})

    return merge(
        solomons,
        *[check(key, gen_children_deploy(children_common, key)) for key in solomons]
    )


def checks_meta():
    # PAYSYSDBA-4474

    children_meta = ['meta-rdbms']
    meta = merge(
        check('fk-indexing-meta', merge(ttl(3700, 300), logic_and, unreach_skip, downtime_skip)),
        check('matview-monitor-meta', merge(ttl(3700, 300), logic_and, unreach_skip, downtime_skip)),
        check('mviewlog-usage-meta', merge(flaps(300, 900), ttl(3700, 1800), logic_and, unreach_skip, downtime_skip)),
        check('matview-refresh-meta', merge(ttl(3700, 300), logic_and, unreach_skip, downtime_skip)),
        check('seq-check-meta', merge(ttl(3700, 300), logic_and, unreach_skip, downtime_skip)),
        check('check-job-sales-dayly', merge(ttl(900, 800), logic_and, unreach_skip, downtime_skip)),
        check('chg-mv-master-tab-meta', merge(ttl(900, 300), logic_and, unreach_skip, downtime_skip)),
        check('zero-record-mviews-meta',
              merge(ttl(3720, 60), flaps(600, 7200), logic_and, unreach_skip, downtime_skip)),
    )
    meta = merge(
        meta,
        *[check(key, gen_children(children_meta, key)) for key in meta]
    )

    children_key_dbf = ['key-db1f.paysys.yandex.net', 'key-db2f.paysys.yandex.net']
    children_key_dbv = ['key-db1v.paysys.yandex.net', 'key-db2v.paysys.yandex.net']
    children_key_dbh = ['key-db1h.paysys.yandex.net', 'key-db2h.paysys.yandex.net']
    mvdb = merge(
        check('mvdb-services-dbf', merge(ttl(350, 60), gen_children(children_key_dbf, 'mvdb-services', 'HOST'),
                                         logic_and, unreach_skip, downtime_skip)),
        check('mvdb-services-dbv', merge(ttl(350, 60), gen_children(children_key_dbv, 'mvdb-services', 'HOST'),
                                         logic_and, unreach_skip, downtime_skip)),
        check('mvdb-services-dbh', merge(ttl(350, 60), gen_children(children_key_dbh, 'mvdb-services', 'HOST'),
                                         logic_and, unreach_skip, downtime_skip)),
        check('mvdb-services', merge(flaps(300, 7200), gen_children(host, ['mvdb-services-dbf',
                                                                           'mvdb-services-dbv',
                                                                           'mvdb-services-dbh'], 'HOST'))),

        check('matview-refresh-mvdb-dbf',
              merge(ttl(350, 60), gen_children(children_key_dbf, 'matview-refresh-mvdb', 'HOST'),
                    logic_and, unreach_skip, downtime_skip)),
        check('matview-refresh-mvdb-dbv',
              merge(ttl(350, 60), gen_children(children_key_dbv, 'matview-refresh-mvdb', 'HOST'),
                    logic_and, unreach_skip, downtime_skip)),
        check('matview-refresh-mvdb-dbh',
              merge(ttl(350, 60), gen_children(children_key_dbh, 'matview-refresh-mvdb', 'HOST'),
                    logic_and, unreach_skip, downtime_skip)),
        check('matview-refresh-mvdb', merge(flaps(300, 7200), gen_children(host, ['matview-refresh-mvdb-dbf',
                                                                                  'matview-refresh-mvdb-dbv',
                                                                                  'matview-refresh-mvdb-dbh'],
                                                                           'HOST'))),

        check('matview-monitor-mvdb-dbf',
              merge(ttl(350, 60), gen_children(children_key_dbf, 'matview-monitor-mvdb', 'HOST'),
                    logic_and, unreach_skip, downtime_skip)),
        check('matview-monitor-mvdb-dbv',
              merge(ttl(350, 60), gen_children(children_key_dbv, 'matview-monitor-mvdb', 'HOST'),
                    logic_and, unreach_skip, downtime_skip)),
        check('matview-monitor-mvdb-dbh',
              merge(ttl(350, 60), gen_children(children_key_dbh, 'matview-monitor-mvdb', 'HOST'),
                    logic_and, unreach_skip, downtime_skip)),
        check('matview-monitor-mvdb', merge(flaps(300, 7200), gen_children(host, ['matview-monitor-mvdb-dbf',
                                                                                  'matview-monitor-mvdb-dbv',
                                                                                  'matview-monitor-mvdb-dbh'],
                                                                           'HOST'))),
    )

    unreachable_ = merge(
        unreachable,
        check('UNREACHABLE', gen_children(children_meta, 'UNREACHABLE', 'CGROUP')),
        check('UNREACHABLE',
              gen_children(children_key_dbf + children_key_dbv + children_key_dbh, 'UNREACHABLE', 'HOST'))
    )

    return merge(meta, mvdb, unreachable_)


def checks():
    return merge(
        checks_common(),
        checks_meta(),
    )


def _get_yt_recency():
    """
        Формирует проверку на свежесть агрегатов
        Агрегаты делятся на группы в зависимости от SLA, для каждой группы устанавливаются свои warn и alarm пороги.
    """
    source = "yt"
    check_name = "{}-aggregates-recency".format(source)
    expr = solomon_expression_custom(
        program_str=(

            'let skip_mview = "bindings|card|V_OPT_2015_ACTS|v_ar_rewards_export|acts|paysup_contracts";\n' +
            'let data = {{project="balance-reports", cluster="Push", sensor="recency", service="YbReports", mview!="{{{{skip_mview}}}}", source="{}"}};\n'.format(
                source) +
            'let last_point = last(data);\n' +

            'let sla_4 = get_label(data, "mview") == "mv_f_sal_today";\n'

            'let sla_8 = get_label(data, "mview") == "t_refund";\n'
            'let sla_9 = get_label(data, "mview") == "t_payment_fraud_status" || get_label(data, "mview") == "t_payment_register_line";\n'
            'let sla_10 = get_label(data, "mview") == "t_consume" || get_label(data, "mview") == "t_order" '
            '|| get_label(data, "mview") == "t_shipment";\n'
            'let sla_10_small = get_label(data, "mview") == "t_ccard_bound_payment";\n'
            'let sla_11 = get_label(data, "mview") == "t_client";\n'

            'let sla_12 = get_label(data, "mview") == "consume_attributes" || get_label(data, "mview") == "t_payment" || '
            'get_label(data, "mview") == "t_passport";\n'
            'let sla_12_acts = get_label(data, "mview") == "tariffied_acts" || get_label(data, "mview") == "tariffied_acts_agg" ' +
            '|| get_label(data, "mview") == "tariffied_acts_no_netting" || get_label(data, "mview") == "tariffied_acts_no_netting_agg";\n'
            'let sla_12_small = get_label(data, "mview") == "t_payment_method" || get_label(data, "mview") == "t_processing" ||'
            'get_label(data, "mview") == "t_service" || get_label(data, "mview") == "t_terminal";\n'

            'let sla_20 = get_label(data, "mview") == "cooked_payments" || get_label(data, "mview") == "f_sales_daily" || '
            'get_label(data, "mview") == "group_order_act_div";\n'

            'let hours = last_point / 60 / 60;\n\n' +
            'let warn_threshold_hours = sla_4 ? 2 : (sla_8 ? 6 : (sla_9 ? 7 : (sla_10 ? 8 : '
            '(sla_10_small ? 7 : (sla_11 ? 9 : (sla_12 ? 10 : (sla_12_acts ? 8 : '
            '(sla_12_small ? 9 : (sla_20 ? 18 : 26)))))))));\n' +
            'let alarm_threshold_hours = sla_4 ? 4 : (sla_8 ? 8 : (sla_9 ? 9 : (sla_10 ? 10 : '
            '(sla_10_small ? 10 : (sla_11 ? 11 : (sla_12 ? 12 : (sla_12_acts ? 12 : '
            '(sla_12_small ? 12 : (sla_20 ? 20 : 36)))))))));\n' +

            'alarm_if(last_point > alarm_threshold_hours*60*60);\n' +
            'warn_if(last_point > warn_threshold_hours*60*60);\n'
        ),
        project_id=namespace,
        annotations=annotations_yt_recency,
        description="{} aggregates recency".format(source),
        window_secs=30 * 60,
        group_by_labels=['mview'],
        juggler_children={"replace": True, "children": [
            juggler_sdk.Child("dwh-prod-{}".format(check_name), "all", "all", "MONITORING_MULTIALERT")]}
    )
    return solomon_check(check_name, expr)


def _get_oracle_recency():
    """
        Формирует проверку на свежесть агрегатов
        Агрегаты делятся на группы в зависимости от SLA, для каждой группы устанавливаются свои warn и alarm пороги.
    """
    source = "oracle"
    check_name = "oracle-aggregates-recency"
    expr = solomon_expression_custom(
        program_str=(

            'let skip_mview = "bindings|card|V_OPT_2015_ACTS|v_ar_rewards_export|acts|paysup_contracts";\n' +
            'let data = {{project="balance-reports", cluster="Push", sensor="recency", service="YbReports", mview!="{{{{skip_mview}}}}", source="{}"}};\n'.format(
                source) +
            'let last_point = last(data);\n'

            'let sla_25 = get_label(data, "mview") == "T_CONTACT" || get_label(data, "mview") == "T_PAYCASH_PAYMENT_V3" || '
            '\tget_label(data, "mview") == "T_INVOICE_REPAYMENT" || get_label(data, "mview") == "T_PAYMENT_TERM" ||'
            '\tget_label(data, "mview") == "T_RECEIPT" || get_label(data, "mview") == "T_INVOICE_ORDER" ||'
            '\tget_label(data, "mview") == "T_REGION_CODES" || get_label(data, "mview") == "T_1C_DOCUMENTS_V2" ||'
            '\tget_label(data, "mview") == "T_REVERSE" || get_label(data, "mview") == "T_EXTPROPS" ||'
            '\tget_label(data, "mview") == "T_MKB_CATEGORY" || get_label(data, "mview") == "T_BAD_DEBT_ACT" || '
            '\tget_label(data, "mview") == "T_INVOICE" || get_label(data, "mview") == "T_1C_PAYMENTS_V2" || '
            '\tget_label(data, "mview") == "T_CLIENT_GROUP" || get_label(data, "mview") == "T_GROUP" || '
            '\tget_label(data, "mview") == "T_CONTRACT_COMSN_TYPE" || get_label(data, "mview") == "T_SUPERCOMMISION_TYPE" || '
            '\tget_label(data, "mview") == "T_CONTRACT_COLLATERAL_TYPES" || get_label(data, "mview") == "T_OVERDRAFT_HISTORY" || '
            '\tget_label(data, "mview") == "T_PAID_ACTS" || get_label(data, "mview") == "T_CONFIG" || '
            '\tget_label(data, "mview") == "T_ROLE" || get_label(data, "mview") == "T_FIRM" || '
            '\tget_label(data, "mview") == "T_PERMISSION" || get_label(data, "mview") == "T_PLACE" || '
            '\tget_label(data, "mview") == "T_CORRECTION_PAYMENT" || get_label(data, "mview") == "T_PREPAYMENT_TRANSFERS" || '
            '\tget_label(data, "mview") == "T_PREPAYMENT_TRANSFER_REPORT" || get_label(data, "mview") == "T_ACT_INTERNALS" || '
            '\tget_label(data, "mview") == "T_COUNTRY_SERVICE_DATA" || get_label(data, "mview") == "T_PAGE_DATA" || '
            '\tget_label(data, "mview") == "T_COUNTRY" || get_label(data, "mview") == "T_EDO_TYPES" || '
            '\tget_label(data, "mview") == "T_BANK" || get_label(data, "mview") == "T_DISCOUNT_TYPE" || '
            '\tget_label(data, "mview") == "T_BUDGET_COEFF" || get_label(data, "mview") == "T_LANGUAGE" || '
            '\tget_label(data, "mview") == "T_BANK_INT_CLASS" || get_label(data, "mview") == "T_SERVICE_GROUP" || '
            '\tget_label(data, "mview") == "T_SERVICE_TERMINAL" || get_label(data, "mview") == "T_OEBS_CASH_PAYMENT_FACT" || '
            '\tget_label(data, "mview") == "T_PARTNER_ACT_DATA" || get_label(data, "mview") == "T_CONTRACT_COLLATERAL" || '
            '\tget_label(data, "mview") == "T_PAYSYS" || get_label(data, "mview") == "T_ENUMS";\n'

            'let minutes = last_point / 60;\n\n' +
            'let warn_threshold_minutes = sla_25? 15 : 1560;\n'
            'let alarm_threshold_minutes = sla_25? 25 : 2160;\n'

            'alarm_if(last_point > alarm_threshold_minutes*60);\n' +
            'warn_if(last_point > warn_threshold_minutes*60);\n'
        ),
        project_id=namespace,
        annotations=annotations_oracle_recency,
        description="{} aggregates recency".format(source),
        window_secs=30 * 60,
        group_by_labels=['mview'],
        juggler_children={"replace": True, "children": [
            juggler_sdk.Child("dwh-prod-{}".format(check_name), "all", "all", "MONITORING_MULTIALERT")]}
    )
    return solomon_check(check_name, expr)


def _get_tables_rows_count_diff(source, first_period, second_period):
    """
        Формирует проверку на расхождения кол-ва строк в таблицах между двумя переданными периодами
        Например, если переданы '1d' и '4w', будут проверяться значения за день назад и за 4 недели назад
    """
    check_name = "yt-rows-count-{}-{}".format(first_period, second_period)
    selectors = '{{project="{}", cluster="Push", service="YbReports", sensor="rows", mview!="bindings|card|mv_f_sal_today", source="{}"}}'.format(
        reports, source)
    expr = solomon_expression_custom(
        program_str=('let first = shift({}, {});\n'.format(selectors, first_period) +
                     'no_data_if(count(first) == 0);\n'
                     'let second = shift({}, {});\n'.format(selectors, second_period) +
                     'no_data_if(count(second) == 0);\n'
                     'let diff = abs(last(first) - last(second));\n' +
                     'let warn_limit = last(first) * 0.1;\n' +
                     'let alarm_limit = last(first) * 0.25;\n' +
                     'alarm_if(diff >= alarm_limit);\n' +
                     'warn_if(diff >= warn_limit);\n'
                     ),
        project_id=namespace,
        annotations=annotations_row_count,
        description="Row count difference {} ago and {} ago".format(first_period, second_period),
        window_secs=30 * 60,
        group_by_labels=['mview'],
        juggler_children={"replace": True, "children": [
            juggler_sdk.Child("dwh-prod-{}".format(check_name), "all", "all", "MONITORING_MULTIALERT")]}

    )

    return solomon_check(check_name, expr)


def _get_mapnodes_curr_month_row_count_diff(source):
    """
        Формирует проверку на расхождения кол-ва строк мапнод вчера и позавчера.
    """

    check_name = "yt-mapnodes-rows-count-1d-2d"
    selectors = '{{project="{}", cluster="Push", service="YbReports", sensor="rows-now", mview!="bindings|card|mv_f_sal_today", source="{}"}}'.format(
        reports, source)
    expr = solomon_expression_custom(
        program_str=('let day_ago = shift({}, 1d);\n'.format(selectors) +
                     'no_data_if(count(day_ago) == 0);\n'
                     'let two_days_ago = shift({}, 2d);\n'.format(selectors) +
                     'no_data_if(count(two_days_ago) == 0);\n'
                     'let diff = abs(last(day_ago) - last(two_days_ago));\n' +
                     'let warn_limit = last(day_ago) * 0.1;\n' +
                     'let alarm_limit = last(day_ago) * 0.25;\n' +
                     'alarm_if(diff >= alarm_limit);\n' +
                     'warn_if(diff >= warn_limit);\n'
                     ),
        project_id=namespace,
        annotations=annotations_row_count,
        description="Row count difference for mapnode 1 day ago and 2 days ago",
        window_secs=30 * 60,
        group_by_labels=['mview'],
        juggler_children={"replace": True, "children": [
            juggler_sdk.Child("dwh-prod-{}".format(check_name), "all", "all", "MONITORING_MULTIALERT")]}

    )
    return solomon_check(check_name, expr)


def _get_mapnodes_prev_months_row_count_diff(source):
    """
        Формирует проверку на расхождение кол-ва строк мапнод в предыдущем и предпредыдущем месяцах.
    """
    check_name = "yt-mapnodes-rows-count-prev-months"
    selectors = '{{{{project="{}", cluster="Push", service="YbReports", sensor="{{}}", mview!="bindings|card|mv_f_sal_today", source="{}"}}}}'.format(
        reports, source)
    expr = solomon_expression_custom(
        program_str=('let prev = {};\n'.format(selectors.format("rows-prev-month")) +
                     'no_data_if(count(prev) == 0);\n'
                     'let prev_prev = {};\n'.format(selectors.format("rows-prev-prev-month")) +
                     'no_data_if(count(prev_prev) == 0);\n'
                     'let diff = abs(last(prev) - last(prev_prev));\n' +
                     'let warn_limit = last(prev) * 0.1;\n' +
                     'let alarm_limit = last(prev) * 0.25;\n' +
                     'alarm_if(diff >= alarm_limit);\n' +
                     'warn_if(diff >= warn_limit);\n'
                     ),
        project_id=namespace,
        annotations=annotations_row_count,
        description="Row count difference for mapnode in the previous month and the month before that",
        window_secs=30 * 60,
        group_by_labels=['mview'],
        juggler_children={"replace": True, "children": [
            juggler_sdk.Child("dwh-prod-{}".format(check_name), "all", "all", "MONITORING_MULTIALERT")]}

    )
    return solomon_check(check_name, expr)


def _get_crossvalidation_multialert(sensor, group_by, warn_threshold, crit_threshold, annotations_desc):
    check_name = 'crossvalidation_{}'.format(sensor)
    expr = solomon_expression_custom(
        program_str=(
            'let data = {{ project="{}", prefix="home/balance/prod", service="dwh_crossvalidation", sensor="{}"}};\n'.format(
                reports, sensor) +
            'let last_point = last(data);\n'
            'let warn_limit = {};\n'.format(warn_threshold) +
            'let alarm_limit = {};\n'.format(crit_threshold) +
            'alarm_if(last_point > alarm_limit);\n'
            'warn_if(last_point > warn_limit);\n'
        ),
        project_id=namespace,
        annotations={
            "description": annotations_desc
        },
        description="Crossvalidation '{}'".format(sensor),
        window_secs=90 * 60,
        group_by_labels=group_by,
        juggler_children={
            "replace": True,
            "children": [
                juggler_sdk.Child("dwh-prod-{}".format(check_name), "all", "all", "MONITORING_MULTIALERT")
            ]
        },
        resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
        no_points_policy="NO_POINTS_NO_DATA",
        channels=[
            solomon_client.AssociatedChannel(
                id='solomon_alert',
                config=solomon_client.ChannelConfig(
                    notify_about_statuses=["NO_DATA", "ALARM", "WARN", "OK", "ERROR"],
                    repeat_delay_secs=0,
                )
            )
        ],
    )
    return solomon_check(check_name, expr)


def _get_clique_healthchecker(cluster):
    """
        Healthchecker работающих инстансов в клике ch_balance
    """
    expr = solomon_expression_custom(
        program_str=(
            'let total = {{project="yt", cluster="{}", '.format(cluster) +
            'service="clickhouse", sensor="yt.clickhouse.yt.health_checker.success", cookie="Aggr", operation_alias="ch_balance", query_index="*"};\n' +
            'let healthcheck = max(total);\n' +
            'let cluster = "{}";\n'.format(cluster) +
            'let warn_threshold = 4;\n' +
            'let alarm_threshold = 1;\n' +
            'alarm_if(healthcheck < alarm_threshold);\n' +
            'warn_if(healthcheck < warn_threshold);'

        ),
        project_id=namespace,
        annotations={
            "description": annotations_healthchecker
        },
        description="ch_balance healthchecker for cluster {}".format(cluster),
        window_secs=3 * 60
    )
    return solomon_check("ch_balance-healthchecker-{}".format(cluster), expr)


def _get_db_structure():
    """
    Алерты db_structure
    """

    expr = solomon_expression_custom(
        program_str=(
            'let data = {project="balance-reports", cluster="Push", sensor="db_structure", service="YbReports"};\n' +
            'let failed_tables = drop_empty_lines(drop_below(data, 1));\n'
            'let failed_tables_labels = map(failed_tables, x -> get_label(x, "mview"));\n' +
            'let is_empty = size(failed_tables_labels) == 0;\n' +
            'alarm_if(!is_empty);'
        ),
        project_id=namespace,
        annotations=annotations_db_structure,
        description="db_structure check",
        window_secs=20*60
    )
    return solomon_check("balance_db_structure_check", expr)


def _get_dwh_work_duration():
    """
    Алерты длительных работ в DWH
    """

    check_name = "dwh-work-duration"
    expr = solomon_expression_custom(
        program_str=(
            'let data = {project="dwh", cluster="default", service="push", sensor="works"};\n'
            'let last_point = last(data);\n'
            'let minutes = last_point / 60;\n'
            'let warn_threshold_minutes = 480;\n'
            'let alarm_threshold_minutes = 720;\n'
            'alarm_if(last_point > alarm_threshold_minutes*60);\n'
            'warn_if(last_point > warn_threshold_minutes*60);\n'
        ),
        project_id=namespace,
        annotations=annotations_dwh_work_duration,
        description="Work duration in DWH.",
        window_secs=20 * 60,
        group_by_labels=['dc', 'host', 'type', 'status', 'params'],
        juggler_children={"replace": True, "children": [
            juggler_sdk.Child("dwh-prod-{}".format(check_name), "all", "all", "MONITORING_MULTIALERT")]}
    )

    return solomon_check(check_name, expr)
