# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from copy import deepcopy
from functools import partial

from paysys.sre.tools.monitorings.lib.checks.active.http import http
from paysys.sre.tools.monitorings.lib.checks.base import unreachable, graphite_client
from paysys.sre.tools.monitorings.lib.checks.nanny import awacs, balancer_5xx
from paysys.sre.tools.monitorings.lib.checks.postgres import postgres
from paysys.sre.tools.monitorings.lib.checks.services import haproxy, pushclient_check
from paysys.sre.tools.monitorings.lib.notifications import Notifications
from paysys.sre.tools.monitorings.lib.util.helpers import (
    create_subchecks, make_aggregated_check, merge, check, gen_children_deploy, ttl)

if False:  # pragma: nocover
    from typing import Union, List, Callable, Optional, Generator, Tuple  # noqa


class WithTemplate(object):
    """Примесь, предлагающая инструменты для работы с шаблонами."""

    def _tpl_context_get(self, context=None):
        # type: (dict) -> dict
        context = (context or {}).copy()
        return context

    def _tpl_context_apply(self, val, context=None):
        # type: (str, dict) -> str

        if context is None:
            context = self._tpl_context_get(context or {})

        for context_key, context_val in context.items():
            val = val.replace('{%s}' % context_key, context_val)

        return val


class WithApplicableEnv(WithTemplate):

    def _apply_env(self, env, fmt):
        # type: (Environment, Callable) -> None
        pass

    def apply_env(self, env, context_base):
        # type: (Environment, dict) -> None
        """Применяет указанное окружение к текущему объекту.

        Реализация применения ожидается в ._apply_env().

        :param env: Объект окружения.
        :param context_base: Базовый контекст для шаблонов.

        """
        context = self._tpl_context_get(context_base)

        fmt = partial(self._tpl_context_apply, context=context)

        for key, value in context.items():
            # поддержим шаблонные переменные внутри значений контекста
            context[key] = fmt(value)

        self._apply_env(env=env, fmt=fmt)


class Environment(object):
    """Среда, в которой исполняется приложение."""

    is_production = False  # type: bool
    """Флаг, указывающий на то что среда боевая."""

    alias = ''
    """Псевдоним среды."""

    notify_iron = None  # type: List[str]
    """Кого оповещать по телефону (железная женщина)."""

    notify_telegram = None  # type: List[str]
    """Кого оповещать через Телеграм."""

    def __init__(self, notify_iron=None, notify_telegram=None):
        # type: (list[str], list[str]) -> None
        """

        :param notify_iron: Кого оповещать по телефону (железная женщина).
        :param notify_telegram: Кого оповещать через Телеграм.

        """
        assert self.alias
        self.notify_iron = notify_iron or self.notify_iron
        self.notify_telegram = notify_telegram or self.notify_telegram
        self.components = []  # type: List[Component]
        """Компоненты, адаптированные под данную среду."""

    def adopt_components(self, components, context):
        # type: (List[Component], dict) -> List[str]
        """Перенимает указанные компоненты с указанным базовым контекстом,
        инициализируя их для данной среды.

        :param components: Перечисление компонентов.
        :param context: Контекст для использования в шаблонах.

        """
        components = deepcopy(components)  # определения из конфига должны оставаться неизменными
        self.components = components

        children = []

        for component in components:
            component.apply_env(self, context)
            children.extend(component.children)

        return children

    def get_notifications(self, hosts=None):
        # type: (Optional[List[str]]) -> dict
        """Возвращает настройки оповещений."""

        hosts = hosts or list()

        notifications = {
            'default': Notifications().noop,
        }

        iron = self.notify_iron
        telegram = self.notify_telegram

        if any((iron, telegram)):

            notify = Notifications()
            prop = None

            if iron:
                prop = notify.iron_woman
                notify.set_iron_woman(delay=1200, logins=iron)

            if telegram:
                prop = notify.telegram
                notify.set_telegram(telegram)

            if iron and telegram:
                prop = notify.iron_woman_and_telegram

            if prop:
                for host in hosts:
                    notifications.setdefault('by_host', {})[host] = prop

        return notifications


class Testing(Environment):
    """Тестовое окружение."""

    alias = 'test'


class Production(Environment):
    """Боевое окружение."""

    is_production = True

    alias = 'prod'


class Component(WithApplicableEnv):
    """Компонент приложения.
    Например: бэкенд, фронтенд.

    """
    location = ['man', 'sas', 'vla']  # type: List[str]
    """Пседовнимы датацентров, в которых развернут компонент."""

    balancer_service = ''  # type: str
    """Определение сервисов балансера. Поддерживаются шаблоны."""

    def __init__(
        self,
        alias,
        fqdn='',
        balancer_service='',
        deploy_group='',
        port=8000,
        ports_balancer=None,
        location=None
    ):
        # type: (str, str, str, str, int, Tuple[List[int], List[int]], list[str]) -> None
        """

        Некоторые параметры дозволяют использовать шаблоны:

            * env - псведоним окружения
            * prj - псевдоним проекта

            * prj[-env] - проект и опционально минус и окружение
            * prj[_env] - проект и опционально подчерк и окружение
            * prj-env - проект и минус и окружение

            * dom_yteam - yandex-team.ru
            * dom_paysys - paysys.yandex.net

            * path_yteam - yandex-team_ru
            * path_paysys - paysys_yandex-team_ru

        :param alias: Псевдоним компонента

        :param fqdn: Полное различительное имя области компонента.
            Поддерживаются шаблоны.

        :param balancer_service: Определение сервисов балансера.
            Поддерживаются шаблоны.

        :param deploy_group: Имя группы машин в Деплое.
            Поддерживаются шаблоны.

        :param port: Порт компонента. Для пинга самого приложения.

        :param ports_balancer: Порты для проверки балансировщиков awacs.
            В виде кортежа (http, https).

        :param location: Псевдонимы датацентров, в которых развернут компонент.

        """
        location = location or self.location
        balancer_service = balancer_service or self.balancer_service or 'rtc_balancer_{prj[-env]}_{path_yteam}'

        self.ports_balancer = ports_balancer or ([80], [443])
        self.location = location
        self.alias = alias
        self.port = port
        self.fqdn = fqdn or '{prj[-env]}.{dom_yteam}'
        self.balancer_services = ['%s_%s' % (balancer_service, dc) for dc in location]
        self.deploy_group = deploy_group or '{deploy_group}'

    def __str__(self):
        return '%s@[%s] | %s:%s' % (self.alias, ', '.join(self.location), self.fqdn, self.port)

    def _tpl_context_get(self, context=None):
        context = super(Component, self)._tpl_context_get(context)
        context.update({
            'component': self.alias,
            'deploy_group': '{prj}@stage={prj-env}-stage;deploy_unit={component}',
        })
        return context

    def _apply_env(self, env, fmt):
        # type: (Environment, Callable) -> None

        self.fqdn = fmt(self.fqdn)

        self.balancer_services = [
            fmt(balancer_service)
            for balancer_service in self.balancer_services
        ]

        self.deploy_group = fmt(self.deploy_group)

    @property
    def children(self):
        # type: () -> List[str]
        """Дочерние элементы компонента в терминах Juggler."""
        return [self.deploy_group]


class CheckRule(object):
    """Правило проверки. Абстракция над словарём определения проверки для Juggler."""

    def __init__(self, check_dict, per_child):
        # type: (dict, bool) -> None
        """

        :param check_dict: Словарь определения проверки
        :param per_child: Флаг. Следует ли распространять проверку на дочерние элементы.

        """
        self.check = check_dict
        self.check_name = list(check_dict.keys())[0]
        self.per_child = per_child


class PgCluster(WithApplicableEnv):
    """Описывает кластер PostgreSQL."""

    def __init__(self, cluster, db_name='', size=''):
        # type: (str, str, str) -> None
        """

        :param cluster: Идентификатор кластера.

        :param db_name: Имя БД. Если не указано, будет произведена
            попытка вычислить по данным окружения.

        :param size: Имя предустановки размера БД:
            nano, micro, small, medium, large, xlarge, 2xlarge, 3xlarge, 4xlarge

        """
        self.cluster = cluster
        self.db_name = db_name
        self.size = size

    def _apply_env(self, env, fmt):
        # type: (Environment, Callable) -> None

        if not self.db_name:
            self.db_name = fmt('{prj[_env]}')

    @property
    def check_rule(self):
        # type: () -> CheckRule
        """Возвращает объект правила проверки для кластера."""
        kwargs = {}

        size = self.size

        # `cpu_wait_limit` на небольших размерах PG почти всегда будет гореть красным.
        # Чтобы избежать ожидаемых критов, мы поднимаем значения для этой проверки.
        if size == 'nano':
            kwargs.update({
                'cpu_wait_limit': 4,
            })

        elif size == 'micro':
            kwargs.update({
                'cpu_wait_limit': 2,
            })

        return CheckRule(postgres(self.cluster, db_name=self.db_name, **kwargs), per_child=False)


class ProjectConfig(WithApplicableEnv):
    """Конфигурация проекта."""

    project = ''  # type: str
    """Псевдоним проекта."""

    project_alt = ''  # type: str
    """Альтернативный псевдоним проекта.
    Используется в случаях, когда упоминание проекта в некоторых наименованих
    (например, в доменных именах) не совпадают с песоднимом проекта из .project

    """

    components = []  # type: list[Component]
    """Перечисление компонентов проекта."""

    check_haproxy = True  # type: bool
    """Флаг. Используется ли в проекте haproxy."""

    check_pushclient = True  # type: bool
    """Флаг. Используется ли в проекте push client."""

    check_graphite = False  # type: bool
    """Флаг. Используется ли в проекте клиент для Graphite."""

    def __init__(self, env):
        # type: (Environment) -> None
        """

        :param env: Объект окружения.

        """
        assert all((self.project, self.components)), 'Project alias and components must be defined'

        self.env = env

        context_base = self._tpl_context_get()
        self.context_base = context_base

        fmt = partial(self._tpl_context_apply, context=context_base)
        self.host_alias = fmt('{prj}.{env}')

        self.children_all = env.adopt_components(self.components, context=context_base)

    def get_defaults(self):
        # type: () -> dict
        """Возвращает умолчательные настройки для Juggler."""
        defaults = merge(
            ttl(620, 60),
            {'namespace': self.project},
        )
        return defaults

    def _tpl_context_get(self, context=None):
        context = super(ProjectConfig, self)._tpl_context_get(context)

        env = self.env
        env_alias = env.alias

        project = self.project
        project_alt = self.project_alt

        env_opt = '' if env.is_production else env.alias
        prj_env_opt = (project, env_opt)
        prjalt_env_opt = (project_alt, env_opt)

        context.update({
            'env': env_alias,
            '[-env]': ('-%s' % env_opt) if env_opt else '',

            'prj': project,
            'prj[-env]': ('%s-%s' % prj_env_opt).rstrip('-'),
            'prj[_env]': ('%s_%s' % prj_env_opt).rstrip('_'),
            'prj-env': '%s-%s' % (project, env_alias),

            'prjalt': project_alt,
            'prjalt[-env]': ('%s-%s' % prjalt_env_opt).rstrip('-'),

            'dom_yteam': 'yandex-team.ru',
            'dom_paysys': 'paysys.yandex.net',
            'dom_paysys_yteam': 'paysys.yandex-team.ru',
            'path_yteam': 'yandex-team_ru',
            'path_paysys': 'paysys_yandex-team_ru',
        })

        return context

    def get_component(self, alias):
        # type: (str) -> Optional[Component]
        """Возвращает компонент по указанному псевдониму,

        :param alias:

        """
        component = None

        for component in self.iter_components():
            if component.alias == alias:
                return component

        return component

    def iter_components(self):
        # type: () -> Generator[Component, None, None]
        """Выдаёт объекты компонентов для текущей конфигурации в текущей среде."""
        for component in self.env.components:
            yield component

    def _get_awacs_checks(self, host_alias, component):
        # type: (str, Component) -> dict

        component_fqdn = component.fqdn
        component_children = component.children

        return merge(
            awacs(
                host_alias,
                component_fqdn,
                component.balancer_services,
                http_ports=component.ports_balancer[0],
                https_ports=component.ports_balancer[1],
                ok_codes=[200, 301, 302],
                host=component_fqdn
            ),
            balancer_5xx(host_alias),
            http('/ping', component.port, ok_codes=[200, 301, 302], crit=0, headers={'Host': component_fqdn}),
            check('/ping', gen_children_deploy(component_children, '/ping')),
            unreachable,
            check('UNREACHABLE', gen_children_deploy(component_children, 'UNREACHABLE'))
        )

    def _get_rules_haproxy(self):
        # type: () -> list[CheckRule]
        backend = self.get_component('backend')

        if not backend:
            return []

        checks = [
            CheckRule(haproxy, per_child=False),
            CheckRule(check('haproxy', gen_children_deploy(backend.children, 'haproxy')), per_child=False),
        ]
        return checks

    def get_check_rules(self, pg_cluster=''):
        # type: (Union[str, PgCluster, List[Union[str, PgCluster]]]) -> list[CheckRule]
        """Возвращает объекты правил проверки.
        Может быть перекрыт наследником, для дополнения/изменения набора правил.

        :param pg_cluster: Определение для конфигурирвоания проверок кластера(ов) PG

        """
        context_base = self.context_base
        env = self.env

        checks = []

        checks.extend(
            CheckRule(
                create_subchecks(component.alias, component.fqdn, self._get_awacs_checks(self.host_alias, component)),
                per_child=False
            )
            for component in self.iter_components()
        )

        checks.extend([
            CheckRule(check('certs'), per_child=True),
            CheckRule(check('unispace'), per_child=True),
            CheckRule(make_aggregated_check(check=unreachable, percent=40), per_child=True),
        ])

        if self.check_graphite:
            checks.append(CheckRule(graphite_client, per_child=True))

        if self.check_pushclient:
            checks.append(CheckRule(pushclient_check, per_child=True))

        if pg_cluster:

            if isinstance(pg_cluster, str):
                pg_cluster = PgCluster(pg_cluster)

            if not isinstance(pg_cluster, list):
                pg_cluster = [pg_cluster]

            for cluster in pg_cluster:  # type: PgCluster
                cluster.apply_env(env, context_base=context_base)
                checks.append(cluster.check_rule)

        if self.check_haproxy:
            checks.extend(self._get_rules_haproxy())

        return checks

    def get_checks(self, pg_cluster=''):
        # type: (ProjectConfig, Union[str, PgCluster, List[Union[str, PgCluster]]]) -> dict
        """Возвращает словарь с определениями проверок для Juggler, yasm и пр.

        :param pg_cluster: Определение для конфигурирвоания проверок кластера(ов) PG

        """
        checks = []
        per_child_checks = []

        for rule in self.get_check_rules(pg_cluster=pg_cluster):
            assert isinstance(rule, CheckRule)

            checks.append(rule.check)
            if rule.per_child:
                per_child_checks.append(rule.check_name)

        checks.extend([
            check(alias, gen_children_deploy(self.children_all, alias))
            for alias in per_child_checks
        ])

        return merge(*checks)
