from paysys.sre.tools.monitorings.lib.checks.awacs import l7_monitoring
from paysys.sre.tools.monitorings.lib.checks.active.http import http, https
from paysys.sre.tools.monitorings.lib.checks.doc import doc_link, doc
from paysys.sre.tools.monitorings.lib.util.aggregators import logic_or
from paysys.sre.tools.monitorings.lib.util.helpers import merge, check, gen_children_deploy, ttl, flaps
from paysys.sre.tools.monitorings.lib.util.yasm import yasm_deploy_threshold_alert, \
    yasm_con_threshold_alert_func, yasm_deploy_threshold_alert_func

# Ссылка на текущего дежурного в доке Diod`a.
SWAT_DUTY = 'https://wiki.yandex-team.ru/balance/swat/#anchor-duty'

# Ссылка на алерты в YASM
# TODO: поменять проект с oplata.testing на oplata
YASM_ALERTS = 'https://yasm.yandex-team.ru/chart-alert/alerts=oplata.testing.%s;'

second = 1
minute = 60
hour = minute * 60
day = hour * 24

swat_docs_duty = doc(
    title="⚠Кого нужно пингануть",
    url="https://wiki.yandex-team.ru/balance/swat/#anchor-duty",
)

swat_docs_moderation = doc(
    title="⚠Что делать, если алерт сработал",
    url="https://wiki.yandex-team.ru/balance/swat/payments/?from=%2Ffinsrv%2Fswat%2Fpayments%2F#mod-issues",
)

swat_docs_payments = doc(
    title='Ссылки и рецепты',
    url='https://wiki.yandex-team.ru/balance/swat/payments/'
)


def _namespace(name, env):
    return 'oplata-{}-stage.{}.{}'.format(env, name, name)


def oplata_aggregator():
    return {
        'forced_aggregator': 'timed_more_than_limit_is_problem',
        'forced_aggregator_kwargs': {
            'nodata_mode': 'force_crit',
            'limits': [
                {
                    'day_start': 1, 'day_end': 5,
                    'time_start': 11, 'time_end': 17,
                    'warn': 0, 'crit': 0,
                },
                {
                    'day_start': 1, 'day_end': 5,
                    'time_start': 18, 'time_end': 10,
                    'warn': 1000, 'crit': 1000,
                },
                {
                    'day_start': 6, 'day_end': 7,
                    'time_start': 0, 'time_end': 23,
                    'warn': 1000, 'crit': 1000,
                }
            ]
        }
    }


def quorum_problem(name=None):
    """
    Использовать модуль агрегации timed_more_than_limit_is_problem
    See: https://docs.yandex-team.ru/juggler/aggregates/aggregators#timed_more_than_limit_is_problem
    """
    aggregator_kwargs = {
        'nodata_mode': 'force_warn',
        'limits': [
            {
                'day_start': 1, 'day_end': 7,
                'time_start': 0, 'time_end': 23,
                'warn': '33%', 'crit': '50%'
            }
        ],
    }

    if name:
        aggregator_kwargs['unreach_mode'] = 'skip'
        aggregator_kwargs['unreach_service'] = [{'check': ':unreachable-%s' % name}]

    return {
        'aggregator': 'timed_more_than_limit_is_problem',
        'aggregator_kwargs': aggregator_kwargs
    }


def _oplata_ephemeral_volume_checks():
    # TODO: https://st.yandex-team.ru/RTCSUPPORT-15367
    pass


def _oplata_moderation_checks(env):
    return [
        check(
            'max-pending-moderation-time-merchant',
            # alert
            yasm_deploy_threshold_alert(
                signal_name='unistat-max_pending_moderation_time_merchant_axxt',
                namespace=_namespace('workers-moderation', env),
                warn_threshold=1 * day,
                crit_threshold=1.5 * day
            ),

            # aggregator settings
            ttl(900, 300),
            oplata_aggregator(),

            # docs
            swat_docs_duty,
            swat_docs_moderation
        ),
        check(
            'max-pending-moderation-time-order',
            # alert
            yasm_deploy_threshold_alert(
                signal_name='unistat-max_pending_moderation_time_order_axxt',
                namespace=_namespace('workers-moderation', env),
                warn_threshold=1 * day,
                crit_threshold=1.5 * day
            ),

            # aggregator settings
            ttl(900, 300),
            oplata_aggregator(),

            # docs
            swat_docs_duty,
            swat_docs_moderation
        ),
        check(
            'failed-tasks-count-core-clear-unhold-order-action',
            # alert
            yasm_deploy_threshold_alert(
                signal_name='unistat-failed_tasks_count_core_clear_unhold_order_action_axxt',
                namespace=_namespace('workers-moderation', env),
                trend_type='up',
                warn_threshold=0.1,
                crit_threshold=1
            ),

            # aggregator settings
            ttl(900, 5),
            {'aggregator': 'logic_or', 'aggregator_kwargs': {'nodata_mode': 'force_crit'}},

            # docs
            swat_docs_duty,
            swat_docs_moderation
        ),
        check(
            'failed-tasks-count-start-refund',

            # alert
            yasm_deploy_threshold_alert(
                signal_name='unistat-failed_tasks_count_start_refund_axxt',
                namespace=_namespace('workers-moderation', env),
                trend_type='up',
                warn_threshold=0.1,
                crit_threshold=1
            ),

            # aggregator settings
            ttl(900, 5),
            {'aggregator': 'logic_or', 'aggregator_kwargs': {'nodata_mode': 'force_crit'}},

            # docs
            swat_docs_duty,
            swat_docs_moderation
        )
    ]


def _oplata_postgres_checks(env):
    clusters = {
        'test': '99ef16ce-6fa8-4711-bb1a-3c80cc9e5609',
        'prod': '8e1fcd92-7b69-47c5-97e2-30a072f318c5'
    }

    return [
        check(
            'pgaas-cpu-usage',

            yasm_con_threshold_alert_func(
                type='mdbdom0',
                ctype=clusters[env],
                func='div',
                signals=['portoinst-cpu_usage_cores_tmmv',
                         'portoinst-cpu_limit_cores_tmmv'],
                tier='primary',
                warn_threshold=0.8,
                crit_threshold=0.9
            ),

            # aggregator settings
            ttl(900, 5),
            {'aggregator': 'logic_or', 'aggregator_kwargs': {'nodata_mode': 'force_crit'}},

            # docs

            swat_docs_duty,
            swat_docs_payments
        ),

        check(
            'pgass-disk-usage',

            yasm_con_threshold_alert_func(
                type='mailpostgresql',
                ctype=clusters[env],
                func='div',
                signals=['push-disk-used_bytes_/var/lib/postgresql_vmmv',
                         'push-disk-total_bytes_/var/lib/postgresql_vmmv'],
                tier='primary',
                warn_threshold=0.9,
                crit_threshold=0.95
            ),

            # aggregator settings
            ttl(900, 5),
            {'aggregator': 'logic_or', 'aggregator_kwargs': {'nodata_mode': 'force_crit'}},

            # docs
            swat_docs_duty,
            swat_docs_payments
        ),

        check(
            'pgass-connections',

            yasm_con_threshold_alert_func(
                type='mailpostgresql',
                ctype=clusters[env],
                func='sum',
                signals=['push-paymentsdb_prod_conn_active_vmmv',
                         'push-paymentsdb_prod_conn_waiting_vmmv'],
                tier=['primary', 'replica'],
                warn_threshold=80,
                crit_threshold=90
            ),

            # aggregator settings
            ttl(900, 5),
            {'aggregator': 'logic_or', 'aggregator_kwargs': {'nodata_mode': 'force_crit'}},

            # docs
            swat_docs_duty,
            swat_docs_payments
        ),

        check(
            'pgass-replica-lag',

            yasm_con_threshold_alert_func(
                type='mailpostgresql',
                ctype=clusters[env],
                func='signal',
                signals='push-postgres-replication_lag_tmmx',
                tier='replica',
                warn_threshold=10,
                crit_threshold=15
            ),

            # aggregator settings
            ttl(900, 5),

            # docs
            swat_docs_duty,
            swat_docs_payments
        )
    ]


def _oplata_nginx_checks(env):
    return [
        check(
            'sdk-nginx-500',

            yasm_deploy_threshold_alert(
                'unistat-request_5xx_count_summ',
                namespace=_namespace('sdk', env),
                warn_threshold=5,
                crit_threshold=15
            ),

            # aggregator settings
            ttl(900, 5),

            # docs
            swat_docs_duty,
            swat_docs_payments
        )
    ]


def _oplata_trust_checks(env):
    return [
        check(
            'trust-errors',

            yasm_deploy_threshold_alert_func(
                func='sum',
                signals=[
                    'unistat-response_status_trust_4xx_summ',
                    'unistat-response_status_trust_5xx_summ'
                ],
                namespace=_namespace('workers-payments', env),
                warn_threshold=10,
                crit_threshold=15
            ),

            # aggregator settings
            ttl(900, 5),

            # docs
            swat_docs_duty,
            swat_docs_payments
        )
    ]


def _oplata_token_checks(env):
    return [
        check(
            'oauth-token-remain-to-expire-workers-payments',

            yasm_deploy_threshold_alert(
                'unistat-oauth_token_min_remain_to_expire_annt',
                namespace=_namespace('workers-payments', env),
                warn_threshold=3100,
                crit_threshold=3000
            ),

            # aggregator settings
            ttl(900, 5),

            # docs
            swat_docs_duty,
            swat_docs_payments
        ),

        check(
            'oauth-tokens-for-refresh-workers-payments',

            yasm_deploy_threshold_alert(
                'unistat-oauth_token_for_refresh_amount_axxt',
                namespace=_namespace('workers-payments', env),
                warn_threshold=25,
                crit_threshold=50,
            ),

            # aggregator settings
            ttl(900, 5),

            # docs
            swat_docs_duty,
            swat_docs_payments
        )
    ]


def _http_check(name, child, check_port=80, path='/ping', unreach_name=None):
    return merge(
        http(
            '%s-http-check' % name, check_port, ok_codes=200,
            headers={'Host': 'localhost'},
            path=path, timeout=5,
            warn='34%', crit='50%', no_unreach=True,
        ),
        check(
            '%s-http-check' % name,
            gen_children_deploy(child, name),
            quorum_problem(name if unreach_name is None else unreach_name)
        ),
    )


def _https_check(name, child, path='/ping'):
    return merge(
        https(
            '%s-https-check' % name, path=path, ok_codes=200, no_unreach=True,
            timeout=5
        ),

        check(
            '%s-https-check' % name,
            gen_children_deploy(child, name),
            quorum_problem(name),
            flaps(600, 3000, 0),
        ),
    )


def get_http_checks(env, children):
    checks = [
        _http_check('admin', children['admin'], path='/admin/ping'),


        _http_check('payments', children['payments']),

        _http_check('db-payments', children['payments'], path='/pingdb', unreach_name='payments'),


        _http_check('sdk', children['sdk']),

        _https_check('sdk', children['sdk']),

        # workers-fast-moderation will be moved to workers-moderation
        _http_check('workers-moderation', children['workers-moderation'], path='/check/workers-health', check_port=8000),

        _http_check('workers-payments', children['workers-payments'], path='/check/workers-health', check_port=8000),
    ]

    if env == 'prod':
        checks += [
            _http_check('documents', children['documents']),

            _http_check('db-documents', children['documents'], path='/pingdb', unreach_name='documents'),
        ]

    return checks


def get_oplata_checks(env):
    res = []

    res += _oplata_moderation_checks(env)
    res += _oplata_postgres_checks(env)
    res += _oplata_nginx_checks(env)
    res += _oplata_trust_checks(env)
    res += _oplata_token_checks(env)
    return res


def get_checks(children, balancers, env):
    args = []

    for name, child in children.items():
        args.append(
            check(
                # check name
                'unreachable-%s' % name,
                # check endpoint
                gen_children_deploy(child, 'unreachable'),
                quorum_problem(),
                # type of the active check
                {"active": "icmpping"},
                doc_link('common'),
                doc(SWAT_DUTY, 'Кого нужно пингануть'),
            )
        )

        args.append(
            check(
                'unispace-%s' % name,
                gen_children_deploy(child, 'unispace'),
                quorum_problem(name)
            )
        )

    args += get_http_checks(env, children)
    args += get_oplata_checks(env)

    args += [
        l7_monitoring(balancers),
        check(
            'certs',
            gen_children_deploy([child for child in children.values()], 'certs'),
            logic_or,
            {'aggregator_kwargs': {'nodata_mode': 'force_warn'}}
        ),
    ]

    return merge(*args)
