import juggler_sdk

import paysys.sre.tools.monitorings.configs.paysys.base as base
import paysys.sre.tools.monitorings.lib.checks.services as services
from paysys.sre.tools.monitorings.configs.trust.base import trust_mongo
from paysys.sre.tools.monitorings.lib.util.aggregators import empty_kwargs
from paysys.sre.tools.monitorings.lib.util.helpers import (
    check,
    merge,
    nodata_skip,
    skip_split_by_dc,
    solomon_check,
)
from paysys.sre.tools.monitorings.lib.util.solomon import (
    create_solomon_alert_id,
    solomon_expression_custom,
)


def mem_free(
    host, solomon_project, solomon_cluster, warn=0.6, crit=0.8, window_secs=60
):
    metric_total = '{{project="paysys",cluster="{cluster}",service="sys",path="/Memory/MemTotal","paysys-service"="trust"}}'.format(
        cluster=solomon_cluster,
    )
    metric_available = '{{project="paysys",cluster="{cluster}",service="sys",path="/Memory/MemAvailable","paysys-service"="trust"}}'.format(
        cluster=solomon_cluster,
    )
    solomon_program = (
        "let total = {total};\n"
        "let available = {available};\n"
        'let description = "No data";\n'
        "no_data_if(count(total) == 0 || size(total) == 0);\n"
        "no_data_if(count(available) == 0 || size(available) == 0);\n"
        "let usage = (total - available) / total;\n"
        "let avg_usage = avg(usage);\n"
        "let usage = 0; // Exclude from Alert chart\n"
        "let avg_usage_perc = avg_usage * 100;\n"
        'let description = "Memory usage on " + get_label(total, "host") + " is " + to_fixed(avg_usage_perc, 2) + "%";\n'
        "alarm_if(avg_usage > {crit_threshold});\n"
        "warn_if(avg_usage > {warn_threshold});\n"
    ).format(
        total=metric_total,
        available=metric_available,
        crit_threshold=crit,
        warn_threshold=warn,
    )
    return solomon_multialert_by_host(
        host, "mem-free", solomon_project, solomon_program, window_secs
    )


def cpu_usage(
    host, solomon_project, solomon_cluster, warn=0.6, crit=0.8, window_secs=60
):
    metric_tmpl = '{{project="paysys",cluster="{cluster}",service="sys",path="/System/{type}","paysys-service"="trust",cpu="-"}}'
    metric_io = metric_tmpl.format(cluster=solomon_cluster, type="IoWaitTime")
    metric_nice = metric_tmpl.format(cluster=solomon_cluster, type="NiceTime")
    metric_irq = metric_tmpl.format(cluster=solomon_cluster, type="IrqTime")
    metric_system = metric_tmpl.format(cluster=solomon_cluster, type="SystemTime")
    metric_user = metric_tmpl.format(cluster=solomon_cluster, type="UserTime")
    metric_idle = metric_tmpl.format(cluster=solomon_cluster, type="IdleTime")
    solomon_program = (
        "let used = {io} + {nice} + {irq} + {system} + {user};\n"
        "let total = {io} + {nice} + {irq} + {system} + {user} + {idle};\n"
        'let description = "No data";\n'
        "no_data_if(count(used) == 0 || size(used) == 0);\n"
        "no_data_if(count(total) == 0 || size(total) == 0);\n"
        "let usage = used / total;\n"
        "let avg_usage = avg(usage);\n"
        "let used = 0; // Exclude from Alert chart\n"
        "let total = 0; // Exclude from Alert chart\n"
        "let avg_usage_perc = avg_usage * 100;\n"
        'let description = "CPU usage on " + get_label(usage, "host") + " is " + to_fixed(avg_usage_perc, 2) + "%";\n'
        "alarm_if(avg_usage > {crit_threshold});\n"
        "warn_if(avg_usage > {warn_threshold});\n"
    ).format(
        io=metric_io,
        nice=metric_nice,
        irq=metric_irq,
        system=metric_system,
        user=metric_user,
        idle=metric_idle,
        crit_threshold=crit,
        warn_threshold=warn,
    )
    return solomon_multialert_by_host(
        host, "cpu_usage", solomon_project, solomon_program, window_secs
    )


def solomon_multialert_by_host(
    check_host, check_name, solomon_project, solomon_program, window_secs=60
):
    juggler_child = juggler_sdk.Child(
        create_solomon_alert_id(check_host, check_name),
        "all",
        "all",
        "MONITORING_MULTIALERT",
    )
    solomon_expr = solomon_expression_custom(
        project_id=solomon_project,
        program_str=solomon_program,
        annotations={"description": "{{expression.description}}"},
        group_by_labels=["host"],
        juggler_children={"replace": True, "children": [juggler_child]},
        window_secs=window_secs,
    )
    return merge(
        solomon_check(check_name, solomon_expr),
        check(check_name, empty_kwargs, nodata_skip, skip_split_by_dc),
    )


def get_checks(children, db_name, ports):
    return merge(
        base.checks_common(children),
        services.nginx,
        services.keepalived,
        services.postfix,
        services.yandex_passport_tvmtool,
        services.haproxy,
        check("trust_queue_worker"),
        check("trust_queue_export"),
        check("trust_queue_notify"),
        check("trust_queue_queue"),
        check("yandex-gideon"),
        check("yandex-gideon"),
        check("unified_agent"),
        check("unified_agent_logbroker"),
        check("solomon-agent"),
        check("solomon-agent_aggregated"),
        trust_mongo.get_mongos_checks(db_name, ports),
    )
