import juggler_sdk

from paysys.sre.tools.monitorings.configs.paysys import base
from paysys.sre.tools.monitorings.lib.checks import mongo, services
from paysys.sre.tools.monitorings.lib.util.aggregators import empty_kwargs
from paysys.sre.tools.monitorings.lib.util.helpers import (
    flaps,
    merge,
    check,
    nodata_skip,
    skip_split_by_dc,
    solomon_check,
)
from paysys.sre.tools.monitorings.lib.util.solomon import (
    create_solomon_alert_id,
    selectors_to_string,
    solomon_expression_custom,
)

mongod_instance_ports = (27101, 27102, 27103, 27219)


def spinlock_delay(
    host,
    port,
    solomon_cluster,
    solomon_project,
    warn_delay_secs,
    crit_delay_secs,
    window_secs=180,
):
    name = "spinlock_delay"

    metric_total_delay = selectors_to_string(
        {
            "project": "trust",
            "service": "mongo",
            "source": "mongo",
            "cluster": solomon_cluster,
            "sensor": "one_min.{}.tcmalloc.tcmalloc.spinlock_total_delay_ns".format(
                port
            ),
        }
    )
    solomon_program = (
        "let total_delay_ns = ({total_delay_ns});\n"
        'let description = "No data";\n'
        "no_data_if(size(total_delay_ns) == 0 || count(total_delay_ns) == 0);\n"
        "let diff_delay_ns = diff(total_delay_ns);\n"
        "let last_diff_delay_ns = last(diff_delay_ns);\n"
        "let last_diff_delay_secs = last_diff_delay_ns / 1G;\n"
        'let description = "Spinlock delay - " + to_fixed(last_diff_delay_secs, 2) + " seconds"'
        ' + " (warn - " + {threshold_warn} + ", crit - " + {threshold_crit} + ")"\n;'
        "alarm_if(last_diff_delay_secs > {threshold_crit});\n"
        "warn_if(last_diff_delay_secs > {threshold_warn});\n"
    ).format(
        total_delay_ns=metric_total_delay,
        threshold_crit=crit_delay_secs,
        threshold_warn=warn_delay_secs,
    )

    juggler_child = juggler_sdk.Child(
        create_solomon_alert_id(host, name),
        "all",
        "all",
        "MONITORING_MULTIALERT",
    )
    solomon_expr = solomon_expression_custom(
        project_id=solomon_project,
        program_str=solomon_program,
        annotations={"description": "{{expression.description}}"},
        group_by_labels=["host"],
        juggler_children={"replace": True, "children": [juggler_child]},
        window_secs=window_secs,
    )
    return merge(
        solomon_check(name, solomon_expr),
        check(name, empty_kwargs, nodata_skip, skip_split_by_dc),
    )


def get_checks(
    children,
    cluster,
    shards,
    split_by_dc=None,
    host=None,
    warn=None,
    crit=None,
    warn_ratio=None,
    crit_ratio=None,
    warn_oplog=15,
    crit_oplog=10,
):
    return merge(
        base.checks_common(children),
        services.keepalived,
        mongo.mongo_with_single_shardgroup(
            children, cluster, shards, host, split_by_dc
        ),
        mongo.mongod_scanned_objects(children, mongod_instance_ports, warn, crit),
        mongo.mongod_scanned_objects(
            children, mongod_instance_ports, warn_ratio, crit_ratio, ratio=True
        ),
        mongo.mongod_oplog_window(
            children, mongod_instance_ports, warn=warn_oplog, crit=crit_oplog
        ),
    )


def get_mongos_checks(db_name, ports):
    checks = []
    for monrun_check in ["balancer_state", "connect", "statistics"]:
        for port in ports:
            args = []
            if monrun_check == "statistics":
                args.append(flaps(180, 300))  # TODO: remove after fix core problem
            name = "mongo_{db_name}-shard-mongos-{port}_{monrun_check}".format(
                db_name=db_name, port=port, monrun_check=monrun_check
            )
            checks.append(check(name, *args))
    return merge(*checks)
