from paysys.sre.tools.monitorings.lib.util import aggregators
from paysys.sre.tools.monitorings.lib.util.helpers import merge, unreach_skip, gen_unreach, flaps


def http(
    name,
    port=80,
    path='/ping',
    ok_codes=None,
    headers=None,
    timeout=1,
    win_threshold=1,
    fail_threshold=2,
    crit=2,
    warn=1,
    limits=None,
    no_unreach=False,
):
    if not limits:
        limits = [
            {
                "day_start": 1,
                "day_end": 7,
                "time_start": 0,
                "time_end": 23,
                "warn": warn,
                "crit": crit,
            }
        ]
    ok_codes = ok_codes if ok_codes else ['200']
    headers = headers if headers else {}
    return {
        name: merge(
            {
                'active': 'http',
                'active_kwargs': {
                    'path': path,
                    'port': port,
                    'ok_codes': ok_codes,
                    'timeout': timeout,
                    'headers': headers,
                    'win_threshold': win_threshold,
                    'fail_threshold': fail_threshold,
                },
            },
            aggregators.timed_more_than_limit_is_problem(
                limits=limits,
            ),
            {} if no_unreach else unreach_skip
        )
    }


def https(
    name,
    port=443,
    path='/ping',
    ok_codes=None,
    headers=None,
    timeout=1,
    win_threshold=1,
    fail_threshold=2,
    crit=2,
    warn=1,
    limits=None,
    no_unreach=False,
    aggregator=None,
    warn_expire=None,
    crit_expire=None,
):
    if not limits:
        limits = [
            {
                "day_start": 1,
                "day_end": 7,
                "time_start": 0,
                "time_end": 23,
                "warn": warn,
                "crit": crit,
            }
        ]
    if aggregator is None:
        aggregator = aggregators.timed_more_than_limit_is_problem(limits=limits)

    ok_codes = ok_codes if ok_codes else ['200']
    headers = headers if headers else {}
    active_kwargs = {
        'path': path,
        'port': port,
        'ok_codes': ok_codes,
        'timeout': timeout,
        'headers': headers,
        'win_threshold': win_threshold,
        'fail_threshold': fail_threshold,
    }
    if warn_expire is not None:
        active_kwargs['warn_expire'] = warn_expire
    if crit_expire is not None:
        active_kwargs['crit_expire'] = crit_expire
    return {
        name: merge(
            {
                'active': 'https',
                'active_kwargs': active_kwargs,
            },
            aggregator,
            {} if no_unreach else unreach_skip
        )
    }


def https_unreach(name, port=443, headers=None):
    headers = headers if headers else {}
    return merge(
        https(name, port, headers=headers),
        {name: gen_unreach(['UNREACHABLE', 'nginx-alive'])},
    )


def https_cert(name, port=443, warn=21, crit=21, host=None, ssl_host=None):
    https_cert_result = {name: merge({'active': 'https_cert',
                                      'active_kwargs': {'port': port,
                                                        'crit_expire': crit,
                                                        'warn_expire': warn,
                                                        'host': host,
                                                        'ssl_hostname': ssl_host
                                                        }
                                      }, unreach_skip)
                         }

    if host is None:
        https_cert_result[name]['active_kwargs'].pop('host')
    if ssl_host is None:
        https_cert_result[name]['active_kwargs'].pop('ssl_hostname')

    return https_cert_result


def https_bundle(
    name,
    port=443,
    path='/ping',
    ok_codes=None,
    headers=None,
    timeout=1,
    cert_warn=21,
    cert_crit=21,
    unreach=['UNREACHABLE', 'nginx-alive'],
    win_threshold=1,
    fail_threshold=2,
    crit=2,
    warn=1,
    crit_workhours=0,
    warn_workhours=0,
    https_crit_expire=None,
    https_warn_expire=None,
):
    """
    Bundle for generating all checks for https service
    """

    # Usual https check (CRIT if more than 2 servants are down)
    https_check_name = 'https_' + name
    https_check = merge(
        https(
            name=https_check_name,
            port=port,
            path=path,
            ok_codes=ok_codes,
            headers=headers,
            timeout=timeout,
            win_threshold=win_threshold,
            fail_threshold=fail_threshold,
            crit=crit,
            warn=warn,
            crit_expire=https_crit_expire,
            warn_expire=https_warn_expire,
        ),
        {https_check_name: gen_unreach(unreach)},
        {https_check_name: {'tags': ['https_main']}},
    )

    # The same https check is above but with flap. Will transform WARN to
    # CRIT after one hour at working hours
    https_workhours_check_name = 'https_' + name + '_workhours'
    https_workhours_check = merge(
        merge(
            https(
                name=https_workhours_check_name,
                port=port,
                path=path,
                ok_codes=ok_codes,
                headers=headers,
                timeout=timeout,
                win_threshold=win_threshold,
                fail_threshold=fail_threshold,
                limits=[
                    {
                        "day_start": 1,
                        "day_end": 5,
                        "time_start": 10,
                        "time_end": 20,
                        "warn": warn_workhours,
                        "crit": crit_workhours,
                    },
                    {
                        "day_start": 6,
                        "day_end": 7,
                        "time_start": 0,
                        "time_end": 23,
                        "warn": "100%",
                        "crit": "100%",
                    },
                    {
                        "day_start": 1,
                        "day_end": 5,
                        "time_start": 20,
                        "time_end": 10,
                        "warn": "100%",
                        "crit": "100%",
                    },
                ],
                crit_expire=https_crit_expire,
                warn_expire=https_warn_expire,
            ),
            {https_workhours_check_name: flaps(180, 1500, 0)},
            {https_workhours_check_name: gen_unreach(unreach)},
            {https_workhours_check_name: {'tags': ['https_workhours']}},
        )
    )

    # Also check https certificate expiration
    https_cert_check_name = 'https_' + name + '_cert'
    https_cert_check = merge(
        https_cert(
            name=https_cert_check_name,
            port=port,
            warn=cert_warn,
            crit=cert_crit
        ),
        {
            https_cert_check_name: aggregators.timed_more_than_limit_is_problem(
                limits=[
                    {
                        "day_start": 1,
                        "day_end": 5,
                        "time_start": 10,
                        "time_end": 20,
                        "warn": warn,
                        "crit": crit,
                    },
                    {
                        "day_start": 6,
                        "day_end": 7,
                        "time_start": 0,
                        "time_end": 23,
                        "warn": "100%",
                        "crit": "100%",
                    },
                    {
                        "day_start": 1,
                        "day_end": 5,
                        "time_start": 20,
                        "time_end": 10,
                        "warn": "100%",
                        "crit": "100%",
                    },
                ]
            )
        },
        {https_cert_check_name: flaps(180, 1500, 0)},
        {https_cert_check_name: gen_unreach(unreach)},
        {https_cert_check_name: {'tags': ['https_cert']}},
    )

    return merge(
        https_check,
        https_workhours_check,
        https_cert_check,
    )


def http_bundle(
    name,
    port=443,
    path='/ping',
    ok_codes=None,
    headers=None,
    timeout=1,
    cert_warn=21,
    cert_crit=21,
    unreach=['UNREACHABLE', 'nginx-alive'],
    win_threshold=1,
    fail_threshold=2,
    crit=2,
    warn=1,
    crit_workhours=0,
    warn_workhours=0,
):
    """
    Bundle for generating all checks for http service
    """

    # Usual http check (CRIT if more than 2 servants are down)
    http_check_name = 'http_' + name
    http_check = merge(
        http(
            name=http_check_name,
            port=port,
            path=path,
            ok_codes=ok_codes,
            headers=headers,
            timeout=timeout,
            win_threshold=win_threshold,
            fail_threshold=fail_threshold,
            crit=crit,
            warn=warn,
        ),
        {http_check_name: gen_unreach(unreach)},
        {http_check_name: {'tags': ['http_main']}},
    )

    # The same http check is above but with flap. Will transform WARN to
    # CRIT after one hour at working hours
    http_workhours_check_name = 'http_' + name + '_workhours'
    http_workhours_check = merge(
        merge(
            http(
                name=http_workhours_check_name,
                port=port,
                path=path,
                ok_codes=ok_codes,
                headers=headers,
                timeout=timeout,
                win_threshold=win_threshold,
                fail_threshold=fail_threshold,
                limits=[
                    {
                        "day_start": 1,
                        "day_end": 5,
                        "time_start": 10,
                        "time_end": 20,
                        "warn": warn_workhours,
                        "crit": crit_workhours,
                    },
                    {
                        "day_start": 6,
                        "day_end": 7,
                        "time_start": 0,
                        "time_end": 23,
                        "warn": "100%",
                        "crit": "100%",
                    },
                    {
                        "day_start": 1,
                        "day_end": 5,
                        "time_start": 20,
                        "time_end": 10,
                        "warn": "100%",
                        "crit": "100%",
                    },
                ]
            ),
            {http_workhours_check_name: flaps(180, 1500, 0)},
            {http_workhours_check_name: gen_unreach(unreach)},
            {http_workhours_check_name: {'tags': ['http_workhours']}},
        )
    )

    return merge(
        http_check,
        http_workhours_check,
    )
