from collections import namedtuple

import yasm_alert
from paysys.sre.tools.monitorings.lib.checks.doc import doc
from paysys.sre.tools.monitorings.lib.util.helpers import (
    merge,
    flaps,
    gen_children_nanny,
    gen_children,
    gen_unreach,
    check,
    create_subchecks,
)
from paysys.sre.tools.monitorings.lib.checks.active import http
from paysys.sre.tools.monitorings.lib.checks import base


def _create_balancer(l7_balancer_data):
    if "http_checks" not in l7_balancer_data:
        l7_balancer_data["http_checks"] = {}
    if "instance_checks" not in l7_balancer_data:
        l7_balancer_data["instance_checks"] = {}

    l7_balancer = namedtuple("l7balancer", l7_balancer_data.keys())
    l7_balancer = l7_balancer(**l7_balancer_data)

    if not all([l7_balancer.namespace, l7_balancer.datacenters, l7_balancer.checks]):
        raise ValueError("Check config balancer")

    if not l7_balancer.services:
        raise ValueError("Check config balancer")

    if not any([l7_balancer.https_ports, l7_balancer.http_ports]):
        raise ValueError("http_ports or https_ports is absent in balancer config")
    if not l7_balancer.host:
        raise ValueError("Check config balancer")

    return l7_balancer


def l7_monitoring(l7_balancers, split_unreachable=False):
    """
    Generate balancer monitorings.
    Monitorings:
        * L7 health-check balancer monitorings.
        * L7 custom YASM monitorings.
        * L3 health-check balancer monitorings.

    :param l7_balancers: list of dicts, that declare balancer checks.
        Fields:
            namespace:  namespace for balancer checks in Juggler.
            host:       balancer host
            datacenters: balancer DC locations. Alerts will be generated per DC.
            http_ports: http ports for checks.
            https_ports: https ports for checks.
            services: balancer instances.
            checks: YASM checks for balancers. Each check is defiend by its corresponding to YAML check postfix.
                For example `cpu_wait` will generate check `<host>_cpu_wait_<dc>`.
                Check fields:
                    warn: warning threshold
                    crit: critical threshold
                    flaps: settings for flap control
            instance_checks: checks  "check_balancer_zombie_procs", "check_enddate_certificate", "reload_config_status"
    :param split_unreachable: generate UNREACHABLE check for each instance_check.
    """
    if isinstance(l7_balancers, dict):
        l7_balancers = [l7_balancers]

    _l7_yasm_checks = []
    _l7_http_checks = []
    _l3_http_checks = []
    _instance_checks = []
    for l7_balancer_data in l7_balancers:
        l7_balancer = _create_balancer(l7_balancer_data)
        _l7_yasm_checks += _gen_yasm_checks(
            l7_balancer.namespace, l7_balancer.datacenters, l7_balancer.checks
        )

        _instance_checks += _gen_instance_checks(
            l7_balancer.namespace,
            l7_balancer.services,
            split_unreachable=split_unreachable,
            aggregator=l7_balancer.instance_checks.get("aggregator"),
        )

        _l7_http_checks += _gen_l7_http_checks(
            l7_balancer.namespace,
            l7_balancer.services,
            l7_balancer.http_ports,
            l7_balancer.https_ports,
            l7_balancer.host,
            aggregator=l7_balancer.http_checks.get("aggregator"),
        )

        _l3_http_checks += _gen_l3_http_checks(
            l7_balancer.namespace,
            l7_balancer.host,
            l7_balancer.http_ports,
            l7_balancer.https_ports,
        )

    return merge(
        *_instance_checks + _l7_yasm_checks + _l7_http_checks + _l3_http_checks
    )


def __get_signal_by_alias(alert):
    # Find in alert: prefix before first _
    prefix_len = alert.find("_")
    prefix = alert[:prefix_len]
    metric = alert[prefix_len + 1 :]

    boards = [
        "balancer_portoinst_panel",
        "balancer_aux_panel",
        "balancer_common_panel",
    ]

    all_signals = {
        "balancer_portoinst_panel": {
            "cpu": {
                "guarantee": "portoinst-portoinst-cpu_guarantee_slot_cores_tmmv",
                "limit": "portoinst-cpu_limit_slot_hgram",
                "throttled": "portoinst-cpu_throttled_cores_tmmv",
                "usage": "quant(portoinst-cpu_limit_usage_perc_hgram, 80)",
                "system_usage": "portoinst-cpu_usage_system_cores_tmmv",
                "wait": "portoinst-cpu_wait_cores_tmmv",
            },
            "mem": {
                "guarantee": "portoinst-memory_guarantee_slot_hgram",
                "limit": "portoinst-memory_limit_slot_hgram",
                "usage": "portoinst-memory_usage_gb_tmmv",
                "anon": "portoinst-anon_usage_gb_tmmv",
                "max_rss": "portoinst-max_rss_gb_tmmv",
            },
            "space": {
                "logs": "portoinst-volume_/logs_usage_perc_txxx",
                "workdir": "portoinst-volume_cwd_usage_perc_txxx",
                "rootfs": "portoinst-volume_root_usage_perc_txxx",
            },
            "page_faults": {
                "major": "portoinst-major_page_faults_summ",
                "minor": "portoinst-minor_page_faults_summ",
            },
            "disk_io": {
                "write": "portoinst-io_write_fs_bytes_tmmv",
                "read": "portoinst-io_read_fs_bytes_tmmv",
            },
            "net_io": {
                "tx": "portoinst-net_mb_summ",
                "rx": "portoinst-net_rx_mb_summ",
                "txpkts": "portoinst-net_packets_summ",
                "rxpkts": "portoinst-net_rx_packets_summ",
                "txdrops": "portoinst-net_drops_summ",
                "rxdrops": "portoinst-net_rx_drops_summ",
            },
        },
        "balancer_aux_panel": {
            "tls": {
                "no_shared_cipher": "balancer_report-ssl_error-ssl3_get_client_hello_no_shared_cipher_summ",
                "unexpected_message": "balancer_report-ssl_error-ssl3_read_bytes_sslv3_alert_unexpected_message_summ",
                "inappropriate_fallback": "balancer_report-ssl_error-ssl_bytes_to_cipher_list_inappropriate_fallback_summ",
                "bad_certificate": "balancer_report-ssl_error-ssl3_read_bytes_sslv3_alert_bad_certificate_summ",
                "unknown_ca": "balancer_report-ssl_error-ssl3_read_bytes_tlsv1_alert_unknown_ca_summ",
                "bad_oscp_response": "balancer_report-ssl_error-ssl3_read_bytes_tlsv1_bad_certificate_status_response_summ",
                "tcp_problems_during_tls_handshake": "balancer_report-ssl_error-zero_code_summ",
                "ssl_routines_total": "balancer_report-ssl_error-total_summ",
            }
        },
        "balancer_common_panel": {
            "codes": {
                "5xx": """perc(
                                  balancer_report-report-service_total-outgoing_5xx_summ,
                              max(
                                  balancer_report-report-service_total-requests_summ,
                                  const(100)
                              )
                          )""",
            },
            "attempts": {
                "backend_errors": "balancer_report-report-service_total-backend_error_summ"
            },
            "timings": {
                "q99": "quant(balancer_report-report-service_total-processing_time_hgram, 99)",
                "q95": "quant(balancer_report-report-service_total-processing_time_hgram, 95)",
                "q50": "quant(balancer_report-report-service_total-processing_time_hgram, 50)",
            }
        },
    }

    for board in boards:
        if prefix in all_signals[board]:
            signal = all_signals[board][prefix][metric]
            break
    else:
        raise ValueError("No such metric {} in all_signals".format(metric))

    return signal, board


def _gen_instance_checks(namespace, services, split_unreachable=False, aggregator=None):
    instance_checks = {
        "zombie_procs": "check_balancer_zombie_procs",
        "cert_enddate": "check_enddate_certificate",
        "reload_config": "reload_config_status",
    }

    unreachable_check = merge(
        base.unreachable,
        check("UNREACHABLE", gen_children_nanny(services, "UNREACHABLE")),
        {"UNREACHABLE": aggregator} if aggregator else {},
    )
    _instance_checks = [
        create_subchecks(namespace, namespace, unreachable_check),
    ]

    for k, v in instance_checks.items():
        check_name = "{0}_{1}".format(namespace, k)
        check_params = [
            check(check_name, gen_children_nanny(services, v))
        ]

        if split_unreachable:
            check_params.append({
                check_name: gen_unreach(["{}:UNREACHABLE".format(namespace)]),
            })
        else:
            check_params += [
                base.unreachable,
                check("UNREACHABLE", gen_children_nanny(services, "UNREACHABLE")),
            ]

        if aggregator:
            check_params.append({check_name: aggregator})

        _instance_checks.append(merge(*check_params))

    return _instance_checks


def _gen_yasm_checks(namespace, datacenters, checks):
    _l7_yasm_checks = []

    for check_name, limits in checks.items():
        signal, dashboard = __get_signal_by_alias(check_name)
        warn = checks[check_name].get("warn")
        crit = checks[check_name].get("crit")
        stable = checks[check_name].get("flaps", {}).get("stable", 20)
        critical = checks[check_name].get("flaps", {}).get("critical", 100)

        if not any([warn, crit]):
            raise ValueError("no warn or crit in check")

        for datacenter in datacenters:
            _l7_yasm_checks.append(
                _gen_yasm_check(namespace, datacenter, dashboard, signal, check_name, warn, crit, stable, critical)
            )

    return _l7_yasm_checks


def _gen_yasm_check(namespace, datacenter, dashboard, signal, check_name, warn, crit, stable, critical):
    if datacenter in ["myt", "iva"]:
        datacenter = "msk"
    yasm_check = {
        "{}_{}_{}".format(namespace, check_name, datacenter): merge(
            {
                "yasm": {
                    "signal": ''.join(signal.split()),
                    "tags": yasm_alert.Tags(
                        itype=["balancer"],
                        ctype=["prod"],
                        geo=[datacenter],
                        prj=[namespace],
                    ),
                    "warn": [warn, crit],
                    "crit": [crit, None],
                    "mgroups": ["ASEARCH"],
                },
                "tags": ["balancer"],
            },
            flaps(stable, critical),
            merge(
                doc(
                    "https://yasm.yandex-team.ru/template/panel/{2}/fqdn={0};itype=balancer;ctype=prod;locations={1};prj={0}".format(
                        namespace, datacenter, dashboard
                    ),
                    title="Dashboard",
                ),
                doc(
                    "https://wiki.yandex-team.ru/cplb/awacs/monitoring/",
                    title="Monitoring Wiki",
                ),
            ),
        )
    }
    return yasm_check


def _gen_l7_http_checks(namespace, services, http_ports, https_ports, host, ok_codes=None, ping_url="/ping",
                        aggregator=None):
    headers = {"Host": host} if host else {}
    port_checks = {"l7_awacs_ping": "/awacs-balancer-health-check", "l7_ping": ping_url}
    _l7_http_checks = {namespace: {}}

    def __prepare_http(prefix, namespace, services, port, k):
        return check(
            "{0}_l7_{1}_{2}_{3}".format(namespace, prefix, port, k),
            gen_children_nanny(
                services, "{0}_l7_http_{1}_{2}".format(namespace, port, k)
            ),
        )

    for port in http_ports:
        for k, v in port_checks.items():
            _l7_http_checks[namespace].update(
                merge(
                    http.http(
                        "{0}_l7_http_{1}_{2}".format(namespace, port, k),
                        port,
                        v,
                        ok_codes=ok_codes,
                        headers=headers,
                        crit=0,
                    ),
                    __prepare_http("http", namespace, services, port, k),
                )
            )

    for port in https_ports:
        for k, path in port_checks.items():
            _l7_http_checks[namespace].update(
                merge(
                    http.https(
                        "{0}_l7_https_{1}_{2}".format(namespace, port, k),
                        port,
                        path,
                        ok_codes=ok_codes,
                        headers=headers,
                        crit=0,
                        aggregator=aggregator,
                    ),
                    __prepare_http("https", namespace, services, port, k),
                )
            )

    l7_sub_checks = [
        create_subchecks(k, namespace, v) for k, v in _l7_http_checks.items()
    ]
    return l7_sub_checks


def _gen_l3_http_checks(
    namespace, host, http_ports, https_ports, ok_codes=None, ping_url="/ping"
):
    headers = {"Host": host} if host else {}
    port_checks = {"awacs_ping": "/awacs-balancer-health-check", "ping": ping_url}
    _l3_http_checks = {namespace: {}}

    def __prepare_http(prefix, port, uri):
        return check(
            "{0}_l3_{1}_{2}_{3}".format(namespace, port, prefix, uri),
            gen_children([], []),
        )

    for port in http_ports:
        for k, v in port_checks.items():
            _l3_http_checks[namespace].update(
                merge(
                    http.http(
                        "{0}_l3_{1}_http_{2}".format(namespace, port, k),
                        port,
                        v,
                        ok_codes=ok_codes,
                        headers=headers,
                        crit=0,
                        no_unreach=True,
                    ),
                    __prepare_http("http", port, k),
                )
            )

    for port in https_ports:
        for k, v in port_checks.items():
            _l3_http_checks[namespace].update(
                merge(
                    http.https(
                        "{0}_l3_{1}_https_{2}".format(namespace, port, k),
                        port,
                        v,
                        ok_codes=ok_codes,
                        headers=headers,
                        crit=0,
                        no_unreach=True,
                    ),
                    __prepare_http("https", port, k),
                )
            )

    l3_sub_checks = [
        create_subchecks(k, namespace, v) for k, v in _l3_http_checks.items()
    ]

    return l3_sub_checks
