from paysys.sre.tools.monitorings.lib.checks.graphite import graphite_check
from paysys.sre.tools.monitorings.lib.util.aggregators import empty_kwargs
from paysys.sre.tools.monitorings.lib.util.helpers import (
    check,
    flaps,
    gen_children_from_tuples,
    merge,
    ttl,
    unreach_skip,
)
from paysys.sre.tools.monitorings.lib.checks.doc import doc, doc_link
from paysys.sre.tools.monitorings.lib.checks.linkeye import link, uplink
from paysys.sre.tools.monitorings.lib.util.conductor import get_fqdn_from_group

host = {
    "bmc": doc_link("common"),
    "check-certs": merge(ttl(7200, 3600), doc_link("common")),
    "clock": doc_link("common"),
    "cpu": doc_link("common"),
    "daytime": doc_link("common"),
    "disk": doc_link("common"),
    "ecc": doc_link("common"),
    "gpu": doc_link("common"),
    "hwerrors2": doc_link("common"),
    "link": doc_link("common"),
    "link-speed": doc_link("common"),
    "mem": doc_link("common"),
    "mount": merge(ttl(1240, 300), doc_link("common")),
    "ntp-stratum": merge(flaps(900, 1500), doc_link("common")),
    "raid": doc_link("common"),
    "raid_check_speed": {},
    "unispace": merge(flaps(300, 900), doc_link("common")),
    "watchdog": doc_link("common"),
}

services = {
    "named-alive": doc_link("common"),
    "salt-minion": merge(flaps(300, 14400, 0), doc_link("common")),
    "skynet": merge(
        flaps(300, 900),
        doc("https://wiki.yandex-team.ru/dljaadminov/paysys/ps/skynet/#monitoring"),
    ),
    "paysys_archiver": merge(
        flaps(300, 86000),
        doc("https://wiki.yandex-team.ru/dljaadminov/paysys/ps/archiver/#monitoring"),
    ),
    "push-client": merge(flaps(900, 900), ttl(900, 300), doc_link("common")),
}
graphite_client = check(
    "graphite-client", merge(flaps(600, 3600, 0), doc_link("common"))
)
paysys_common_settings = check(
    "paysys-common-settings", merge(ttl(620, 310), flaps(900, 1800, 0))
)
host_memory_usage = check("host_memory_usage", ttl(900, 300))
unbound = check("unbound", ttl(620, 300))
unreachable = check(
    "UNREACHABLE", {"active": "icmpping"}, ttl(600, 60), doc_link("common")
)
iptunnels = check("ip-tunnels", merge(ttl(700, 300), doc_link("common")))
pkgver = check("pkgver", ttl(1500, 300), doc_link("common"))
kvm = merge(
    check("kvm-virt-autostart", ttl(7200, 3600), doc_link("kvm")),
    check("kvm-virt-resources", ttl(7200, 3600), doc_link("kvm")),
)

hbf = {
    "hbf_service_last_update_status": merge(
        ttl(refresh_time=300), flaps(300, 900), doc_link("common")
    ),
    "hbf_service_status": merge(
        ttl(refresh_time=300), flaps(300, 900), doc_link("common")
    ),
}

load_average_paysys = check(
    "load_average_paysys", merge(ttl(620, 300), doc_link("common"))
)

iptruler = {
    "iptruler": merge(
        ttl(900, 180),
        unreach_skip,
        doc_link("common"),
    )
}

sysctl = check("sysctl", merge(ttl(900, 600), doc_link("common")))
check_configs = check("configs", merge(ttl(900, 600), doc_link("common")))

grub = check("grub", doc_link("common"))
netconsole = check("netconsole", merge(ttl(600, 60), doc_link("common")))

unispace5 = check("unispace5", merge(flaps(300, 900), ttl(620, 300), doc_link("common")))

syslog_alive = check("syslog-alive", doc("https://wiki.yandex-team.ru/dljaadminov/paysys/ps/syslog-ng/"))

iptables = check("iptables", ttl(180, 60))


def common_fin_fin(children):
    checks = merge(
        check_configs,
        graphite_client,
        grub,
        hbf,
        host,
        host_memory_usage,
        link,
        load_average_paysys,
        netconsole,
        paysys_common_settings,
        sysctl,
        unbound,
        unreachable,
        uplink,
    )
    checks.pop("check-certs")
    return checks


def common(children):
    return merge(
        host,
        services,
        unreachable,
        iptruler,
        graphite_client,
        hbf,
        mem_free(children),
        link,
        uplink,
    )


def mem_free(children, warn=0.8, crit=0.8):
    subchecks = {}
    hosts = get_fqdn_from_group(children)
    template = (
        "divideSeries("
        "diffSeries("
        "one_min.{0}.meminfo.used,"
        "one_min.{0}.meminfo.cached,"
        "one_min.{0}.meminfo.buffers"
        "), one_min.{0}.meminfo.total)"
    )

    for _host in hosts:
        subchecks[_host] = merge(
            graphite_check(
                "mem-free",
                template.format(_host.replace(".", "_")),
                crit,
                warn,
                "-20min",
            ),
            check("mem-free", flaps(120, 300), {"children": []}),
        )

    return check(
        "mem-free",
        flaps(120, 300),
        {"tags": ["subchecks"]},
        {"subchecks": subchecks},
        gen_children_from_tuples([x, "mem-free", "HOST"] for x in hosts),
        empty_kwargs,
    )


def _disk_template_check(children, name, metric_template, warn=50, crit=60):
    subchecks = {}
    hosts = get_fqdn_from_group(children)
    for _host in hosts:
        subchecks[_host] = merge(
            graphite_check(
                name,
                metric_template.format(_host.replace(".", "_")),
                crit,
                warn,
                "-20min",
            ),
            check(name, flaps(120, 300), {"children": []}),
        )
    return check(
        name,
        flaps(120, 300),
        {"tags": ["subchecks"]},
        {"subchecks": subchecks},
        gen_children_from_tuples([host, name, "HOST"] for host in hosts),
        empty_kwargs,
    )


def disk_util(children, warn=50, crit=60):
    name = "disk-utilization"
    template = "maxSeries(one_min.{0}.diskstat_util_*.util)"
    return _disk_template_check(children, name, template, warn, crit)


def disk_wait(children, warn=500, crit=1000):
    name = "disk-wait"
    template = "averageSeries(one_min.{0}.diskstat_wait_*.wait)"
    return _disk_template_check(children, name, template, warn, crit)


def disk_hdd(children, wait_warn=500, wait_crit=1000, util_warn=50, util_crit=60):
    return disk(children, wait_warn, wait_crit, util_warn, util_crit)


def disk_ssd(children, wait_warn=20, wait_crit=40, util_warn=20, util_crit=40):
    return disk(children, wait_warn, wait_crit, util_warn, util_crit)


def disk(children, wait_warn, wait_crit, util_warn, util_crit):
    return merge(
        disk_util(children, warn=util_warn, crit=util_crit,),
        disk_wait(children, warn=wait_warn, crit=wait_crit,),
    )


def cpu_usage(children, warn=60, crit=80):
    name = "cpu_usage"
    total_metric_template = "sumSeries({0})"
    metric_template = "one_min.{0}.cpuload.{1}"

    subchecks = {}
    hosts = get_fqdn_from_group(children)
    for _host in hosts:
        metrics = [
            metric_template.format(_host.replace(".", "_"), metric)
            for metric in ["io", "ni", "si", "sy", "us"]
        ]
        subchecks[_host] = merge(
            graphite_check(
                name,
                total_metric_template.format(",".join(metrics)),
                crit,
                warn,
                "-20min",
            ),
            check(name, flaps(180, 900), {"children": []}),
        )
    return check(
        name,
        flaps(180, 900),
        {"tags": ["subchecks"]},
        {"subchecks": subchecks},
        gen_children_from_tuples([host, name, "HOST"] for host in hosts),
        empty_kwargs,
    )


BYTES_IN_GIGABIT = 1073741824/8


def link_utilization_graphite(children, warn=0.8, crit=0.8, max=BYTES_IN_GIGABIT):
    subchecks = {}
    hosts = get_fqdn_from_group(children)
    template = "one_min.{0}.netstat_bytes_eth0.{1}"

    for _host in hosts:
        subchecks[_host] = merge(
            graphite_check(
                "link_utilization_rx",
                template.format(_host.replace(".", "_"), "rx"),
                crit * max,
                warn * max,
                "-20min",
            ),
            graphite_check(
                "link_utilization_tx",
                template.format(_host.replace(".", "_"), "tx"),
                crit * max,
                warn * max,
                "-20min",
            ),
            check("link_utilization_tx", flaps(120, 300), {"children": []}),
            check("link_utilization_rx", flaps(120, 300), {"children": []}),
        )

    return check(
        "link_utilization_graphite",
        flaps(120, 300),
        {"tags": ["subchecks"]},
        {"subchecks": subchecks},
        gen_children_from_tuples(
            [[x, "link_utilization_rx", "HOST"] for x in hosts] +
            [[x, "link_utilization_tx", "HOST"] for x in hosts]
        ),
        empty_kwargs,
    )
