import requests
from paysys.sre.tools.monitorings.lib.util.helpers import merge


def __graphite_expand_wildcard(
    metric,
    host='https://ps-mg.yandex-team.ru',
    url="/metrics/find"
):
    tokens = metric.split('*')
    data = {'query': tokens[0] + '*'}
    result = requests.get(host + url, data).json()
    if len(tokens) > 2:
        raise LookupError("Found more than one wildcard, aborted")

    return {item['text']: item['id'] + tokens[1] if len(tokens) > 1 else item['id'] for item in result}


def graphite_check_wildcard(
    name,
    metric,
    request,
    crit,
    warn,
    window,
    timeout=45,
    url='https://ps-mg.yandex-team.ru',
    less=False
):
    metrics = __graphite_expand_wildcard(metric, url)
    return merge(
        *[
            graphite_check(
                name + "_" + token,
                request.format(token_metric),
                crit,
                warn,
                window,
                timeout,
                url,
                less
            ) for token, token_metric in metrics.items()
        ]
    )


def _process_graphite_arg(expr, limit, less=True):
    result = ''
    if expr:
        result = expr
    elif limit is not None:
        result = 'metric {1} {0}'.format(limit, '<' if less else '>')
    return result


def graphite_check(
    name,
    metric,
    crit=None,
    warn=None,
    window='-20min',
    timeout=45,
    url='https://ps-mg.yandex-team.ru',
    less=False,
    crit_expr=None,
    warn_expr=None,
    null_status='WARN',
    refresh_time=60
):
    graph_url = "{}/render/?target={}&target=constantLine({})&target=constantLine({})&width=1024&height=512&from=-3h".format(
        url,
        metric,
        warn,
        crit,
    )

    return {
        name: {
            'active': 'graphite',
            'active_kwargs': {
                'CRIT': _process_graphite_arg(crit_expr, crit, less),
                'WARN': _process_graphite_arg(warn_expr, warn, less),
                'base_url': url,
                'metric': metric,
                'time_window': window,
                'timeout': timeout,
                'null_status': null_status,
                'ignore_errors': True,
            },
            'aggregator_kwargs': {
                'nodata_mode': 'skip'
            },
            'children': [],
            'refresh_time': refresh_time,
            'meta': {
                'urls': [
                    {
                        'title': "Graph",
                        'url': graph_url,
                        'type': 'screenshot_url'
                    }
                ],
            },
        }
    }


# https://graphite.readthedocs.io/en/latest/functions.html#graphite.render.functions.smartSummarize
def smart_summarize(metric, window, aggr='sum', allign='min'):
    return "smartSummarize({0}, '{1}', '{2}', '{3}')".format(
        metric,
        window,
        aggr,
        allign
    )


def summarize(metric, window, aggr='sum'):
    return "summarize({0}, '{1}', '{2}', true)".format(metric, window, aggr)


def movingAvg(metric, window):
    return summarize(metric, window, aggr='avg')


def movingMin(metric, window):
    return summarize(metric, window, aggr='min')


def movingMax(metric, window):
    return summarize(metric, window, aggr='max')


def graphite_crit_on_zero(name, metric, eps=0.3):
    return graphite_check(name, metric, crit=eps, warn=eps, less=True)


def divide_series(a, b):
    return "divideSeries({0}, {1})".format(a, b)


def graphite_get_all_metric(metric_find_expr,
                            host='https://ps-mg.yandex-team.ru',
                            url="/metrics/expand"):
    return requests.get(host + url,
                        {'query': metric_find_expr}).json()['results']
