from paysys.sre.tools.monitorings.lib.checks import base
from paysys.sre.tools.monitorings.lib.checks.active import http
from paysys.sre.tools.monitorings.lib.util.helpers import (
    check,
    merge,
    gen_children_nanny,
    gen_children,
    flaps,
)
import yasm_alert


def awacs(
    hostname,
    balancer,
    services,
    http_ports,
    https_ports,
    ok_codes=None,
    host=None,
    ping_uri="/ping",
):
    """
    Creates check bundle for balancer managed by awacs
    :param hostname: host in juggler
    :param balancer: balancer fqdn
    :param services: list of nanny services - separate nanny service per each dc
    :param http_ports: list of ports routed by slb, slb-ping will be checked
    :param https_ports: list of ports routed by slb, slb-ping will be checked
    :return: dict representation of awacs juggler checks
    """

    instance_checks = {
        "zombie_procs": "check_balancer_zombie_procs",
        "cert_enddate": "check_enddate_certificate",
        "reload_config": "reload_config_status",
    }

    port_checks = {"l7_awacs_ping": "/awacs-balancer-health-check", "l7_ping": ping_uri}

    headers = {"Host": host} if host else {}
    _instance_checks = [
        check(k, gen_children_nanny(services, v))
        for k, v in instance_checks.iteritems()
    ]
    _l7_http_checks = [
        merge(
            http.http(
                "l7_http_{0}_{1}".format(port, k),
                port,
                v,
                ok_codes=ok_codes,
                headers=headers,
                crit=0,
            ),
            check(
                "l7_http_{0}_{1}".format(port, k),
                gen_children_nanny(services, "l7_http_{0}_{1}".format(port, k)),
            ),
        )
        for k, v in port_checks.iteritems()
        for port in http_ports
    ]
    _l7_https_checks = [
        merge(
            http.https(
                "l7_https_{0}_{1}".format(port, k),
                port,
                v,
                ok_codes=ok_codes,
                headers=headers,
                crit=0,
            ),
            check(
                "l7_https_{0}_{1}".format(port, k),
                gen_children_nanny(services, "l7_https_{0}_{1}".format(port, k)),
            ),
        )
        for k, v in port_checks.iteritems()
        for port in https_ports
    ]
    _l3_http_checks = [
        merge(
            http.http(
                "l3_http_{0}_{1}".format(port, k),
                port,
                v,
                ok_codes=ok_codes,
                headers=headers,
                crit=0,
                no_unreach=True,
            ),
            check(
                "l3_http_{0}_{1}".format(port, k),
                gen_children([], []),
            ),
        )
        for k, v in port_checks.iteritems()
        for port in http_ports
    ]
    _l3_https_checks = [
        merge(
            http.https(
                "l3_https_{0}_{1}".format(port, k),
                port,
                v,
                ok_codes=ok_codes,
                headers=headers,
                crit=0,
                no_unreach=True,
            ),
            check(
                "l3_https_{0}_{1}".format(port, k),
                gen_children([], []),
            ),
        )
        for k, v in port_checks.iteritems()
        for port in https_ports
    ]
    return merge(
        base.unreachable,
        check("UNREACHABLE", gen_children_nanny(services, "UNREACHABLE")),
        *_instance_checks
        + _l7_http_checks
        + _l7_https_checks
        + _l3_http_checks
        + _l3_https_checks
    )


def balancer_5xx(namespace, warn=0.01, crit=0.2):
    return check(
        "http_5xx",
        merge(
            {
                "yasm": {
                    "signal": """perc(
                                    balancer_report-report-service_total-outgoing_5xx_summ,
                                    max(balancer_report-report-service_total-requests_summ, const(100))
                              )""",
                    "tags": yasm_alert.Tags(
                        itype=["balancer"],
                        ctype=["prod"],
                        prj=[namespace],
                    ),
                    "warn": [warn, crit],
                    "crit": [crit, None],
                    "mgroups": ["ASEARCH"],
                },
                "tags": [
                    "balancer",
                ],
            },
            flaps(stable=20, critical=100),
        ),
    )
