from itertools import chain
from collections import namedtuple
from paysys.sre.tools.monitorings.lib.checks import graphite
from paysys.sre.tools.monitorings.lib.util import conductor
from paysys.sre.tools.monitorings.lib.util.helpers import merge, gen_children_from_tuples, check


CombaineSettings = namedtuple('CombineSettings', ['crit_by_host', 'warn_by_host'])
CombaineWindowSettings = namedtuple('CombineWindowSettings', ['immediate_multiplier', 'window_multiplier', 'window'])
CombainePercentSettings = namedtuple('CombineWindowSettings', ['crit', 'warn'])


def _get_combine_metric(env, path, postfix):
    return 'paysys.{env}.combaine.{path}.{postfix}'.format(env=env, path=path, postfix=postfix)


def _combaine_metric_wrapper(name, metric, crit, warn):
    return graphite.graphite_check(name, metric, crit, warn, '-20min', url='https://gr-mg.yandex-team.ru')


def nginx_combaine_bundle_windowed(
    conductor_children,
    combaine_env,
    combaine_path,
    combaine_error_settings=None,
    combaine_499_settings=None,
    combaine_error_window_settings=None,
    combaine_499_window_settings=None,
    combaine_error_percent_settings=None,
    combaine_499_percent_settings=None,
    prefix=None
):
    _prefix = "" if prefix is None else prefix + "_"

    num_of_hosts = conductor.num_of_hosts(conductor_children)
    result = []

    if combaine_error_settings is not None:
        crit_errors = combaine_error_settings.crit_by_host * num_of_hosts
        warn_errors = combaine_error_settings.warn_by_host * num_of_hosts

        if combaine_error_window_settings is not None:
            crit_errors *= combaine_error_window_settings.immediate_multiplier
            warn_errors *= combaine_error_window_settings.immediate_multiplier
            crit_errors_window = crit_errors * combaine_error_window_settings.window_multiplier
            warn_errors_window = warn_errors * combaine_error_window_settings.window_multiplier

            result.append(
                _combaine_metric_wrapper(
                    '{}nginx_errors_{}'.format(
                        _prefix,
                        combaine_error_window_settings.window
                    ),
                    graphite.smart_summarize(
                        _get_combine_metric(
                            combaine_env, combaine_path, 'errors',
                        ),
                        combaine_error_window_settings.window
                    ),
                    crit_errors_window,
                    warn_errors_window
                )
            )

        nginx_errors = _combaine_metric_wrapper(
            '{}nginx_errors'.format(_prefix),
            _get_combine_metric(combaine_env, combaine_path, 'errors'),
            crit_errors,
            warn_errors
        )
        result.append(nginx_errors)

    if combaine_499_settings is not None:
        crit_499 = combaine_499_settings.crit_by_host * num_of_hosts
        warn_499 = combaine_499_settings.warn_by_host * num_of_hosts

        if combaine_499_window_settings is not None:
            crit_499 *= combaine_499_window_settings.immediate_multiplier
            warn_499 *= combaine_499_window_settings.immediate_multiplier
            crit_499_window = crit_499 * combaine_499_window_settings.window_multiplier
            warn_499_window = warn_499 * combaine_499_window_settings.window_multiplier

            result.append(
                _combaine_metric_wrapper(
                    '{}nginx_499_{}'.format(_prefix, combaine_499_window_settings.window),
                    graphite.smart_summarize(
                        _get_combine_metric(
                            combaine_env, combaine_path, 'errors',
                        ),
                        combaine_499_window_settings.window,
                    ),
                    crit_499_window,
                    warn_499_window
                )
            )

        nginx_499 = _combaine_metric_wrapper(
            '{}nginx_499'.format(_prefix),
            _get_combine_metric(combaine_env, combaine_path, 'client_closed'),
            crit_499,
            warn_499,
        )
        result.append(nginx_499)

    if combaine_error_percent_settings is not None:
        result.append(
            _combaine_metric_wrapper(
                '{}nginx_errors_percent'.format(_prefix),
                graphite.divide_series(
                    _get_combine_metric(combaine_env, combaine_path, 'errors'),
                    _get_combine_metric(combaine_env, combaine_path, 'total'),
                ),
                combaine_error_percent_settings.crit / 100.0,
                combaine_error_percent_settings.warn / 100.0
            )
        )

    if combaine_499_percent_settings is not None:
        result.append(
            _combaine_metric_wrapper(
                '{}nginx_499_percent'.format(_prefix),
                graphite.divide_series(
                    _get_combine_metric(combaine_env, combaine_path, 'client_closed'),
                    _get_combine_metric(combaine_env, combaine_path, 'total'),
                ),
                combaine_499_percent_settings.crit / 100.0,
                combaine_499_percent_settings.warn / 100.0
            )
        )

    return merge(*result)


def nginx_combaine_bundle(
    conductor_children,
    combaine_env,
    combaine_path,
    crit_errors_by_host=3.0 / 60,
    warn_errors_by_host=3.0 / 60,
    prefix=None
):
    return nginx_combaine_bundle_windowed(
        conductor_children,
        combaine_env,
        combaine_path,
        CombaineSettings(crit_by_host=crit_errors_by_host, warn_by_host=warn_errors_by_host),  # nginx_errors
        None,  # CombaineSettings(crit_by_host=crit_499_by_host, warn_by_host=warn_499_by_host),  # off nginx_499
        None,  # CombaineWindowSettings(immediate_multiplier=5.0, window_multiplier=5.0, window='5min'),  # set to None
        None,  # combaine_499_window_settings override to None
        None,  # combaine_error_percent_settings  override to None
        CombainePercentSettings(1.5, 0.5),  # override 499 percentile for backward compatibility  # nginx_499_percent
        prefix=prefix
    )


def nginx_elasticsearch_bundle(
    conductor_children,
    elasticsearch_agg,
    crit_errors_by_host=15.0,
    warn_errors_by_host=15.0,
    crit_499_by_host=15.0,
    warn_499_by_host=15.0,
    prefix=None,
    disable_499_checks=False
):
    _prefix = "" if prefix is None else prefix + "_"

    num_of_hosts = conductor.num_of_hosts(conductor_children)

    crit_errors = crit_errors_by_host * num_of_hosts
    warn_errors = warn_errors_by_host * num_of_hosts
    crit_499 = crit_499_by_host * num_of_hosts
    warn_499 = warn_499_by_host * num_of_hosts

    nginx_errors = graphite.graphite_check(
        '{}nginx_errors'.format(_prefix),
        'maxSeries(five_min.ps-elastic0*h_yandex_ru.aggregations.all.buckets.{}.errors.doc_count)'.format(
            elasticsearch_agg,
        ),
        crit_errors,
        warn_errors,
        '-20min',
        url='https://ps-mg.yandex-team.ru',
    )

    nginx_499 = graphite.graphite_check(
        '{}nginx_499'.format(_prefix),
        'maxSeries(five_min.ps-elastic0*h_yandex_ru.aggregations.all.buckets.{}.499.doc_count)'.format(
            elasticsearch_agg,
        ),
        crit_499,
        warn_499,
        '-20min',
        url='https://ps-mg.yandex-team.ru',
    )

    if disable_499_checks:
        return merge(
            nginx_errors
        )
    else:
        return merge(
            nginx_errors,
            nginx_499,
        )


def nginx_percentile_bundle(
        name,
        children,
        warn=1,
        crit=1,
        url="https://gr-mg.yandex-team.ru",
        postfix='',
        window="-3min"
):
    check_name = 'nginx_' + name
    subchecks = {}
    hosts = [host.fqdn for host in chain(*[conductor.get_hosts_in_group(child) for child in children])]
    template = "paysys.prod.combaine.{0}.{1}.not_ping_or_redirect_timings.{2}"
    for _host in hosts:
        subchecks[_host] = merge(
            graphite.graphite_check(
                check_name,
                template.format(_host.replace('.', '_'), postfix, name),
                url=url,
                crit=crit,
                warn=warn,
                window=window,
            ),
            check(check_name, {'children': []})
        )
    return check(
        check_name,
        {'tags': ['subchecks']},
        {'subchecks': subchecks},
        gen_children_from_tuples(
            [host, check_name, 'HOST'] for host in hosts
        )
    )
