from paysys.sre.tools.monitorings.lib.util.helpers import solomon_check, merge
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import OutputMixin, ProgramMixin, UpperThresholdAlertBlock, \
    compose_program


class QuotaProgram(OutputMixin, ProgramMixin):
    def __init__(self, metric, project_name, env="prod"):
        self.metric = metric
        self.project_name = project_name
        if env not in ("test", "prod"):
            raise RuntimeError("nirvana env must be either test or prod, {env} provided".format(env=env))
        self.cluster = "quota_prod" if env == "prod" else "quota_dev"

    @property
    def output_names(self):  # type: () -> set[str]
        return {'quota_ratio_used'}

    def program(self):  # type: () -> str
        queries = []
        for attribute in ("actual", "max"):
            queries.append((
                'series_sum({{project="nirvana", cluster="{cluster}", service="quota", '.format(cluster=self.cluster) +
                'sensor="{project}/{metric}", attribute="{attribute}"}})'.format(
                    project=self.project_name, metric=self.metric, attribute=attribute)
            ))

        current, max_value = queries[0], queries[1]
        return "\n".join([
            "let current_value_series = {query};".format(query=current),
            "let max_value_series = {query};".format(query=max_value),
            "let current_value = last(current_value_series);",
            "let max_value = last(max_value_series);",
            "let quota_ratio_used = current_value / max_value;"
        ])


class Checks(object):
    def __init__(self, project_name, project_id, merge_args_dict=None, nirvana_env="prod"):
        self.project_name = project_name
        self.project_id = project_id
        self.nirvana_env = nirvana_env
        self.merge_args_dict = {} if merge_args_dict is None else merge_args_dict

    def check(self, name, metric, alarm_descr, ok_descr,
              crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        name = "nirvana_" + name
        program = QuotaProgram(metric, self.project_name, self.nirvana_env)
        alert_block = UpperThresholdAlertBlock(
            alarm_limit=crit_threshold_ratio,
            warn_limit=warn_threshold_ratio,
            decision_value=program.quota_ratio_used,
        )

        test_prefix = "test." if self.nirvana_env == "test" else ""
        link = "https://{prefix}nirvana.yandex-team.ru/quota/{project_name}".format(
            prefix=test_prefix,
            project_name=self.project_name,
        )

        expr = solomon_expression_custom(
            project_id=self.project_id,
            program_str=compose_program(program, alert_block),
            annotations={
                "description": "{{{{#isAlarm}}}}{alarm_descr}{{{{/isAlarm}}}}\n".format(alarm_descr=alarm_descr) +
                               "{{{{#isWarn}}}}{alarm_descr}{{{{/isWarn}}}}\n".format(alarm_descr=alarm_descr) +
                               "{{{{#isOk}}}}{ok_descr}{{{{/isOk}}}}\n\n".format(ok_descr=ok_descr) +
                               link + "\n",
            },
            window_secs=2 * 60
        )
        return solomon_check(
            name,
            expr,
            {"aggregator_kwargs": {}},
            self.merge_args_dict,
        )

    def bundle(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        # TODO: add yt quotas?
        return merge(
            self.s3_disk(crit_threshold_ratio, warn_threshold_ratio),
            self.workflow_block_start_speed(crit_threshold_ratio, warn_threshold_ratio),
            self.workflow_created_speed(crit_threshold_ratio, warn_threshold_ratio),
            self.workflow_started_speed(crit_threshold_ratio, warn_threshold_ratio),
        )

    def s3_disk(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("s3_disk", "s3-disk",
                          "\n".join([
                              "{{expression.quota_ratio_used}} of s3 disk quota is used",
                              "action: increase quota or decrease storage usage"
                          ]),
                          "you don't have problems with s3 disk quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def workflow_block_start_speed(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("workflow_block_start_velocity", "workflow-block-start",
                          "\n".join([
                              "blocks started in 10min is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new graphs won't be started!",
                              "action: increase quota or decrease blocks start velocity"
                          ]),
                          "you don't have problems with blocks per 10min",
                          crit_threshold_ratio, warn_threshold_ratio)

    def workflow_created_speed(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("workflow_graph_create_velocity", "workflow-create",
                          "\n".join([
                              "graphs created in 10min is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new graphs won't be created!",
                              "action: increase quota or decrease graph creation velocity"
                          ]),
                          "you don't have problems with graphs creation per 10min",
                          crit_threshold_ratio, warn_threshold_ratio)

    def workflow_started_speed(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("workflow_graph_start_velocity", "workflow-start",
                          "\n".join([
                              "graphs started in 10min is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new graphs won't be started!",
                              "action: increase quota or decrease graph start velocity"
                          ]),
                          "you don't have problems with graphs starts per 10min",
                          crit_threshold_ratio, warn_threshold_ratio)
