from collections import OrderedDict
import yasm_alert

from paysys.sre.tools.monitorings.lib.util.helpers import merge, flaps
from paysys.sre.tools.monitorings.lib.checks.doc import doc


def postgres(cluster_id, db_name, cluster_name='', users=None, users_limits=None, **kwargs):
    db_name = '{}-{}'.format(cluster_name, db_name) if cluster_name else db_name
    conn_limit = kwargs.get("conn_limit", 10)
    transaction_time_limit = kwargs.get("transaction_time_limit", 1000)  # ms
    query_time_limit = kwargs.get("query_time_limit", 20)  # ms
    cpu_usage_limit_percent = kwargs.get("cpu_usage_limit_percent", 60)
    cpu_wait_limit = kwargs.get("cpu_wait_limit", 0.75)
    cpu_wait_warn_threshold = kwargs.get("cpu_wait_warn_threshold", cpu_wait_limit)
    cpu_wait_crit_threshold = kwargs.get("cpu_wait_crit_threshold", cpu_wait_limit)
    disk_space_warn_threshold = kwargs.get("disk_space_warn_threshold", 80)
    disk_space_crit_threshold = kwargs.get("disk_space_crit_threshold", 90)
    flaps_stable = kwargs.get("flaps_stable", 120)
    flaps_critical = kwargs.get("flaps_critical", 300)
    tags = kwargs.get("tags", ["postgres"])
    alive_hosts = kwargs.get("alive_hosts", False)
    alive_hosts_warn_threshold = kwargs.get("alive_hosts_warn_threshold", 2)
    alive_hosts_crit_threshold = kwargs.get("alive_hosts_crit_threshold", 1)
    if users is None:
        users = []
    if users_limits is None:
        users_limits = OrderedDict()
        users_limits.setdefault("default", OrderedDict({"crit_percent": 75, "warn_percent": 60}))

    def _postgres_alert(check_name, signal, itype, tier, warn, crit, invert=False):
        """
        Создает описание алерта для pg

        :param check_name: Название проверки
        :param signal: Сигнал, на который алерт будет срабатывать
        :param itype: Тип (обычно, "mailpostgresql")
        :param tier: Тир (обычно, ["primary"])
        :param warn: Уровень для WARNING
        :param crit: Уровень для CRITICAL
        :param invert: Надо ли инвертировать пороги (надо для кол-ва живых хостов, например. см. GOLOVANSUPPORT-1955)
        :return:
        """
        check_name = "{db_name}-" + check_name
        check_name = check_name.format(db_name=db_name)
        warn_ranges = [warn, crit]
        crit_ranges = [crit, None]
        if invert:
            warn_ranges = [crit, warn]
            crit_ranges = [None, crit]

        check = merge(
            {
                "yasm": {
                    "signal": signal.format(db_name=db_name),
                    "tags": yasm_alert.Tags(
                        itype=[itype],
                        ctype=[cluster_id],
                        tier=tier,
                    ),
                    "warn": warn_ranges,
                    "crit": crit_ranges,
                    "mgroups": ["CON"],
                },
                "tags": tags
            },
            flaps(flaps_stable, flaps_critical),
            doc("https://wiki.yandex-team.ru/dljaadminov/paysys/ps/postgres/mon/"),
            doc(
                "https://yasm.yandex-team.ru/template/panel/dbaas_postgres_metrics/cid={}".format(cluster_id),
                title="Dashboard YASM"
            ),
            doc(
                "https://solomon.yandex-team.ru/?cluster=mdb_{}&project=internal-mdb&service=mdb&dashboard=internal-mdb-cluster-postgres&b=1h&e=".format(cluster_id),
                title="Dashboard Solomon"
            )
        )
        return check_name, check

    checks_list = [
        ["pg-conn", "push-{db_name}_conn_active_vmmv", "mailpostgresql", ["primary"], conn_limit, conn_limit, False],
        ["pg-transaction-time", "push-pooler-avg_xact_time_vmmv", "mailpostgresql", ["primary"], transaction_time_limit, transaction_time_limit, False],
        ["pg-query-time", "push-pooler-avg_query_time_vmmv", "mailpostgresql", ["primary"], query_time_limit, query_time_limit, False],
        ["pg-cpu-usage", "perc(portoinst-cpu_usage_cores_tmmv,portoinst-cpu_limit_cores_tmmv)", "mdbdom0", ["primary"], cpu_usage_limit_percent, cpu_usage_limit_percent, False],
        ["pg-cpu-wait", "portoinst-cpu_wait_cores_tmmv", "mdbdom0", ["primary"], cpu_wait_warn_threshold, cpu_wait_crit_threshold, False],
        ["pg-unispace", "perc(push-disk-used_bytes_pgdata_tmmx,push-disk-total_bytes_pgdata_tmmx)", "mailpostgresql", ["primary"], disk_space_warn_threshold, disk_space_crit_threshold, False]
    ]

    # https://wiki.yandex-team.ru/MDB/alerts/#postgresql
    if alive_hosts:
        checks_list.append(
            [
                "pg-alive-hosts",
                "push-postgres-is_alive_tmmx",
                "mailpostgresql",
                None,
                alive_hosts_warn_threshold,
                alive_hosts_crit_threshold,
                True
            ]
        )

    for user in users:
        limits = users_limits.get(user, users_limits.get("default"))
        warn, crit = limits.get("warn_percent"), limits.get("crit_percent")
        checks_list.append(
            [
                "pg-conn-{}".format(user),
                "perc(push-postgres_role_sessions_{user}_tmmx,push-postgres_role_conn_limit_{user}_tmmx)".format(user=user),
                "mailpostgresql", ["primary"],
                warn, crit, False
            ]
        )

    checks = OrderedDict()
    for check_name, signal, itype, tier, warn, crit, invert in checks_list:
        check_name_result, check_dict = _postgres_alert(check_name, signal, itype, tier, warn, crit, invert)
        checks[check_name_result] = check_dict

    return checks
