import yasm_alert
from paysys.sre.tools.monitorings.lib.checks import base, services, doc
from paysys.sre.tools.monitorings.lib.checks.active import http
from paysys.sre.tools.monitorings.lib.util import helpers


def base_checks(
        app,
        children,
        balancer,
        service,
        hostname,
        env,
        path='/ping',
        app_port=8080,
        http_error_warn=3.0,
        http_error_crit=3.0,
        http_error_interval=60,
        http_499_warn=3.0,
        http_499_crit=3.0,
        http_499_interval=60,
        http_5xx_warn=3.0,
        http_5xx_crit=3.0,
        http_5xx_interval=60
):
    """Config for creating base checks for application in qloud
        Args:
            app (str): <project>.<app>
            children (str): <project>.<app>.<env>.<component>@type=ext
            balancer (str): <project>.<app>.<balancer_app>.balancer-l7@type=ext
            service (str): <app>
            hostname (str): <balancer fqdn>
            env (str): <app_environment>
            path (str): <ping location>
            app_port (int): 8080

        Returns:
            Dict representation for qloud base checks

    """
    headers = {"Host": "{}".format(hostname)}
    return helpers.merge(
        http.https(
            "https_" + service,
            headers=headers,
            crit="10%",
            warn="10%",
            path=path
        ),
        {"https_" + service: helpers.gen_children_qloud(balancer, service)},
        {
            "https_"
            + service: {"tags": [
                "qloud-ext.{}.{}".format(app, env)
            ]}
        },
        http.http(service, port=app_port, path=path, headers=headers, warn="10%", crit="10%"),
        {
            service: helpers.gen_children_qloud(children, service)
        },
        {
            service: {"tags": [
                "qloud-ext.{}.{}".format(app, env)
            ]}
        },
        services.pushclient_check,
        {
            "pushclient_check": helpers.gen_children_qloud(
                children,
                "pushclient_check"
            )
        },
        {
            "pushclient_check": {
                "tags": [
                    "qloud-ext.{}.{}".format(app, env)
                ]
            }
        },
        base.unreachable,
        {
            "UNREACHABLE": helpers.gen_children_qloud(
                [children, balancer],
                ['UNREACHABLE']
            )
        },
        {
            "http_error": helpers.merge(
                {
                    "yasm": {
                        "signal": "push-proxy_errors_summ",
                        "tags": yasm_alert.Tags(
                            itype=["qloudrouter"],
                            ctype=[balancer.split(".")[2]],
                            prj=[children.rsplit(".", 1)[0]],
                        ),
                        "warn": [http_error_warn, http_error_warn],
                        "crit": [http_error_crit, None],
                        "value_modify": {"type": "summ", "window": http_error_interval},
                        "mgroups": ["QLOUD"],
                    },
                    "tags": [
                        "qloud-balancer",
                    ]
                },
                doc.doc("https://wiki.yandex-team.ru/DljaAdminov/paysys/PS/qloud/response-codes-monitorings/"),
            ),
        },
        {
            "http_5xx": helpers.merge(
                {
                    "yasm": {
                        "signal": "push-response_5xx_summ",
                        "tags": yasm_alert.Tags(
                            itype=["qloudrouter"],
                            ctype=[balancer.split(".")[2]],
                            prj=[children.rsplit(".", 1)[0]],
                        ),
                        "warn": [http_5xx_warn, http_5xx_warn],
                        "crit": [http_5xx_crit, None],
                        "value_modify": {"type": "summ", "window": http_5xx_interval},
                        "mgroups": ["QLOUD"],
                    },
                    "tags": [
                        "qloud-balancer",
                    ]
                },
                doc.doc("https://wiki.yandex-team.ru/DljaAdminov/paysys/PS/qloud/response-codes-monitorings/"),
            ),
        },
        {
            "http_499": helpers.merge(
                {
                    "yasm": {
                        "signal": "push-http_499_summ",
                        "tags": yasm_alert.Tags(
                            itype=["qloudrouter"],
                            ctype=[balancer.split(".")[2]],
                            prj=[children.rsplit(".", 1)[0]],
                        ),
                        "warn": [http_499_warn, http_499_warn],
                        "crit": [http_499_crit, None],
                        "value_modify": {"type": "summ", "window": http_499_interval},
                        "mgroups": ["QLOUD"],
                    },
                    "tags": [
                        "qloud-balancer",
                    ]
                },
                doc.doc("https://wiki.yandex-team.ru/DljaAdminov/paysys/PS/qloud/response-codes-monitorings/"),
            ),
        },
    )
