from paysys.sre.tools.monitorings.lib.util.helpers import solomon_check, merge
from paysys.sre.tools.monitorings.lib.util.solomon import solomon_expression_custom
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import OutputMixin, ProgramMixin, UpperThresholdAlertBlock, \
    compose_program


class QuotaProgram(OutputMixin, ProgramMixin):
    def __init__(self, metric, project_key, env="prod"):
        self.metric = metric
        self.project_key = project_key
        if env not in ("test", "prod"):
            raise RuntimeError("reactor env must be either test or prod, {env} provided".format(env=env))
        self.env = env  # this might be test if using https://test.reactor.yandex-team.ru

    @property
    def output_names(self):  # type: () -> set[str]
        return {'quota_ratio_used'}

    def program(self):  # type: () -> str
        queries = []
        for sensor in ("currentValue", "maxValue"):
            queries.append((
                'series_sum({{project="reactor", cluster="{env}", service="{env}_pull", '.format(env=self.env) +
                'group="projectQuota", projectKey="{project_key}", '.format(project_key=self.project_key) +
                'resourceTypeKey="{metric}", sensor="{sensor}"}})'.format(metric=self.metric, sensor=sensor)
            ))

        current, max_value = queries[0], queries[1]
        return "\n".join([
            "let current_value_series = {query};".format(query=current),
            "let max_value_series = {query};".format(query=max_value),
            "let current_value = last(current_value_series);",
            "let max_value = last(max_value_series);",
            "let quota_ratio_used = current_value / max_value;"
        ])


class Checks(object):
    def __init__(self, project_key, project_key_id, project_id, merge_args_dict=None, reactor_env="prod"):
        self.project_key = project_key
        self.project_key_id = project_key_id  # id from https://reactor.yandex-team.ru/browse?selected={id}
        self.project_id = project_id
        self.reactor_env = reactor_env
        self.merge_args_dict = {} if merge_args_dict is None else merge_args_dict

    def check(self, name, metric, alarm_descr, ok_descr,
              crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        name = "reactor_" + name
        program = QuotaProgram(metric, self.project_key, self.reactor_env)
        alert_block = UpperThresholdAlertBlock(
            alarm_limit=crit_threshold_ratio,
            warn_limit=warn_threshold_ratio,
            decision_value=program.quota_ratio_used,
        )

        test_prefix = "test." if self.reactor_env == "test" else ""
        link = "https://{prefix}reactor.yandex-team.ru/browse?selected={project_key_id}&selectedTab=quota".format(
            prefix=test_prefix,
            project_key_id=self.project_key_id,
        )

        expr = solomon_expression_custom(
            project_id=self.project_id,
            program_str=compose_program(program, alert_block),
            annotations={
                "description": "{{{{#isAlarm}}}}{alarm_descr}{{{{/isAlarm}}}}\n".format(alarm_descr=alarm_descr) +
                               "{{{{#isWarn}}}}{alarm_descr}{{{{/isWarn}}}}\n".format(alarm_descr=alarm_descr) +
                               "{{{{#isOk}}}}{ok_descr}{{{{/isOk}}}}\n\n".format(ok_descr=ok_descr) +
                               link + "\n",
            },
            window_secs=2 * 60
        )
        return solomon_check(
            name,
            expr,
            {"aggregator_kwargs": {}},
            self.merge_args_dict,
        )

    def bundle(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        # TODO: add trigger and demand alerts?
        return merge(
            self.artifact_bundle(crit_threshold_ratio, warn_threshold_ratio),
            self.artifact_instances_bundle(crit_threshold_ratio, warn_threshold_ratio),
            self.reaction_bundle(crit_threshold_ratio, warn_threshold_ratio),
            self.reaction_instances_bundle(crit_threshold_ratio, warn_threshold_ratio),
        )

    def artifact_bundle(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return merge(
            self.artifacts_total(crit_threshold_ratio, warn_threshold_ratio),
            self.artifact_created_velocity(crit_threshold_ratio, warn_threshold_ratio),
        )

    def artifact_instances_bundle(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return merge(
            self.artifact_instances_total(crit_threshold_ratio, warn_threshold_ratio),
            self.artifact_instances_created_velocity(crit_threshold_ratio, warn_threshold_ratio),
        )

    def reaction_bundle(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return merge(
            self.reaction_total_number(crit_threshold_ratio, warn_threshold_ratio),
            self.reaction_active_total_number(crit_threshold_ratio, warn_threshold_ratio),
        )

    def reaction_instances_bundle(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return merge(
            self.reaction_instances_total(crit_threshold_ratio, warn_threshold_ratio),
            self.running_reaction_instances_number(crit_threshold_ratio, warn_threshold_ratio),
            self.queued_reaction_instances_number(crit_threshold_ratio, warn_threshold_ratio),
            self.timed_out_reaction_instances_number(crit_threshold_ratio, warn_threshold_ratio),
            self.reaction_instances_created_velocity(crit_threshold_ratio, warn_threshold_ratio),
        )

    def reaction_instances_total(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("reaction_instances_total", "reaction.instance.total_number",
                          "\n".join([
                              "total reaction instances is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new reaction instances won't be created!",
                              "action: try to evict old instances or get quota"
                          ]),
                          "you don't have problems with total reaction instances quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def artifacts_total(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("artifacts_total", "artifact.total_number",
                          "\n".join([
                              "total artifacts number (not instances!) is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new artifacts won't be created!",
                              "action: try to remove unnecessary artifacts"
                          ]),
                          "you don't have problems with total artifacts number quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def artifact_instances_total(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("artifact_instances_total", "artifact.instance.total_number",
                          "\n".join([
                              "total artifact instances number is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new artifact instances won't be created!",
                              "action: try to remove unnecessary artifact instances",
                              "CleanupStrategy in UI might also help",
                          ]),
                          "you don't have problems with total artifact instances number quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def artifact_created_velocity(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("artifact_created_velocity", "artifact.created.velocity",
                          "\n".join([
                              "artifacts (not instances!) number created today is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new artifacts won't be created!",
                              "action: don't create artifacts too quickly",
                          ]),
                          "you don't have problems with artifacts creation speed quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def artifact_instances_created_velocity(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("artifact_instances_created_velocity", "artifact.instance.created.velocity",
                          "\n".join([
                              "artifact instances number created today is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new artifact instances won't be created!",
                              "action: don't create artifact instances too quickly",
                          ]),
                          "you don't have problems with artifact instances creation speed quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def reaction_total_number(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("reaction_total_number", "reaction.total_number",
                          "\n".join([
                              "total reactions (not instances!) number is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new reactions won't be created!",
                              "action: increase quota or delete unnecessary reactions",
                          ]),
                          "you don't have problems with total reactions quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def reaction_active_total_number(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("reaction_active_total_number", "reaction.active.total_number",
                          "\n".join([
                              "total active reactions (not instances!) number is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new reactions cannot be activated!",
                              "action: increase quota or delete/deactivate unnecessary reactions",
                          ]),
                          "you don't have problems with total active reactions quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def running_reaction_instances_number(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("reaction_instances_running_number", "reaction.instance.running.total_number",
                          "\n".join([
                              "total running reaction instances number is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new reaction instances will be canceled!",
                              "action: did some reactions hung? Need to stop some instances",
                          ]),
                          "you don't have problems with running reaction instances quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def queued_reaction_instances_number(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("reaction_instances_queued_number", "reaction.instance.queued.total_number",
                          "\n".join([
                              "total queued reaction instances number is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new reaction instances will be canceled on queue overflow!",
                              "action: is there enough throughput for reaction instances or queue is flooding too fast?",
                          ]),
                          "you don't have problems with queued reaction instances quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def timed_out_reaction_instances_number(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("reaction_instances_timed_out_number", "reaction.instance.timeout.total_number",
                          "\n".join([
                              "total delayed reaction instances number is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new reaction instances will be canceled on delayed start!",
                              "action: don't delay instances too much",
                          ]),
                          "you don't have problems with delayed reaction instances quota",
                          crit_threshold_ratio, warn_threshold_ratio)

    def reaction_instances_created_velocity(self, crit_threshold_ratio=0.9, warn_threshold_ratio=0.8):
        return self.check("reaction_instances_created_velocity", "reaction.instance.created.velocity",
                          "\n".join([
                              "reaction instances number today is {{expression.current_value}}, max is {{expression.max_value}}",
                              "new reaction instances won't be created!",
                              "action: slow down reaction instances creation rate",
                          ]),
                          "you don't have problems with reaction instances create speed quota",
                          crit_threshold_ratio, warn_threshold_ratio)
