from paysys.sre.tools.monitorings.lib.util.helpers import (
    merge,
    flaps,
)

from paysys.sre.tools.monitorings.lib.util.yasm import yasm_deploy_threshold_alert


MAX_ALERT_NAME_LENGTH = 128

# Генерация rtc специфичных алертов на системные ресурсы


def get_juggler_check_name(prj, ctype, itype, geo, check_name):
    items = [prj]
    if ctype != 'none':
        items.append(ctype)
    if itype != 'deploy':
        items.append(itype)
    items.append(check_name)
    items.append(geo)

    name = "_".join(items)

    if len(name) > MAX_ALERT_NAME_LENGTH:
        raise ValueError("too long check name {}".format(name))

    return name


def _gen_yasm_check(prj, ctype, itype, geo, signal, check_name, warn, crit, stable=None, critical=None):
    if geo in ["myt", "iva"]:
        geo = "msk"
    yasm_check = {
        get_juggler_check_name(prj, ctype, itype, geo, check_name): merge(
            yasm_deploy_threshold_alert(
                signal_name=signal,
                namespace=prj,
                warn_threshold=warn,
                crit_threshold=crit,
                geo=geo,
                ctype='none'),
            flaps(stable, critical)
        )
    }
    return yasm_check


def rtc_monitorings(geos, prj, ctype, itype, disks=[]):
    """
    Generate deploy unit system monitorings.
    Monitorings:
        * anonymous + unvectible memory usage.
        * cpu usage
        * cpu throtle
        * OOMs
        * crashes
        * network limits

    :param geos: list of str with geos
    :param prj: str
    :param ctype: str
    :param itype: str
    :param disks: list
    """
    assert isinstance(prj, str)
    assert isinstance(ctype, str)
    assert isinstance(itype, str)
    assert isinstance(geos, list) and all(isinstance(geo, str) for geo in geos)

    result = list()
    for geo in geos:
        common_args = {
            "prj": prj,
            "ctype": ctype,
            "itype": itype,
            "geo": geo,
        }
        for disk in disks:
            result.extend(([
                _gen_yasm_check(
                    signal='portoinst-capacity-perc_usage_/virtual_disks/{}_txxx'.format(disk),
                    check_name='disk_usage',
                    warn=75,
                    crit=80,
                    stable=20,
                    critical=100,
                    **common_args
                ),
            ]))

        result.extend([
            _gen_yasm_check(
                signal='quant(portoinst-memory_anon_unevict_limit_usage_perc_hgram, max)',
                check_name='memory_usage',
                warn=75,
                crit=80,
                stable=20,
                critical=100,
                **common_args
            ),
            _gen_yasm_check(
                signal='quant(portoinst-cpu_limit_usage_perc_hgram, max)',
                check_name='cpu_usage',
                warn=50,
                crit=60,
                stable=20,
                critical=100,
                **common_args
            ),
            _gen_yasm_check(
                signal='max(portoinst-ooms_summ, hsum(portoinst-ooms_slot_hgram))',
                check_name='ooms',
                warn=0.1,
                crit=1,
                **common_args
            ),
            _gen_yasm_check(
                signal='max(hsum(portoinst-cores_total_hgram), hsum(portoinst-cores_dumped_hgram))',
                check_name='cores',
                warn=0.1,
                crit=1,
                **common_args
            ),
            _gen_yasm_check(
                signal='max(hsum(portoinst-cores_total_hgram), hsum(portoinst-cores_dumped_hgram))',
                check_name='cores',
                warn=0.1,
                crit=1,
                **common_args
            ),
            _gen_yasm_check(
                signal='quant(portoinst-net_rx_utilization_hgram, max)',
                check_name='net_tx_usage',
                warn=50,
                crit=60,
                stable=20,
                critical=100,
                **common_args
            ),
            _gen_yasm_check(
                signal='quant(portoinst-net_rx_utilization_hgram, max)',
                check_name='net_rx_usage',
                warn=50,
                crit=60,
                stable=20,
                critical=100,
                **common_args
            ),
        ]
        )

    r = merge(*result)
    return r
