import juggler_sdk


from paysys.sre.tools.monitorings.lib.util.solomon import (
    solomon_threshold_expression_monitoring,
    solomon_expression_custom,
    create_solomon_alert_id,
)
from paysys.sre.tools.monitorings.lib.util.solomon_blocks import (
    OutputMixin,
    ProgramMixin,
    UpperThresholdAlertBlock,
    compose_program,
)
from paysys.sre.tools.monitorings.lib.util.helpers import (
    solomon_check,
    merge,
    solomon_equal_selector,
    solomon_by_status_annotation,
)

ACCOUNTS_DASHBOARD_URL = "Quota dashboard: https://solomon.yandex-team.ru/?project=yt&service=accounts&cluster={{labels.cluster}}&account={{labels.account}}&dashboard=yt-account-limits-media"


def yt_used_space(
    project_id,
    account,
    warn_percent_threshold,
    crit_percent_threshold,
    custom_selectors=None,
):
    selectors = {
        "project": "yt",
        "service": "accounts",
        "account": account,
        "sensor": "disk_space_in_gb",
    }

    limit_selectors = {
        "project": "yt",
        "service": "accounts",
        "account": account,
        "sensor": "disk_space_limit_in_gb",
    }

    selectors.update(custom_selectors or {})
    limit_selectors.update(custom_selectors or {})
    key = "{}-used-space".format(account)
    if custom_selectors.get("cluster"):
        key = "{}-{}-used-space".format(custom_selectors.get("cluster"), account)
    description = (
        "{{#isAlarm}}Критическая заполненность квоты YT!{{/isAlarm}}\n"
        "{{#isWarn}}Умеренная заполненность квоты YT{{/isWarn}}"
    )
    annotations = {"description": description}
    checks = {
        key: {
            "solomon": solomon_threshold_expression_monitoring(
                project_id,
                selectors,
                limit_selectors,
                warn_percent_threshold,
                crit_percent_threshold,
                annotations,
                "YT used space monitoring",
                repeat_delay_secs=0,
            )
        }
    }
    return checks


class QuotaProgram(OutputMixin, ProgramMixin):
    def __init__(
        self,
        usage_selector,  # type: str
        limit_selector,  # type: str
    ):
        self.usage_selector = usage_selector
        self.limit_selector = limit_selector

    def program(self):  # type: () -> str
        return "\n".join([
            "let usage = {0};".format(self.usage_selector),
            "let limit = {0};".format(self.limit_selector),

            "let avg_usage = avg(usage);",
            "let avg_limit = avg(limit);",
            "let use_percent = avg_usage/avg_limit * 100;",

            "let avg_usage_str = to_fixed(avg_usage, 2);",
            "let avg_limit_str = to_fixed(avg_limit, 2);",
            "let usage_percent_str = to_fixed(use_percent, 2) + '%';"
        ])

    @property
    def output_names(self):  # type: () -> set[str]
        return {'avg_usage_str', 'avg_limit_str', 'use_percent', 'usage_percent_str'}


class Checks(object):
    """
    Check from YT solomon templates
    """
    def __init__(
        self,
        solomon_project_id,  # type: str
        host,  # type: str
        clusters_list=None,  # type: Optional[list[str]]
    ):
        """
        empty lists for `clusters_list` means "all"
        """
        self.solomon_project_id = solomon_project_id
        self.host = host
        self.selected_cluster = "|".join(clusters_list) if clusters_list else None

    def account_bundle(
        self,
        account_list=None,  # type: Optional[list[str]]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        alert_suffix="",  # type: str
    ):
        selected_accounts = "|".join(account_list) if account_list else None

        return merge(
            self.chunk_count_quota(selected_accounts, crit_threshold_percent, warn_threshold_percent, alert_suffix),
            self.disk_space_quota(selected_accounts, crit_threshold_percent, warn_threshold_percent, alert_suffix),
            self.node_count_quota(selected_accounts, crit_threshold_percent, warn_threshold_percent, alert_suffix),
        )

    def pool_bundle(
        self,
        pools_list=None,  # type: Optional[list[str]]
        tree_list=None,  # type: Optional[list[str]]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        alert_suffix="",  # type: str
    ):
        selected_pools = "|".join(pools_list) if pools_list else None
        selected_trees = "|".join(tree_list) if tree_list else None

        return merge(
            self.running_operation_count_limit(
                selected_pools, selected_trees,
                crit_threshold_percent, warn_threshold_percent,
                alert_suffix
            ),
            self.operation_count_limit(
                selected_pools, selected_trees,
                crit_threshold_percent, warn_threshold_percent,
                alert_suffix
            ),
        )

    def _check(
        self,
        name,  # type: str
        usage_selector,  # type: str
        limit_selector,  # type: str
        value_description,  # type: str
        alarm_description,  # type: str
        labels_grouping,  # type: list[str]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        extra_description='',  # type: str
        alert_suffix="",  # type: str
    ):
        if not labels_grouping:
            raise RuntimeError('Alert %s provided no label grouping' % (self.host + name))

        alert_name = 'yt_' + name + alert_suffix
        queries_block = QuotaProgram(usage_selector, limit_selector)
        alert_block = UpperThresholdAlertBlock(
            alarm_limit=crit_threshold_percent,
            warn_limit=warn_threshold_percent,
            decision_value=queries_block.use_percent,
        )

        expr = solomon_expression_custom(
            project_id=self.solomon_project_id,
            program_str=compose_program(queries_block, alert_block),
            window_secs=10 * 60,
            delay_secs=60,
            group_by_labels=labels_grouping,
            juggler_children={
                "replace": True,
                "children": [
                    juggler_sdk.Child(
                        create_solomon_alert_id(self.host, alert_name),
                        "all",
                        "all",
                        "MONITORING_MULTIALERT",
                    ),
                ],
            },
            annotations={
                "description": "\n".join([
                    "Usage is {used}/{limit} ({percent}) {descr}".format(
                        used=queries_block.expr_avg_usage_str,
                        limit=queries_block.expr_avg_limit_str,
                        percent=queries_block.expr_usage_percent_str,
                        descr=value_description
                    ),
                    solomon_by_status_annotation(
                        'Alarm',
                        alarm_description,
                    ),
                    extra_description,
                ])
            }
        )

        return solomon_check(
            alert_name,
            expr,
            {"aggregator_kwargs": {}},
        )

    def _account_selector(self, sensor, account):
        return solomon_equal_selector(
            project='yt',
            service='accounts',
            sensor=sensor,
            cluster=self.selected_cluster,
            account=account,
        )

    def chunk_count_quota(
        self,
        selected_accounts,  # type: Optional[str]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        alert_suffix="",  # type: str
    ):
        return self._check(
            name='chunk_count_quota',
            usage_selector=self._account_selector('chunk_count', selected_accounts),
            limit_selector=self._account_selector('chunk_count_limit', selected_accounts),
            value_description='chunks at {{labels.account}} on {{labels.cluster}}',
            alarm_description='YT Chunk quota usage is extreme!',
            extra_description=ACCOUNTS_DASHBOARD_URL,
            labels_grouping=['cluster', 'account'],
            crit_threshold_percent=crit_threshold_percent,
            warn_threshold_percent=warn_threshold_percent,
            alert_suffix=alert_suffix,
        )

    def disk_space_quota(
        self,
        selected_accounts,  # type: Optional[str]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        alert_suffix="",  # type: str
    ):
        selector = '{{project="yt", service="accounts", medium!="-", ' \
                   'sensor="{sensor}", cluster="{cluster}", account="{account}"}}'

        return self._check(
            name='disk_space_quota',
            usage_selector=selector.format(
                sensor='disk_space_in_gb',
                account=selected_accounts,
                cluster=self.selected_cluster,
            ),
            limit_selector=selector.format(
                sensor='disk_space_limit_in_gb',
                account=selected_accounts,
                cluster=self.selected_cluster,
            ),
            value_description='GB of medium {{labels.medium}} at {{labels.account}} on {{labels.cluster}}',
            alarm_description='YT Disk quota usage is extreme!',
            extra_description=ACCOUNTS_DASHBOARD_URL,
            labels_grouping=['cluster', 'account', 'medium'],
            crit_threshold_percent=crit_threshold_percent,
            warn_threshold_percent=warn_threshold_percent,
            alert_suffix=alert_suffix,
        )

    def node_count_quota(
        self,
        selected_accounts,  # type: Optional[str]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        alert_suffix="",  # type: str
    ):
        return self._check(
            name='node_count_quota',
            usage_selector=self._account_selector('node_count', selected_accounts),
            limit_selector=self._account_selector('node_count_limit', selected_accounts),
            value_description='nodes at {{labels.account}} on {{labels.cluster}}',
            alarm_description='YT Node quota usage is extreme!',
            extra_description=ACCOUNTS_DASHBOARD_URL,
            labels_grouping=['cluster', 'account'],
            crit_threshold_percent=crit_threshold_percent,
            warn_threshold_percent=warn_threshold_percent,
            alert_suffix=alert_suffix,
        )

    def _pool_selector(self, sensor, pool, tree):
        return solomon_equal_selector(
            project='yt',
            service='scheduler_pools',
            sensor=sensor,
            cluster=self.selected_cluster,
            pool=pool,  # type: str
            tree=tree,  # type: str
        )

    def running_operation_count_limit(
        self,
        selected_pools,  # type: Optional[str]
        selected_trees,  # type: Optional[str]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        alert_suffix="",  # type: str
    ):
        return self._check(
            name='running_operation_count_limit',
            usage_selector=self._pool_selector(
                'yt.scheduler.pools.running_operation_count',
                selected_pools,
                selected_trees,
            ),
            limit_selector=self._pool_selector(
                'yt.scheduler.pools.max_running_operation_count',
                selected_pools,
                selected_trees,
            ),
            value_description='running operations on {{labels.pool}}:{{labels.tree}} at {{labels.cluster}}',
            alarm_description='Running operation count is close to the limit',
            labels_grouping=['pool', 'tree'],
            crit_threshold_percent=crit_threshold_percent,
            warn_threshold_percent=warn_threshold_percent,
            alert_suffix=alert_suffix,
        )

    def operation_count_limit(
        self,
        selected_pools,  # type: Optional[str]
        selected_trees,  # type: Optional[str]
        crit_threshold_percent=90,  # type: int
        warn_threshold_percent=80,  # type: int
        alert_suffix="",  # type: str
    ):
        return self._check(
            name='operation_count_limit',
            usage_selector=self._pool_selector(
                'yt.scheduler.pools.total_operation_count',
                selected_pools,
                selected_trees,
            ),
            limit_selector=self._pool_selector(
                'yt.scheduler.pools.max_operation_count',
                selected_pools,
                selected_trees,
            ),
            value_description='total operations on {{labels.pool}}:{{labels.tree}} at {{labels.cluster}}',
            alarm_description='Total operation count is close to the limit',
            labels_grouping=['cluster', 'pool', 'tree'],
            crit_threshold_percent=crit_threshold_percent,
            warn_threshold_percent=warn_threshold_percent,
            alert_suffix=alert_suffix,
        )
