import os
import logging
from conductor_client import Group, set_token, set_path
from conductor_client import models
from functools32 import lru_cache
from itertools import chain


def get_conductor_token():
    try:
        from juggler_token import conductor

        return conductor
    except ImportError:
        return os.environ["CONDUCTOR_TOKEN"]


@lru_cache(None)
def init_conductor():
    set_token(get_conductor_token())
    set_path("http://c.yandex-team.ru/api/v1")


@lru_cache(None)
def get_hosts_in_group(name):
    init_conductor()
    logging.debug("conductor: getting hosts from [{0}] group.".format(name))
    group = Group(name=name)
    result = []
    if group.exists():
        result += group.hosts
        for g in group.child_groups:
            logging.debug("conductor: child group [{0}].".format(g.name))
            result += g.hosts
    else:
        logging.error("conductor: group [{0}] does not exists".format(name))
    return result


@lru_cache(None)
def is_group_exists(name):
    init_conductor()
    group = Group(name=name)
    return group.exists()


def num_of_hosts(groups_list):
    init_conductor()
    return len(
        [h for g in groups_list for h in get_hosts_in_group(g)]
    )


@lru_cache(None)
def get_subgroups_by_dc(name):
    init_conductor()

    postfixes = [
        '-sas',
        '-vla',
        '-myt',
        '-iva',
        '-man'
    ]

    result = []
    group = Group(name=name[0])

    if group.exists():
        for g in group.child_groups:
            for p in postfixes:
                if g.name.endswith(p):
                    result.append(g.name)
                    break
            else:
                return []

    return result


@lru_cache(None)
def split_hosts_by_dc(groups_tuple):
    """
    Loads all host from conductor group and splits them by dc
    """
    init_conductor()
    result = {}
    group = ""
    if isinstance(groups_tuple, tuple):
        group = list(groups_tuple)[0]
    else:
        group = groups_tuple

    hosts = get_hosts_in_group(group)

    for host in hosts:
        dc = "." + (host.datacenter.parent.name if isinstance(host.datacenter.parent, models.Datacenter) else host.datacenter.name)
        if dc not in result:
            result[dc] = set()
        result[dc].add(host.fqdn)

    return result


def get_fqdn_from_group(group_list):
    return sorted(
        [
            host.fqdn
            for host in chain(
                *[get_hosts_in_group(group) for group in group_list]
            )
        ]
    )
