import collections
import copy
import itertools
import logging
import os

from juggler_sdk import Child, Check, FlapOptions, JugglerApi

import paysys.sre.tools.monitorings.lib.util.conductor as conductor
import solomon
import yasm
from paysys.sre.tools.monitorings.lib.util.aggregators import logic_or, empty_kwargs, more_than_percent_is_crit


def get_solomon_token():
    return os.environ["SOLOMON_TOKEN"]


def get_juggler_token():
    try:
        from juggler_token import juggler

        return juggler
    except ImportError:
        return os.environ["JUGGLER_TOKEN"]


def get_yc_token():
    return os.environ.get("YC_TOKEN")


def get_juggler_api(mark, dry_run=False):
    api_rw = JugglerApi(
        "http://juggler-api.search.yandex.net",
        mark=mark,
        oauth_token=get_juggler_token(),
        dry_run=dry_run,
    )
    return api_rw


def update(a, b):
    for k in b.keys():
        if isinstance(a, collections.Mapping):
            if isinstance(b[k], collections.Mapping) and b[k]:
                a[k] = update(a.get(k, {}), b[k])
            elif isinstance(b[k], list) and b[k]:
                a[k] = [item for item in a.get(k, []) if item not in b[k]]
                a[k] += b[k]
            else:
                a[k] = b[k]
        else:
            a = {k: b[k]}
    return a


def merge(*args):
    """
    :param: args: checks dicts
    :return: merged check dict
    """
    if not args:
        return {}
    c = copy.deepcopy(args[0])
    for b in args[1:]:
        c = update(c, copy.deepcopy(b))
    return c


def gen_unreach(checks, mode='skip'):
    services = []
    for check in checks:
        if ':' in check:
            services.append({'check': check})
            continue
        services.append({'check': ':{0}'.format(check)})

    return {
        'aggregator_kwargs': {
            'unreach_service': services,
            'unreach_mode': mode
        }
    }


def unreach_checks(checks):
    return {
        'aggregator_kwargs': {
            'unreach_checks': checks,
        }
    }


def gen_unreach_by_service(checks):
    tiers = {
        'uptime':
            [
                # common
                'bmc', 'bora-configs', 'check_slb', 'clock', 'configs', 'cpu',
                'crs-autostart', 'daytime', 'disk', 'ecc', 'gluster_health',
                'gpu', 'graphite-client', 'grid-up', 'grub', 'haproxy',
                'hbf_service_last_update_status', 'hbf_service_status',
                'host_memory_usage', 'hugepages', 'hwerrors2', 'ip-tunnels',
                'iscsi-ips', 'iscsi-params', 'jmx2graphite-config-generator',
                'jmx2graphite-process', 'keepalived_conf', 'ldaps', 'link',
                'link-speed', 'link_drop', 'link_errs', 'link_flaps',
                'link_utilization', 'load_average_paysys', 'mail_queue',
                'mem', 'mount', 'nginx-alive', 'nginx-errors', 'ntp-stratum',
                'rdbms-audit-tmpfs', 'oracle-patches', 'oracleasm_iid_cnt',
                'paysys-common-settings', 'raid', 'raid_check_speed',
                'stunnel', 'switch_uplink_drop', 'switch_uplink_errs',
                'switch_uplink_flaps', 'switch_uplink_utilization', 'sysctl',
                'tvm_health', 'unbound', 'unispace', 'unispace5', 'watchdog'
            ] + [
                # ebs-specific
                'check-certs', 'cpu_utilize', 'ebs_homes', 'epel-syncer',
                'ldap-limits', 'locker-server', 'pipeline-sender', 'prn_stat',
                'scheduler', 'ssl-certs', 'tomcat-alive', 'yandex-cauth-cache'
            ],
        'grid-up':
            [
                # common
                'asm-status-ng', 'database-up', 'icm', 'oebs-up', 'oracle_uptime',
                'oracle_uptime_standby'
            ] + [
                # ebs-specific
                'context_files', 'httpd-webdav', 'icm', 'java-OOM-count',
                'keystore-certs', 'monitor_custom', 'oebs_syncer, check_acfs'
            ],
        'database-up':
            [
                # common
                'audit-splunk',
                'db-file_locations', 'db-parameters', 'mvlogmove',
                'net_services', 'oracle_memory_usage', 'ps_segment_info',
                'ps_ts_info', 'psdbstat', 'rm-logs', 'rm-traces',
                'shared_pool_free_mem', 'standby-lag', 'store_audit',
                'user-objects-in-sys', 'oracle-balance-patches'
            ] + [
                # ebs-specific
                'adop-nodes', 'common-dirs', 'ebs-processes', 'etcc-check',
                'fk-indexing', 'matview-monitor', 'matview-refresh',
                'oebs-five-min', 'oebs-one-day', 'oebs-one-hour',
                'oebs-one-min', 'oracle-status', 'ro-services',
                'runbackup_rman', 'seq-check', 'mviewlog-usage',
                's3backup-pcrm', 's3backup-mhyp', 's3backup-scatnew',
                's3backup-meta', 's3backup-balance', 's3backup-billingutf',
                's3backup-oebs', 's3backup-bi', 's3backup-hyp',
            ] + [
                # oracle-statuses
                'oracle-status-scat1', 'oracle-status-scat2',
                'oracle-status-meta1', 'oracle-status-meta2',
                'oracle-status-balance1', 'oracle-status-balance2',
                'oracle-status-cft1'
            ],
        # oebs.function
        'oebs.stable.db:database-up':
            [
                'a-timeouted-conc_automat', 'f-disabled-xx-pay_trig_events',
                'f-duplicated-pqh_roles', 'f-duplicated-summary_accounts',
                'f-enddate_records_wf_relation', 'f-error-fisc_check',
                'f-error-fnd_flex_hierarchies', 'f-error-fnd_svc_components',
                'f-error-phase_conc', 'f-error-programs',
                'f-error-programs-taxi', 'f-error-xxya_log_messages',
                'f-fnd_lookup_values', 'f-full-concurrent_queue',
                'f-inactive-cost_manager', 'f-interfaces',
                'f-invalid-xx-all_objects', 'f-invalid-xx-triggers',
                'f-large-output_programs', 'f-long-concurrent',
                'f-long-running-programs', 'f-long-running-programs-taxi',
                'f-long-running-system', 'f-meta-endocs',
                'f-non-comp-xx_xla_prd_rul', 'f-profile-values',
                'f-send-error-fisc_check', 'f-sla_gl_final_mode',
                'f-timeouted-conc_automat', 'f-timeouted-conc_automat-taxi',
                'f-wf-out-errors', 'f-workflow-mail',
                'f-ws-ping', 'f-error-programs-billing-int'
            ],
        # 'oebs-up':
        'icm':
            [
                'OACore_monitor', 'XXSA', 'autopatch', 'bot_queue_size',
                'check_depersonal', 'oebs-apache', 'oebs_notify',
                'oebs_ports', 'opp_check', 'opp_flap', 'outsource-mailer',
                'rwrun', 'tar-apps', 'weblogic', 'wf_bgproc_check',
                'workflow'
            ],
        # bi
        'mstr_health':
            [
                'mstr_web',
                'mstr_aux',
                'mstr_config',
                'mstr_coredumps',
                'keystores',
                'mstr_library',
            ],
        'tomcat_health':
            [
                'tomcat_config',
            ],
        # hyp
        'weblogic_health':
            [
                'weblogic_oom',
                'hyp-health',
                'weblogic_overall_health_state',
            ],
        # vertica
        'database_health':
            [
                'database_copycluster',
                'database_config',
                'database_backup',
                'database_audit_size',
                'vcpuperf',
            ],
        # misc
        'locker-server':
            [
                'failed_locks', 'staled_clients', 'staled_locks',
                'unfetched_locks'
            ],
        'nginx-alive':
            [
                'https_certificate'
            ],
        'tomcat-alive':
            [
                'ping-oebsapi', 'ping-wto'
            ]
    }
    result = {}
    for check_name, params in checks.items():
        if check_name == 'UNREACHABLE':
            result[check_name] = params
            continue
        for tier, childs in tiers.items():
            if check_name in childs:
                result[check_name] = merge(gen_unreach(['UNREACHABLE', tier]), params)
        if check_name not in result:
            result[check_name] = params
    return result


def ttl(ttl=600, refresh_time=60):
    return {'ttl': ttl, 'refresh_time': refresh_time}


def flaps(stable, critical, boost=0):
    return {'flaps_config': FlapOptions(stable, critical, boost)}


unreach_force_ok = gen_unreach(['UNREACHABLE'], 'force_ok')
unreach_skip = gen_unreach(['UNREACHABLE'], 'skip')

# Legacy aggregator kwargs are used as default in YASM instead of `unreach_skip`
#  because in older commits all aggregator_kwargs were overwritten in
#  https://a.yandex-team.ru/arc_vcs/paysys/sre/tools/monitorings/lib/util/yasm.py#L155
#  using
#    kwargs.update('aggregator_kwargs':....)
#  instead of
#    if 'aggregator_kwargs' not in kwargs:
#      kwargs.update('aggregator_kwargs':....)
# Hence, all YASM kwargs have these settings by default and if we just fix updating of defaults arguments
#  it will screw many projects. Nevertheless, changing behaviour of the method `yasm_set_juggler_check_defaults`
#  is essential and must be done, so we rewrite default aggregator_kwargs for YASM checks using these settings.
legacy_aggregator_kwargs = {'aggregator_kwargs': {'unreach_service': [{'check': 'yasm_alert:virtual-meta'}],
                                                  'nodata_mode': 'force_ok', 'unreach_mode': 'force_ok'}}

nodata_force_ok = {'aggregator_kwargs': {'nodata_mode': 'force_ok'}}
nodata_skip = {'aggregator_kwargs': {'nodata_mode': 'skip'}}

skip_split_by_dc = {"skip_split_by_dc": True}


def gen_children(children, services, group_type='CGROUP'):
    if not isinstance(children, list):
        children = [children]
    if not isinstance(services, list):
        services = [services]
    return {
        'children': [Child(child, service, group_type=group_type) for child, service in
                     itertools.product(children, services)]
    }


def gen_children_from_tuples(children_list):
    return {'children': [Child(child, service, group_type=type) for child, service, type in children_list]}


def gen_children_qloud(children, services):
    return gen_children(children, services, group_type='QLOUD')


def gen_children_nanny(children, services):
    return gen_children(children, services, group_type='NANNY')


def gen_children_deploy(children, services):
    return gen_children(children, services, group_type='DEPLOY')


config_defaults = merge(
    ttl(620, 60),
    {'namespace': 'psadmin.unsorted'},
)


def set_defaults(defaults=None, **kwargs):
    _defaults = defaults if defaults is not None else config_defaults
    kwargs = merge(_defaults, kwargs)
    if 'aggregator' not in kwargs:
        kwargs.update(logic_or)
    if 'responsible' in kwargs:
        logging.warning("Field 'responsible' is deprecated, please remove it from your checks")
    return kwargs


def gen_check(host, children_list, check, defaults, **kwargs):
    kwargs = set_defaults(defaults, **kwargs)

    children = gen_children(children_list, check) if kwargs and 'children' not in kwargs else {
        'children': kwargs['children']}
    kwargs.update(children)

    if 'active' not in kwargs.keys():
        if 'aggregator_kwargs' not in kwargs:
            kwargs.update(unreach_skip)
        if 'yasm' in kwargs.keys():
            # overriding default aggregator kwargs deliberately, see legacy_aggregator_kwargs description.
            if 'forced_aggregator_kwargs' not in kwargs:
                kwargs.update(legacy_aggregator_kwargs)
            else:
                kwargs['aggregator_kwargs'] = kwargs['forced_aggregator_kwargs']
                kwargs.pop('forced_aggregator_kwargs')

            if 'forced_aggregator' in kwargs:
                kwargs['aggregator'] = kwargs['forced_aggregator']
                kwargs.pop('forced_aggregator')

            kwargs = yasm.yasm_set_juggler_check_defaults(kwargs, host, check)
        elif 'solomon' in kwargs.keys():
            kwargs = solomon.solomon_set_juggler_check_defaults(host, check, **kwargs)
    return Check(host, service=check, **kwargs)


def update_checks(checks, patch):
    """
    :param: checks: dict with checks
    :param: patch:  dict with patch
    :return: patched checks dict with override parameters from patch dict
    """
    for key in checks.keys():
        checks[key] = merge(
            checks[key],
            patch,
        )

    return checks


def update_notifications(check, host, args, notifications):
    """
    Update notifications based on notification settings.
    By default the 'default' notification settings will be applied.
    If there are 'by_service', 'by_tag' or 'by_host' key in notifications dict
    than it will be applied.
    Settings precedence in increasing order: default, by_tag, by_service, by_host.
    """
    if 'notifications' not in args.keys():
        settings = notifications.get('default')

        by_service = notifications.get('by_service') or {}
        by_tag = notifications.get('by_tag') or {}
        by_host = notifications.get('by_host') or {}

        check_tags = args.get('tags') or []
        tags_matched = [tag for tag in by_tag.keys() if tag in check_tags]

        if len(tags_matched) > 2:
            logging.warning(
                "Matched more than 1 tag: {}, resulting notification settings."
                "will be undetermined.".format(tags_matched))

        if tags_matched:
            settings = by_tag[tags_matched[0]]

        if check in by_service.keys():
            settings = by_service[check]

        if host in by_host.keys():
            settings = by_host[host]

        args = merge(args, settings)
    return args


def __extract_subchecks(check):
    subchecks = {}
    if 'tags' in check.keys():
        if 'subchecks' in check['tags']:
            check['tags'].remove('subchecks')
            subchecks = check.pop('subchecks')
    return subchecks


def __update_tags(host, juggler_check):
    # Tags for easy searching
    new_tags = []
    for item in host.split('.'):
        new_tags.append(
            (new_tags[-1] if len(new_tags) else 'a_mark') + '_' + item
        )

    # UZEDOADMIN-417: we dont want to see this alerts on our dashboard in juggler
    notifications = juggler_check.notifications if juggler_check.notifications else []
    args = [notification.template_kwargs.pop("on_dashboard", True) for notification in notifications]
    on_dashboard = False
    if False not in args:
        on_dashboard = True
    if notifications and on_dashboard:
        new_tags.append(
            'a_mark_{}_with_notifications'.format(
                host.split('.')[0]
            )
        )

    else:
        new_tags.append(
            'a_mark_{}_without_notifications'.format(
                host.split('.')[0]
            )
        )

    tags = juggler_check.tags
    tags += new_tags
    juggler_check.tags = tags
    # Allow disable tags with special tag
    if 'no_tags' in juggler_check.tags:
        juggler_check.tags = []


def __gen_check(host, children_list, service, defaults, split_by_dc, **kwargs):
    def __prepare_check(_dc, _children=None, **kwargs):
        _host = "{0}{1}".format(host, _dc)
        return gen_check(
            _host,
            _children,
            service,
            defaults,
            **kwargs
        )

    result = []
    skip_split_by_dc = kwargs.pop('skip_split_by_dc', False)
    if split_by_dc and kwargs and 'children' not in kwargs and not skip_split_by_dc:
        subroups_by_dc = conductor.get_subgroups_by_dc(tuple(children_list))
        if subroups_by_dc:
            skipped_groups = kwargs.pop('skipped_groups', [])
            for group in subroups_by_dc:
                if group in skipped_groups:
                    continue
                result.append(__prepare_check("." + group.split('-')[-1], group, **kwargs))
        else:
            for dc, hosts in conductor.split_hosts_by_dc(tuple(children_list)).items():
                _kwargs = merge(kwargs, gen_children(list(hosts), service, group_type='HOST'))
                result.append(__prepare_check(dc, ['children'], **_kwargs))
    else:
        result.append(__prepare_check("", children_list, **kwargs))

    return result


def gen_checks_for_host(host, children, checks, notifications=None, defaults=None, split_by_dc=None):
    notifications = notifications if notifications else {}
    juggler_checks = []

    for check, kwargs in checks.items():
        kwargs = update_notifications(check, host, kwargs, notifications)
        subchecks = __extract_subchecks(kwargs)

        result = __gen_check(host, children, check, defaults, split_by_dc, **kwargs)

        for _host, _check in subchecks.iteritems():
            for _service, _kwargs in _check.items():
                result += __gen_check(_host, children, _service, defaults, split_by_dc, **_kwargs)

        for r in result:
            __update_tags(host, r)

        juggler_checks += result

    return juggler_checks


kwargs_day_time = {
    "limits": [
        {
            "crit": 0,
            "day_end": 7,
            "day_start": 1,
            "time_end": 21,
            "time_start": 10,
            "warn": 0
        },
        {
            "crit": "101%",
            "day_end": 7,
            "day_start": 1,
            "time_end": 9,
            "time_start": 22,
            "warn": 0
        }
    ]
}

kwargs_work_day_time = {
    "limits": [
        {
            "crit": 0,
            "day_end": 5,
            "day_start": 1,
            "time_end": 20,
            "time_start": 12,
            "warn": 0
        },
        {
            "crit": "101%",
            "day_end": 5,
            "day_start": 1,
            "time_end": 11,
            "time_start": 21,
            "warn": 0
        },
        {
            "crit": "101%",
            "day_end": 7,
            "day_start": 6,
            "time_end": 23,
            "time_start": 0,
            "warn": 0
        }
    ]
}


def _dict(name, *args):
    _args = {}
    for arg in args:
        if arg:
            _args.update(arg)
    return {name: _args}


check = _dict


def solomon_check(name, solomon_expr, *args):
    return check(name, {"solomon": solomon_expr}, *args)


def create_subchecks(name, subhost, checks):
    return check(
        name,
        {'tags': ['subchecks']},
        {'subchecks': {subhost: checks}},
        empty_kwargs,
        gen_children_from_tuples(
            [subhost, c, 'HOST'] for c in checks.keys()
        ),
    )


empty_children = []


def make_aggregated_check(check, percent):
    return {check.keys()[0]: merge(check.values()[0], more_than_percent_is_crit(percent))}


def skip_check_by_groups(name, groups):
    return {name: {"skipped_groups": groups}}


def cleanup_unreacheable(checks):
    mask = {'check': ':UNREACHABLE'}
    for k, v in list(checks.items()):
        if 'aggregator_kwargs' not in v:
            continue
        if 'unreach_service' not in v['aggregator_kwargs']:
            continue
        if mask not in v['aggregator_kwargs']['unreach_service']:
            continue
        v['aggregator_kwargs']['unreach_service'].remove(mask)
        if not v['aggregator_kwargs']['unreach_service']:
            _ = v['aggregator_kwargs'].pop('unreach_mode')
            _ = v['aggregator_kwargs'].pop('unreach_service')
        checks[k] = v
    return checks


def solomon_equal_selector(**conditions):
    return "{" + ",".join('{0}="{1}"'.format(k, v) for k, v in conditions.items() if v is not None) + "}"


def solomon_by_status_annotation(
    status,
    text,
    revert_status=False,
):
    existing_statuses = ['Ok', 'Warn', 'Alarm', 'NoData', 'Error']
    if status not in existing_statuses:
        raise RuntimeError(
            "Status must be one of {acceptable} got: {got}".format(
                acceptable=existing_statuses,
                got=status
            )
        )

    return (
        "{{{{{opening}is{status}}}}}".format(
            opening='^' if revert_status else '#',
            status=status
        )
        + text
        + "{{{{/is{status}}}}}".format(
            status=status
        )
    )


def join_solomon_program_code(*program_rows):
    """
    like default join, but has delimiter at the end of output
    :param program_rows: solomon custom program string rows
    :return: string concatenated with the delimiter and has delimiter at the end
    """
    delimiter = ';\n'
    if not program_rows:
        raise ValueError(
            'The program rows cannot be an empty value'
        )
    else:
        return delimiter.join(program_rows) + delimiter


def get_annotation(text, status='Alarm'):
    return {"description": solomon_by_status_annotation(status=status, text=text)}
