# coding=utf-8
import juggler_sdk
import logging
import solomon_client
from prettytable import PrettyTable

# if nested field needs to be ignored write keys with dot Ex: 'alerts.some_field'
# indexes in lists are ignored Ex: 'alerts.some_field' is applied for every item of list field 'alerts'
IGNORE_ON_COMPARE_FIELDS = {'created_by', 'created_at', 'updated_at', 'version', 'updated_by', 'notification_channels'}

SORT_BEFORE_COMPARE_LISTS = {"channels.config.notify_about_statuses", }

# If an alert name does not start with the following prefix,
# the MAAC will not delete the alert from Solomon
MAAC_MANAGED_PREFIX = "maac_"
MAAC_JUGGLER_EVENT_DESCRIPTION = "\n".join([
    "{{{annotations.description}}}",
    "Changed at: {{evaluatedAt}}",
    "Url: {{{url}}}",
])


def create_solomon_alert_id(host, check):
    return "{}-{}".format(host.replace(".", "-"), check)


def selectors_to_string(selectors):
    return "{%s}" % ", ".join("{}='{}'".format(k, v) for k, v in selectors.items())


class SolomonAlertingService(object):
    def __init__(
        self,
        token,
        is_dry_run=True,
        no_cleanup=True,
    ):
        self.active_alerts = set()
        self.collected_alerts = []
        self.client = solomon_client.AlertingApi(
            solomon_client.ApiClient(
                solomon_client.Configuration(),
                "Authorization",
                "OAuth " + token
            )
        )
        self.is_dry_run = is_dry_run
        self.no_cleanup = no_cleanup

    def collect_alerts(
        self,
        alerts,  # type: list[dict[str, Any]]
    ):
        self.collected_alerts.extend(alerts)

    def operate(self):
        self.collected_alerts.sort(key=lambda x: x.name)

        self._create_juggler_channels(self.collected_alerts)
        for alert in self.collected_alerts:
            self._upsert_alert(alert)

        self._cleanup_alerts(self.collected_alerts)

    def _upsert_alert(self, alert):
        """
        Insert new alert or update existing

        :param alert: alert object to be inserted or updated
        """
        self.active_alerts.add(alert.id)

        try:
            existing = self.client.get_alert_using_get(alert.id, alert.project_id)
        except solomon_client.rest.ApiException:
            if not self.is_dry_run:
                self.client.create_alert_using_post(alert, alert.project_id)
            logging.info("solomon: %s - [added]", alert.id)
        else:
            diff = {}
            self._compare_obj(existing.to_dict(), alert.to_dict(), diff)
            if diff:
                logging.warning(str(self._pretty_diff(diff, alert.id)))
                alert.version = existing.version
                if not self.is_dry_run:
                    self.client.update_alert_using_put(alert, alert.id, alert.project_id)
                logging.info("solomon: %s - [updated]", alert.id)

    def _pretty_diff(self, diff, alert_id):
        table = PrettyTable([alert_id + " alert field", "Existing value", "New value"])
        for key, (value1, value2) in diff.items():
            table.add_row([key, value1, value2])
        return table

    def _compare_obj(self, obj1, obj2, diff, path='', indexed_path=''):
        if type(obj1) != type(obj2):
            diff[indexed_path] = ("[TYPE IS %s] %s" % (type(obj1), obj1), "[TYPE IS %s] %s" % (type(obj2), obj2))
        elif isinstance(obj1, dict):
            self._compare_dict(obj1, obj2, diff, path, indexed_path)
        elif isinstance(obj1, list):
            if path in SORT_BEFORE_COMPARE_LISTS:
                obj1 = sorted(obj1)
                obj2 = sorted(obj2)
            self._compare_list(obj1, obj2, diff, path, indexed_path)
        else:
            if path not in IGNORE_ON_COMPARE_FIELDS and obj1 != obj2:
                diff[indexed_path] = (obj1, obj2)

    def _compare_list(self, obj1, obj2, diff, path='', indexed_path=''):
        if len(obj1) != len(obj2):
            diff[indexed_path] = (obj1, obj2)
        else:
            for i in range(len(obj1)):
                self._compare_obj(obj1[i], obj2[i], diff, path, indexed_path + ".%d" % i if indexed_path else str(i))

    def _compare_dict(self, obj1, obj2, diff, path='', indexed_path=''):
        all_fields = set(obj1.keys()).union(set(obj2.keys()))
        for key in all_fields:
            tmp_path = path + "." + key if path else key
            tmp_indexed_path = indexed_path + "." + key if indexed_path else key
            if tmp_path in IGNORE_ON_COMPARE_FIELDS:
                continue
            self._compare_obj(obj1.get(key), obj2.get(key), diff, tmp_path, tmp_indexed_path)

    def _cleanup_alerts(self, alerts):
        """
        Remove not longer existing alerts
        """
        if self.no_cleanup:
            logging.info('Solomon cleanup is skipped.')
            return

        project_ids = [alert.project_id for alert in alerts]
        for project_id in set(project_ids):
            for alert_item in self._get_maac_alert_list(project_id):
                if alert_item.id not in self.active_alerts:
                    if not self.is_dry_run:
                        self.client.delete_alert_using_delete(alert_item.id, alert_item.project_id)
                    logging.info("solomon: %s - [removed]", alert_item.id)

    def _create_juggler_channels(self, alerts):
        channel_ids = set(
            [(channel.id, alert.project_id, alert.name) for alert in alerts for channel in alert.channels])
        for channel_id, project_id, alert_name in channel_ids:
            try:
                self.client.get_notification_using_get(channel_id, project_id)
            except solomon_client.rest.ApiException:
                logging.info("Creating notification channel for: channel_id: %s, project_id: %s, alert_name: %s",
                             channel_id, project_id, alert_name)
                if not self.is_dry_run:
                    channel = solomon_client.NotificationChannel(
                        id=channel_id,
                        project_id=project_id,
                        name=channel_id,
                        method=solomon_client.Method(
                            juggler=solomon_client.Juggler(
                                host=channel_id,
                                description=MAAC_JUGGLER_EVENT_DESCRIPTION,
                            ),
                        ),
                        notify_about_statuses=["ALARM", "WARN", "OK", "ERROR"],
                    )
                    self.client.create_notification_using_post(channel, project_id)

    def _get_all_alert_list(
        self,
        project_id,  # type: str
        page_size,  # type: int
    ):  # type: (...) -> Generator[solomon_client.AlertListItem]
        if page_size <= 0:
            raise RuntimeError("Page size must be positive, got %s" % page_size)

        page_token = 0  # type: Optional[int]
        while page_token is not None:
            alert_page = self.client.list_alerts_by_project_using_get(
                project_id,
                page_size=page_size,
                page_token=page_token,
            )
            if alert_page and alert_page.items:
                for alert in alert_page.items:
                    yield alert
            page_token = getattr(alert_page, 'next_page_token', None)

    def _get_maac_alert_list(
        self,
        project_id,  # type: str
        page_size=25,  # type: int
    ):  # type: (...) -> list[solomon_client.AlertListItem]
        if page_size <= 0:
            raise RuntimeError("Page size must be positive, got %s" % page_size)

        maac_managed_alerts = [
            alert
            for alert in self._get_all_alert_list(project_id, page_size)
            if alert.name.startswith(MAAC_MANAGED_PREFIX)
        ]

        return maac_managed_alerts


def solomon_set_juggler_check_defaults(host, check, **kwargs):
    """
        Set defaults for solomon juggler check kwargs

        :param kwargs: kwargs to be prepared to pass
                       in juggler_sdk.Check constructor
        :param host: juggler host
        :param check: juggler check or service
        """
    project_id = kwargs["solomon"]["project_id"]

    tags = []
    for item in host.split('.'):
        tags.append((tags[-1] if len(tags) else 'a_solomon_prefix') + '_' + item)

    solomon_alert_id = create_solomon_alert_id(host, check)
    solomon_alert_url = 'https://solomon.yandex-team.ru/admin/projects/{}/alerts/{}'.format(
        project_id,
        solomon_alert_id
    )

    juggler_children = [juggler_sdk.Child(
        # Host is the same for all alerts.
        host="solomon_alert",
        # By default, raw events are sent with service=solomon_alert_id.
        service=solomon_alert_id,
        group_type='HOST'
    )]

    # added possibility to pass custom params for juggler children so that multialerts would work
    # kwargs["solomon"]["juggler_children"] = {"replace": bool, "children": list[juggler_sdk.Child]}
    juggler_children_add = kwargs["solomon"].get("juggler_children")
    if juggler_children_add:
        if juggler_children_add["replace"]:
            juggler_children = juggler_children_add["children"]
        else:
            juggler_children.extend(juggler_children_add["children"])

    kwargs.pop('solomon')
    kwargs.pop('children')

    kwargs.update({
        'refresh_time': 5,
        'ttl': 900,
        'children': juggler_children
    })

    if 'aggregator' not in kwargs.keys():
        kwargs['aggregator'] = 'logic_or'

    if 'aggregator_kwargs' not in kwargs.keys():
        kwargs.update({
            'aggregator_kwargs': {
                'unreach_service': [],
                'nodata_mode': 'force_ok',
                'unreach_mode': 'force_ok',
            }
        })

    if "tags" not in kwargs:
        kwargs["tags"] = []
    for tag in tags:
        kwargs["tags"].append(tag)
    if "meta" not in kwargs:
        kwargs["meta"] = {}
    kwargs["meta"]["solomon_alert_name"] = solomon_alert_id
    if "urls" not in kwargs["meta"]:
        kwargs["meta"]["urls"] = []
    kwargs["meta"]["urls"].append({
        'url': solomon_alert_url,
        'type': "solomon_alert",
        'title': u"Ссылка на алерт в Соломоне",
    })

    return kwargs


def solomon_gen_alert(host, check, state="ACTIVE", resolved_empty_policy="RESOLVED_EMPTY_DEFAULT",
                      group_by_labels=None, no_points_policy="NO_POINTS_DEFAULT",
                      annotations=None, description="", delay_secs=0, window_secs=60 * 2,
                      repeat_delay_secs=0, **kwargs):
    """Generate solomon alert object

    :param host: hostname of alert
                 (in terms of solomon it will be prefix for alert name)
    :param check: name of check
                  (postfix of alert name in solomon)
    :param state: State of current alert, only ACTIVE alerts will be periodically checked
    :resolved_empty_policy: policy for selectors that resolve to empty set of metrics
    :group_by_labels: List of label key that should be use to group sensors, each group
                it's separate sub alert that check independently from other group
    :no_points_policy: policy for timeseries without data points ["NO_POINTS_DEFAULT", "NO_POINTS_OK", "NO_POINTS_WARN",
                            "NO_POINTS_ALARM", "NO_POINTS_NO_DATA", "NO_POINTS_MANUAL", "UNRECOGNIZED"]
    :annotations: The annotations of this AlertEvaluationStatusDto
    :description: Alert description
    :delay_secs: Alert delay in seconds
    :param window_secs: Alert window width in seconds
    :return: solomon_client.Alert()
    """
    kwargs.update({
        'id': create_solomon_alert_id(host, check),
    })
    kwargs["name"] = MAAC_MANAGED_PREFIX + kwargs["id"]
    kwargs["state"] = state
    kwargs["window_secs"] = window_secs
    kwargs["period_millis"] = window_secs * 1000
    kwargs["resolved_empty_policy"] = resolved_empty_policy
    kwargs["group_by_labels"] = group_by_labels
    kwargs["delay_seconds"] = delay_secs  # Deprecated, but it makes false diff BALANCEDUTY-2776
    kwargs["no_points_policy"] = no_points_policy
    kwargs["annotations"] = annotations or {}
    kwargs["description"] = description
    kwargs["delay_secs"] = delay_secs
    if "channels" not in kwargs:
        kwargs["channels"] = [
            solomon_client.AssociatedChannel(
                id="solomon_alert",
                config=solomon_client.ChannelConfig(
                    notify_about_statuses=["ALARM", "WARN", "OK", "ERROR"],
                    repeat_delay_secs=repeat_delay_secs,
                )
            )
        ]
    type_args = {
        "threshold": None,
        "expression": None
    }
    selectors = kwargs.pop("selectors", None)
    if kwargs.get("threshold"):
        type_args["threshold"] = get_threshold(selectors, **kwargs.pop("threshold"))
    elif kwargs.get("expression"):
        type_args["expression"] = get_threshold_expression(selectors, **kwargs.pop("expression"))
    elif kwargs.get("expression-timing"):
        type_args["expression"] = get_timing_expression(selectors, **kwargs.pop("expression-timing"))
    elif kwargs.get("expression-absence"):
        type_args["expression"] = get_absence_expression(selectors, **kwargs.pop("expression-absence"))
    elif kwargs.get("expression-presence"):
        type_args["expression"] = get_presence_expression(selectors, **kwargs.pop("expression-presence"))
    elif kwargs.get("expression-custom"):
        # ignores pre-built selectors as they are not flexible
        type_args["expression"] = parse_expression_custom(kwargs.pop("expression-custom"))

    kwargs["type"] = solomon_client.Type(
        **type_args
    )
    if "juggler_children" in kwargs:
        kwargs.pop("juggler_children")
    return solomon_client.Alert(**kwargs)


def get_threshold(selectors, warn_threshold, crit_threshold):
    return solomon_client.Threshold(
        time_aggregation="AT_LEAST_ONE",
        predicate="GTE",
        threshold=crit_threshold,
        selectors=selectors_to_string(selectors),
        predicate_rules=[
            solomon_client.PredicateRule(
                threshold_type="AT_LEAST_ONE",
                comparison="GTE",
                threshold=crit_threshold,
                target_status="ALARM"
            ),
            solomon_client.PredicateRule(
                threshold_type="AT_LEAST_ONE",
                comparison="GTE",
                threshold=warn_threshold,
                target_status="WARN"
            )
        ],
        transformations=None,
    )


def get_absence_expression(selectors, warn_threshold, crit_threshold, fn="sum", total_fn=None):
    return _get_absence_presence_expression(selectors, warn_threshold, crit_threshold, "<", fn, total_fn=total_fn)


def get_presence_expression(selectors, warn_threshold, crit_threshold, fn="sum", total_fn=None):
    return _get_absence_presence_expression(selectors, warn_threshold, crit_threshold, ">", fn, total_fn=total_fn)


def _get_absence_presence_expression(selectors,
                                     warn_threshold, crit_threshold,
                                     comparison, fn="sum", total_fn=None):
    program_template = "let data = group_lines('{fn}', {selectors});\n" \
                       "let total = {total_fn}(data);\n" \
                       "alarm_if(total {comparison} {crit_threshold});\n" \
                       "warn_if(total {comparison} {warn_threshold});\n"

    if total_fn is None:
        total_fn = fn

    return solomon_client.TExpression(
        program=program_template.format(
            fn=fn,
            total_fn=total_fn,
            selectors=selectors_to_string(selectors),
            warn_threshold=warn_threshold,
            crit_threshold=crit_threshold,
            comparison=comparison
        ),
        check_expression=""
    )


def get_timing_expression(selectors, percentile, warn_timing_threshold, crit_timing_threshold, fn="last"):
    program_template = "let timing = histogram_percentile({percentile}, 'bin', {selectors});\n" \
                       "let percentile = {fn}(timing);\n" \
                       "let alarm_limit = {crit_timing_threshold};\n" \
                       "let warn_limit = {warn_timing_threshold};\n" \
                       "alarm_if(percentile >= alarm_limit);\n" \
                       "warn_if(percentile >= warn_limit);\n"

    return solomon_client.TExpression(
        program=program_template.format(
            selectors=selectors_to_string(selectors),
            percentile=percentile,
            warn_timing_threshold=warn_timing_threshold,
            crit_timing_threshold=crit_timing_threshold,
            fn=fn,
        ),
        check_expression=""
    )


def get_threshold_expression(selectors, limit_selectors, warn_percent_threshold, crit_percent_threshold, fn=""):
    program_template = "let used = {fn}({selectors});\n" \
                       "let limit = {fn}({limit_selectors});\n" \
                       "let percent = last(used / limit);\n" \
                       "alarm_if(percent >= {crit_percent_threshold});\n" \
                       "warn_if(percent >= {warn_percent_threshold});\n"
    return solomon_client.TExpression(
        program=program_template.format(
            fn=fn,
            selectors=selectors_to_string(selectors),
            limit_selectors=selectors_to_string(limit_selectors),
            warn_percent_threshold=warn_percent_threshold,
            crit_percent_threshold=crit_percent_threshold
        ),
        check_expression=""
    )


def solomon_gen_alerts_for_host(host, checks):
    """Generate solomon alerts list for specified host

    :param host: hostname of alerts
    :param checks: list of checks

    :return: [solomon_client.Alert(), solomon_client.Alert(), ...]
    """

    solomon_alerts_for_host = []

    for check, kwargs in checks.items():
        if 'solomon' in kwargs:
            solomon_alert_kwargs = kwargs['solomon']
            solomon_alerts_for_host.append(
                solomon_gen_alert(host, check, **solomon_alert_kwargs)
            )

    return solomon_alerts_for_host


def solomon_threshold_monitoring(project_id, selectors, warn_threshold, crit_threshold, description=""):
    return {
        "selectors": selectors,
        "threshold": {
            "warn_threshold": warn_threshold,
            "crit_threshold": crit_threshold,
        },
        "project_id": project_id,
        "description": description
    }


def solomon_threshold_expression_monitoring(project_id, selectors, limit_selectors, warn_percent_threshold,
                                            crit_percent_threshold, annotations=None, description="", fn="",
                                            **kwargs):
    kwargs.update({
        "selectors": selectors,
        "expression": {
            "limit_selectors": limit_selectors,
            "warn_percent_threshold": warn_percent_threshold,
            "crit_percent_threshold": crit_percent_threshold,
            "fn": fn,
        },
        "project_id": project_id,
        "description": description,
        "annotations": annotations,

    })
    return kwargs


def solomon_timing_expression_monitoring(project_id, selectors, percentile, warn_timing_threshold,
                                         crit_timing_threshold, annotations=None, description="",
                                         window_secs=60 * 2, fn="last"):
    return {
        "selectors": selectors,
        "window_secs": window_secs,
        "expression-timing": {
            "percentile": percentile,
            "warn_timing_threshold": warn_timing_threshold,
            "crit_timing_threshold": crit_timing_threshold,
            "fn": fn,
        },
        "project_id": project_id,
        "description": description,
        "annotations": annotations,
    }


def solomon_absence_expression_monitoring(project_id, selectors,
                                          warn_threshold, crit_threshold,
                                          annotations=None, description="",
                                          window_secs=60 * 2):
    """Для обработки ситуаций, когда какой-то сигнал перестает поступать или падает ниже уровня"""
    return _solomon_absence_presence_monitoring(
        project_id, selectors,
        warn_threshold, crit_threshold,
        annotations, description,
        window_secs,
        expression_key="expression-absence"
    )


def solomon_presence_expression_monitoring(project_id, selectors,
                                           warn_threshold=0, crit_threshold=0,
                                           annotations=None, description="",
                                           window_secs=60 * 2, fn="sum", total_fn=None):
    """Для обработки ситуаций, когда какой-то сигнал ошибок начинает поступать.
    В отличие от других сигналов, здесь вполне может быть ситуация,
    когда метрики вообще нет. Чтобы соломон не падал в таком случае, используется
    специальная обработка этого случая."""
    return _solomon_absence_presence_monitoring(
        project_id, selectors,
        warn_threshold, crit_threshold,
        annotations, description,
        window_secs,
        expression_key="expression-presence",
        resolved_empty_policy="RESOLVED_EMPTY_OK",
        fn=fn,
        total_fn=total_fn,
    )


def _solomon_absence_presence_monitoring(project_id, selectors,
                                         warn_threshold, crit_threshold,
                                         annotations, description,
                                         window_secs, expression_key,
                                         resolved_empty_policy="RESOLVED_EMPTY_DEFAULT",
                                         fn="sum", total_fn=None):
    return {
        "selectors": selectors,
        "window_secs": window_secs,
        expression_key: {
            "warn_threshold": warn_threshold,
            "crit_threshold": crit_threshold,
            "fn": fn,
            "total_fn": total_fn
        },
        "project_id": project_id,
        "description": description,
        "annotations": annotations,
        "resolved_empty_policy": resolved_empty_policy,
    }


def parse_expression_custom(
    program_str,  # type: str
):  # type: (...) -> solomon_client.TExpression
    return solomon_client.TExpression(
        program=program_str,
        check_expression="",
    )


def solomon_expression_custom(
    program_str,  # type: str
    project_id,  # type: str
    **kwargs
):
    """
    Custom expression for solomon
    kwargs may contain additional values to be passed to solomon_gen_alert
    """
    kwargs.update({
        'expression-custom': program_str,
        'project_id': project_id,
    })
    return kwargs
