import abc


class InputMixin(object):
    __metaclass__ = abc.ABCMeta

    def __init__(
        self,
        **kwargs  # type: str
    ):
        self._input_mapping = {}

        expected_arguments = self.expected_inputs.copy()
        for input_name, mapped_name in kwargs.items():
            if input_name in expected_arguments:
                expected_arguments.remove(input_name)
                self._input_mapping[input_name] = mapped_name

        if len(expected_arguments) > 0:
            raise ValueError('Did not map the following inputs: %s' % expected_arguments)

    @abc.abstractproperty
    def expected_inputs(self):  # type: () -> set[str]
        """
        Set of inputs the mixin expects to receive for proper operations
        """
        pass

    def in_var(self, item):
        return self._input_mapping[item]


class OutputMixin(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractproperty
    def output_names(self):  # type: () -> set[str]
        """
        Set of output names the mixin exposes.
        Currently it is just a convention
        """
        pass

    def _get_output_var_name_optional(self, item):
        if item in self.output_names:
            return item

    def __getattr__(self, item):
        expression_prefix = "expr_"
        format_as_expr = item.startswith(expression_prefix)
        output_var_key = item if not format_as_expr else item[len(expression_prefix):]

        output_var = self._get_output_var_name_optional(output_var_key)

        if output_var:
            return output_var if not format_as_expr else "{{{{expression.{0}}}}}".format(output_var)

        raise AttributeError("Unable to get attr: %s" % item)


class ProgramMixin(object):
    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def program(self):  # type: () -> str
        """
        Return a solomon program in form of string
        """
        pass


class AlertMixin(InputMixin, ProgramMixin):
    __metaclass__ = abc.ABCMeta

    def ok_if(self):  # type: () -> str
        """
        A string condition for ok_if check
        """
        return ""

    def warn_if(self):  # type: () -> str
        """
        A string condition for warn_if check
        """
        return ""

    def alarm_if(self):  # type: () -> str
        """
        A string condition for alarm_if check
        """
        return ""

    def program(self):  # type: () -> str
        checks = []
        for check_func_name in ['alarm_if', 'warn_if', 'ok_if']:
            check_condition = getattr(self, check_func_name)()
            if check_condition:
                checks.append(
                    '{check_func_name}({condition});'.format(
                        check_func_name=check_func_name,
                        condition=check_condition,
                    )
                )
        if not checks:
            raise RuntimeError("No checks were provided")

        return '\n'.join(checks)


class TimeModifier(OutputMixin, ProgramMixin):

    def __init__(
        self,
        utc_start,  # type: int
        utc_end,  # type: int
        in_range_modifier,  # type: int
        out_range_modifier,  # type: int
    ):
        self.utc_start = utc_start
        self.utc_end = utc_end
        self.in_range_modifier = in_range_modifier
        self.out_range_modifier = out_range_modifier

    @property
    def output_names(self):
        return {'time_modifier'}

    def program(self):
        return '\n'.join([
            'let is_in_range = count(filter_by_time(constant_line(0), "[{start}h-{end}h]")) < 2;'.format(
                start=self.utc_start,
                end=self.utc_end,
            ),
            'let time_modifier = is_in_range ? {in_range} : {out_range};'.format(
                in_range=self.in_range_modifier,
                out_range=self.out_range_modifier,
            ),
        ])


class QueueAlertBlock(AlertMixin):
    def __init__(self, alarm_size, **kwargs):
        super(QueueAlertBlock, self).__init__(**kwargs)
        self.alarm_size = alarm_size

    @property
    def expected_inputs(self):
        return {'queue_size'}

    def alarm_if(self):  # type: () -> str
        return "{queue_size} >= {size}".format(
            queue_size=self.in_var('queue_size'),
            size=self.alarm_size,
        )

    def ok_if(self):  # type: () -> str
        return "{queue_size} < {size}".format(
            queue_size=self.in_var('queue_size'),
            size=self.alarm_size,
        )


class UpperThresholdAlertBlock(AlertMixin):
    def __init__(self, alarm_limit, warn_limit=None, **kwargs):
        super(UpperThresholdAlertBlock, self).__init__(**kwargs)
        if warn_limit is not None and alarm_limit is not None:
            if warn_limit > alarm_limit:
                raise RuntimeError("there is no point in setting warn_limit higher than alarm_limit")
        self.alarm_limit = alarm_limit
        self.warn_limit = warn_limit

    @property
    def expected_inputs(self):
        return {'decision_value'}

    def alarm_if(self):  # type: () -> str
        if self.alarm_limit is None:
            return ""

        return "{decision_value} > {limit}".format(
            decision_value=self.in_var('decision_value'),
            limit=self.alarm_limit,
        )

    def warn_if(self):  # type: () -> str
        if self.warn_limit is None:
            return ""
        if self.alarm_limit is None:
            return "{decision_value} > {limit}".format(
                decision_value=self.in_var('decision_value'),
                limit=self.warn_limit,
            )

        return "({decision_value} > {limit}) && ({decision_value} <= {alarm_limit})".format(
            decision_value=self.in_var('decision_value'),
            limit=self.warn_limit,
            alarm_limit=self.alarm_limit,
        )


class LowerThresholdAlertBlock(AlertMixin):
    def __init__(self, alarm_limit, warn_limit=None, **kwargs):
        super(LowerThresholdAlertBlock, self).__init__(**kwargs)
        if warn_limit is not None and alarm_limit is not None:
            if alarm_limit > warn_limit:
                raise RuntimeError("there is no point in setting warn_limit lower than alarm_limit")
        self.alarm_limit = alarm_limit
        self.warn_limit = warn_limit

    @property
    def expected_inputs(self):
        return {'decision_value'}

    def alarm_if(self):  # type: () -> str
        if self.alarm_limit is None:
            return ""

        return "{decision_value} < {limit}".format(
            decision_value=self.in_var('decision_value'),
            limit=self.alarm_limit,
        )

    def warn_if(self):  # type: () -> str
        if self.warn_limit is None:
            return ""
        if self.alarm_limit is None:
            return "{decision_value} < {limit}".format(
                decision_value=self.in_var('decision_value'),
                limit=self.warn_limit,
            )

        return "({decision_value} < {limit}) && ({decision_value} >= {alarm_limit})".format(
            decision_value=self.in_var('decision_value'),
            limit=self.warn_limit,
            alarm_limit=self.alarm_limit,
        )


class AggregatedSelector(OutputMixin, ProgramMixin):
    def __init__(
        self,
        var_name,  # type: str
        selector,  # type: str
        aggregation_func,  # type: str
    ):
        self.var_name = var_name
        self.selector = selector
        self.aggregation_func = aggregation_func

    @property
    def aggregated_var_name(self):
        return "{name}_{aggr_func}".format(
            name=self.var_name,
            aggr_func=self.aggregation_func,
        )

    def program(self):  # type: () -> str
        return "\n".join([
            "let {name} = {selector};".format(
                name=self.var_name,
                selector=self.selector
            ),
            "let {aggregated_var_name} = {aggr_func}({name});".format(
                aggregated_var_name=self.aggregated_var_name,
                aggr_func=self.aggregation_func,
                name=self.var_name,
            ),
        ])

    def output_names(self):  # type: () -> set[str]
        return {self.aggregated_var_name}


def compose_program(
    *blocks  # type: ProgramMixin
):  # type: (...) -> str
    return "\n\n".join([
        "// Block: {blockname}\n{program}".format(
            blockname=type(program_block).__name__,
            program=program_block.program()
        )
        for program_block in blocks
    ])
