# -*- coding: utf-8 -*-
"""
Common functions and classes for managing alerts in Yasm (Golovan)

It is using Yasm alert api https://wiki.yandex-team.ru/golovan/alerts/api/

Api is being used through library:
    https://github.yandex-team.ru/ps-admin/python-yasm-alert
"""

import logging

import juggler_sdk
import yasm_alert
from urllib import quote

logger = logging.getLogger()  # pylint: disable=invalid-name


class YasmAlertApiExtendedException(Exception):
    """
    Common exception for YasmAlertApiExtended
    """


class YasmAlertApiExtended(yasm_alert.DefaultApi):
    """
    Exteded yasm_alert.DefaultApi with following extensions:
    * Added debug toggle
    * Binded to host (in terms of juggler). Constructor gets list of alert of
      this hosts from yasm and puts it to :param host:
    * Has method for cleaning up no longer existing alert

    :param api_client: yasm alert api client instance
    :param host: host to get alert list for
    :param debug: if True debug is enabled
    """

    def __init__(self, host=None, debug=False):
        # Enable debug
        api_configuration = yasm_alert.Configuration()
        api_configuration.debug = debug
        api_configuration.logger["package_logger"].propagate = False
        api_configuration.logger["urllib3_logger"].propagate = False

        api_client = yasm_alert.ApiClient(configuration=api_configuration)

        # Run constructor from parent class
        super(YasmAlertApiExtended, self).__init__(api_client=api_client)

        if host is None:
            raise YasmAlertApiExtendedException('Argument host must be defined')
        else:
            self.host = host

        self._get_alert_list()

    def _get_alert_list(self):
        list_alerts_result = self.list_alerts(
            name_prefix=self.host, with_checks=True
        )

        self.alerts = {}

        # Create lists for tracking alerts that no longer exists
        self.alert_names_active_list = []
        self.alert_names_remote_list = []

        if list_alerts_result is not None:
            for alert in list_alerts_result.response.result:
                self.alerts[alert.name] = alert
                self.alert_names_remote_list.append(alert.name)

    def upsert_alert(self, alert):
        """
        Insert new alert or update existing

        :param alert: alert object to be inserted or updated
        """

        # juggler_check_host = self.host
        # juggler_check_service = alert.name[len(juggler_check_host):].lstrip('.')

        if alert.name in self.alerts:
            self.alert_names_active_list.append(alert.name)

        try:
            get_alert_result = self.get_alert(
                name=alert.name, with_checks='true'
            )

            existing_alert = get_alert_result.response

            existing_alert_cleaned = existing_alert.to_dict()
            existing_alert_cleaned.pop('juggler_check')
            alert_cleaned = alert.to_dict()
            alert_cleaned.pop('juggler_check')

            if existing_alert_cleaned == alert_cleaned:
                logger.debug(
                    'yasm: %s - [not changed]', alert.name
                )
            else:
                self.update_alert(alert.name, alert)
                logger.info('yasm: %s - [updated]', alert.name)
                logger.info('yasm: old alert: %s', existing_alert_cleaned)
                logger.info('yasm: new alert: %s', alert_cleaned)
        except yasm_alert.rest.ApiException as exception:
            if exception.status == 404:
                self.create_alert(alert)

                # pylint: disable=fixme
                # FIXME: Do not bind check by now it is breaking removing alert.
                # In order to properly remove alert/unbind check from alert
                # juggler check must exist. We cannot leave check in juggler
                # because juggler_sdk cleaning it up.

                # self.bind_check(
                #    alert_name=alert.name,
                #    check_host=juggler_check_host,
                #    check_service=juggler_check_service
                # )

                logger.info('yasm: %s - [added]', alert.name)
            else:
                raise

    def cleanup_alerts(self):
        """
        Remove not longer existing alerts
        """

        for alert_name in self.alert_names_remote_list:
            if alert_name not in self.alert_names_active_list:
                # pylint: disable=fixme
                # FIXME: see cleanup for comment expanation
                # self.unbind_check(alert_name)

                self.delete_alert(alert_name)

                logger.warn('yasm: %s - [removed]', alert_name)


def yasm_set_juggler_check_defaults(kwargs, host, check):
    """
    Set defaults for yasm juggler check kwargs

    :param kwargs: kwargs to be prepared to pass
                   in juggler_sdk.Check constructor
    :param host: juggler host
    :param check: juggler check or service
    """

    yasm_settings = kwargs.get('yasm')
    yasm_chart_uri = "signals={};".format(yasm_settings.get("signal"))
    yasm_chart_uri += "hosts={};".format(','.join(yasm_settings.get("mgroups")))
    for tag, values in yasm_settings.get("tags").to_dict().items():
        value = values
        if isinstance(values, list) or isinstance(values, set) or isinstance(values, tuple):
            value = ','.join(values)
        if value is not None:
            yasm_chart_uri += "{}={};".format(tag, value)

    yasm_urls = [
        {
            'url': "https://s.yasm.yandex-team.ru/chart/" + quote(yasm_chart_uri),
            'type': "screenshot_url",
            'title': u"Снапшот графика алерта",
        },
        {
            'url': "https://yasm.yandex-team.ru/chart/" + quote(yasm_chart_uri),
            'type': "graph_url",
            'title': u"График алерта",
        }
    ]

    kwargs.pop('yasm')

    tags = []
    for item in host.split('.'):
        tags.append((tags[-1] if len(tags) else 'a_yasm_prefix') + '_' + item)

    yasm_alert_name = host + '.' + check
    yasm_alert_url = 'https://yasm.yandex-team.ru/alert/' + yasm_alert_name
    yasm_child_service = yasm_alert_name

    kwargs.pop('children')

    kwargs.update({'children': [juggler_sdk.Child('yasm_alert', yasm_child_service, group_type='HOST')]})

    if 'refresh_time' not in kwargs:
        kwargs['refresh_time'] = 5

    if 'ttl' not in kwargs:
        kwargs['ttl'] = 900

    if 'aggregator' not in kwargs:
        kwargs['aggregator'] = 'logic_or'

    if 'aggregator_kwargs' not in kwargs:
        kwargs.update({
            'aggregator_kwargs': {
                'unreach_service': [],
                'nodata_mode': 'force_ok',
                'unreach_mode': 'force_ok',
            }
        })

    alert_url = {
        'url': yasm_alert_url,
        'type': "yasm_alert",
        'title': u"Ссылка на алерт в Головане",
    }

    yasm_urls.append(alert_url)

    if "tags" in kwargs:
        for tag in tags:
            kwargs["tags"].append(tag)
    else:
        kwargs["tags"] = tags

    if "meta" in kwargs:
        kwargs["meta"]["yasm_alert_name"] = yasm_alert_name
        if "urls" in kwargs["meta"]:
            kwargs["meta"]["urls"].extend(yasm_urls)
        else:
            kwargs["meta"]["urls"] = yasm_urls
    else:
        kwargs["meta"] = {
            "yasm_alert_name": yasm_alert_name,
            "urls": yasm_urls
        }

    return kwargs


def yasm_gen_alert(host, check, **kwargs):
    """Generate yasm alert object


    :param host: hostname of alert
                 (in terms of yasm it will be prefix for alert name)
    :param check: name of check
                  (postfix of alert name in yasm)

    :return: yasm_alert.Alert()
    """

    kwargs.update({
        'name': host + '.' + check,
    })

    return yasm_alert.Alert(**kwargs)


def __extract_subchecks(check):
    subchecks = {}
    if 'tags' in check.keys():
        if 'subchecks' in check['tags']:
            check['tags'].remove('subchecks')
            subchecks = check.pop('subchecks')
    return subchecks


def __prepare_yasm_alert(host, check, kwargs):
    if 'yasm' in kwargs:
        yasm_alert_kwargs = kwargs['yasm']
        yasm_alert_kwargs['juggler_check'] = None
        return [yasm_gen_alert(host, check, **yasm_alert_kwargs)]
    return []


def yasm_gen_alerts_for_host(host, checks):
    """Generate yasm alerts list for specified host

    :param host: hostname of alerts
    :param check: list of checks

    :retrurn: [yasm_alert.Alert(), yasm_alert.Alert(), ...]
    """

    yasm_alerts_for_host = []

    for check, kwargs in checks.items():
        subchecks = __extract_subchecks(kwargs)

        for _host, _check in subchecks.iteritems():
            for _service, _kwargs in _check.items():
                yasm_alerts_for_host += __prepare_yasm_alert(_host, _service, _kwargs)

        if 'yasm' in kwargs:
            yasm_alerts_for_host += __prepare_yasm_alert(host, check, kwargs)

    return yasm_alerts_for_host


def yasm_threshold_alert_func(type, ctype, func, signals, tier, warn_threshold, crit_threshold, trend_type=None, group="ASEARCH"):
    """
    Generate threshold alert with custom func.

    :param type: type of the signal.
    :param ctype: ctype of the signal.
    :param func: function to use as signal or 'signal' to use signals as one signal.
    :param signals: signals.
    :param tier: tier of the base.
    :param warn_threshold: beginning of the warning range. Warning range is between warn and crit thresholds.
    :param crit_threshold: beginning of the critical range.
    :param trend_type: type of the trend. One of ['up', 'down', 'None']. None means default threshold.
    :param group: name of the group in YASM.
    """

    if func == 'signal':
        signal = signals
    else:
        signal = ('%s(' % func) + ','.join(signals) + ')'

    if not isinstance(tier, list):
        tier = [tier]

    if not isinstance(group, list):
        group = [group]

    if warn_threshold <= crit_threshold:
        warn_zone = [warn_threshold, crit_threshold]
        crit_zone = [crit_threshold, None]
    else:
        crit_zone = [None, crit_threshold]
        warn_zone = [crit_threshold, warn_threshold]

    return {
        "yasm": {
            "signal": signal,
            "tags": yasm_alert.Tags(
                itype=[type],
                ctype=ctype,
                tier=tier,
            ),
            "warn": warn_zone,
            "crit": crit_zone,
            "mgroups": group,
            "trend": trend_type,
        }
    }


def yasm_deploy_threshold_alert(signal_name, namespace, warn_threshold, crit_threshold, trend_type=None, geo=None, ctype=None, itype='deploy'):
    """
    Generate YASM alert with thresholds.

            warn       crit
    |---OK---|---Warn---|---Crit---|
   -∞      ................        ∞

    :param signal_name: name of the signal in YASM
    :param namespace: signal project. For Deploy: <stage-id>-<deploy-unit-id>-<workload-id>
    :param warn_threshold: beginning of the warning range. Warning range is between warn and crit thresholds.
    :param crit_threshold: beginning of the critical range.
    :param trend_type: type of the trend. One of ['up', 'down', 'None']. None means default threshold.
    """

    if warn_threshold <= crit_threshold:
        warn_zone = [warn_threshold, crit_threshold]
        crit_zone = [crit_threshold, None]
    else:
        crit_zone = [None, crit_threshold]
        warn_zone = [crit_threshold, warn_threshold]

    assert isinstance(itype, str)
    assert isinstance(namespace, str)
    if geo is not None:
        assert isinstance(geo, str)
        geo = [geo]

    if ctype is not None:
        assert isinstance(ctype, str)
        ctype = [ctype]

    return {
        "yasm": {
            "signal": signal_name,
            "tags": yasm_alert.Tags(
                itype=[itype],
                prj=[namespace],
                geo=geo,
                ctype=ctype
            ),
            "warn": warn_zone,
            "crit": crit_zone,
            "mgroups": ["ASEARCH"],
            "trend": trend_type,
        }
    }


def yasm_deploy_threshold_alert_func(func, signals, namespace, warn_threshold, crit_threshold, trend_type=None):
    signal = ('%s(' % func) + ','.join(signals) + ')'
    return yasm_deploy_threshold_alert(signal, namespace, warn_threshold, crit_threshold, trend_type)


def yasm_con_threshold_alert_func(type, ctype, func, signals, tier, warn_threshold, crit_threshold, trend_type=None):
    return yasm_threshold_alert_func(type, ctype, func, signals, tier, warn_threshold, crit_threshold, trend_type, group="CON")
