import logging

import requests
from prettytable import PrettyTable


logger = logging.getLogger()  # pylint: disable=invalid-name


def get_key(host):
    return "maac.{}".format(host)


class AlertTemplate(object):
    def __init__(self, key, content, owners=[], abc="", is_periodic=True):
        self.key = key
        self.owners = owners
        self.abc = abc
        self.content = content
        self.is_periodic = is_periodic

    def to_dict(self):
        return {
            "key": self.key,
            "owners": self.owners,
            "abc": self.abc,
            "content": self.content,
            "is_periodic": self.is_periodic,
        }

    @staticmethod
    def from_dict(data):
        return AlertTemplate(
            key=data["key"],
            content=data["content"],
            owners=data.get("owners", []),
            abc=data.get("abc", ""),
            is_periodic=data.get("is_periodic", True),
        )

    @staticmethod
    def create_for_host(host, data):
        data["key"] = get_key(host)
        return AlertTemplate.from_dict(data)


class YasmAlertTemplateAPI(object):
    # https://wiki.yandex-team.ru/golovan/userdocs/templatium/alerts/api/?from=%2Fgolovan%2Fuserdocs%2Ftemplatium%2Falerts-api%2F

    BASE_URL = "https://yasm.yandex-team.ru/srvambry/tmpl/alerts"

    def create(self, alert):
        url = self.BASE_URL + "/create"
        resp = requests.post(url, json=alert.to_dict())
        resp.raise_for_status()

    def update(self, alert):
        url = self.BASE_URL + "/update?key={}".format(alert.key)
        resp = requests.post(url, json=alert.to_dict())
        resp.raise_for_status()

    def get(self, key):
        url = self.BASE_URL + "/get?key={}".format(key)
        resp = requests.get(url)
        if resp.status_code == 404:
            return None
        else:
            resp.raise_for_status()
        return AlertTemplate.from_dict(resp.json()["response"])

    def delete(self, key):
        url = self.BASE_URL + "/delete?key={}".format(key)
        resp = requests.post(url)
        resp.raise_for_status()

    def apply(self, key):
        url = self.BASE_URL + "/apply/{}".format(key)
        resp = requests.post(url)
        resp.raise_for_status()


def diff_dicts(dict1, dict2):
    diff = {}
    keys = set(dict1.keys()).union(set(dict2.keys()))
    for k in sorted(list(keys)):
        if k not in dict1:
            diff[k] = (None, dict2[k])
        elif k not in dict2:
            diff[k] = (dict1[k], None)
        elif dict1[k] != dict2[k]:
            diff[k] = (dict1[k], dict2[k])
    return diff


def upsert(host, new_template, dry_run=False):
    api = YasmAlertTemplateAPI()
    old_template = api.get(get_key(host))

    if old_template is None:
        if not dry_run:
            api.create(new_template)
        logger.info("yasm: template %s - [added]", new_template.key)
        return

    raw_new_template = new_template.to_dict()
    raw_old_template = old_template.to_dict()
    if raw_new_template == raw_old_template:
        logger.debug("yasm: template %s - [not changed]", new_template.key)
        return

    if not dry_run:
        api.update(new_template)
        api.apply(new_template.key)

    table_fields = [
        new_template.key + " alert template field",
        "Existing value",
        "New value",
    ]
    table = PrettyTable(table_fields)
    diff = diff_dicts(raw_old_template, raw_new_template)
    for key, (value1, value2) in diff.items():
        table.add_row([key, value1, value2])

    logging.warning(str(table))
    logger.info("yasm: template %s - [updated]", new_template.key)
