import click
import requests
from datetime import datetime
from pytz import timezone
import json
from requests.packages.urllib3.exceptions import InsecureRequestWarning
from pipelines_api.launcher import PipelineLauncher, buildLaunchCommand
from pipelines_api.pipes import pipes

from pipelines_cli.settings import SETTINGS

# FIXME(iandreyev) Disabled SNI warnings, until migrate on Ubuntu 16.04
requests.packages.urllib3.disable_warnings(InsecureRequestWarning)


@click.group()
@click.option('--env', default='production', type=click.Choice(['testing', 'production']),
              help='Timeline API server')
@click.option('--dry', is_flag=True, help='Dry run, without sending')
@click.option('--epoch', is_flag=True, help='Use unix epoch time for start, end')
@click.pass_context
def cli(ctx, env, dry, epoch):
    """
    \b
    pipelines-cli --env testing launch oebs other title="This a test event" start="2017-05-26 12:34:56+03:00" end="$(date --rfc-3339=seconds)"
    pipelines-cli start oebs local_patch finish 5bec3310207a5d14b1b06de4"

    """
    ctx.obj['env'] = env
    ctx.obj['dry'] = dry
    ctx.obj['epoch'] = epoch


def _get_launcher(ctx):
    return PipelineLauncher(SETTINGS[ctx.obj['env']]['host'])


def parse_args(args):
    _dict = {}
    for item in args:
        _dict.update([item.split('=')])
    return _dict


@cli.command()
@click.argument('project')
@click.argument('pipeline')
@click.argument('fields', nargs=-1, type=click.UNPROCESSED)
@click.pass_context
def launch(ctx, project, pipeline, fields):
    """
    \b
    Launch pipeline from project
    \b
    optional kwargs-style [RESOURCE fields]:
    """
    builder = _check_pipeline(project, pipeline).get_builder()
    resource = builder(parse_args(fields), ctx.obj['epoch']).build().as_resource()
    launcher = _get_launcher(ctx)
    if ctx.obj['dry']:
        print(resource)
    else:
        result = launcher.launch(project, pipeline, buildLaunchCommand([], resource))
        try:
            decoded = json.loads(result)
            ids = decoded["pipeLaunchIds"]
            if len(ids) == 1:
                print u"launched as {}".format(ids[0])
            else:
                print u"More than one id, check latest {}".format(ids)
        except Exception as e:
            print result
            print u"Cannot decode json {}".format(e)


@cli.command()
@click.argument('project')
@click.argument('pipeline')
@click.argument('job')
@click.argument('launch_id')
@click.pass_context
def start(ctx, project, pipeline, job, launch_id):
    """
       \b
       Launch job in already started project
       \b
       """
    _check_job(project, pipeline, job)
    launcher = _get_launcher(ctx)
    if ctx.obj['dry']:
        print u"Dry run: start job [{0}] at pipe [{1}] in project [{2}]".format(job, pipeline, job)
    else:
        result = launcher.launch_job(launch_id, job)
        try:
            decoded = json.loads(result)

            for job in decoded["jobs"]:
                print u"ID: {0}, attempt #{1}, status: {2}, started {3}".format(
                    job['id'],
                    job['launchNumber'],
                    job['status'],
                    datetime.fromtimestamp(job['startDate']/1000.0, timezone("Europe/Moscow"))
                )
        except Exception as e:
            print e
            print result


def _check_pipeline(project, pipeline):
    if project in pipes.keys():
        if pipeline in pipes[project].pipelines:
            return pipes[project].arguments[pipeline]
    raise NotImplementedError(
        u"Cannot launch pipeline {0} from project {1}, use web-interface".format(project, pipeline)
    )


def _check_job(project, pipeline, job):
    if project in pipes.keys():
        if pipeline in pipes[project].pipelines:
            if job in pipes[project].jobs[pipeline]:
                return

    raise NotImplementedError(
        u"Cannot launch job {0} in pipeline {1} from project {2}, use web-interface".format(job, project, pipeline)
    )


def main():
    cli(obj={})
