import calendar
import datetime
import getpass
import socket
import uuid

import pytz
import market.pylibrary.yenv as yenv
from dateutil import parser
from timeline_events import EventStatus, Event, MicroEvent


class Builder():
    def __init__(self, args, micro):
        self.__builder = self.__MicroEventBuilder() if micro else self.__EventBuilder()
        self.__builder.update(args)
        self.__builder.update_time(args)
        self.__builder.update_tags(args)
        self.__builder.update_status(args)

    def build(self, args=None):
        return self.__builder.build(args)

    class _AbstractBuilder():
        def __init__(self):
            self._args = {
                'tags': [],
                'type': 'cli',
                'project': 'PAYSYS',
                'source': socket.getfqdn(),
                'text': None
            }

        def update_status(self, args):
            status = args.get('status', 'INFO')
            return self.update({'status': EventStatus.Value(status)})

        def _parse_or_now(self, dt=None):
            dt = parser.parse(dt) if dt else datetime.datetime.now(pytz.utc)
            dt = dt.replace(tzinfo=dt.tzinfo).astimezone(pytz.utc)
            return calendar.timegm(dt.timetuple())

        def update_tags(self, args):
            tags = []
            tags_arg = args.get('tags')
            if tags_arg:
                tags = tags_arg.split(',')
            tags.append('env:{}'.format(yenv.environment_type()))
            self.update({'tags': tags})
            return self

        def update(self, args=None):
            if isinstance(args, dict):
                self._args.update(args)
            return self

    class __EventBuilder(_AbstractBuilder):
        def __init__(self):
            Builder._AbstractBuilder.__init__(self)
            self.update({
                'id': 'cli-{}'.format(uuid.uuid4().hex),
                'title': None,
                'author': getpass.getuser()
            })

        def update_time(self, args):
            start = self._parse_or_now(args.get('startTimeSeconds'))
            end = self._parse_or_now(args.get('endTimeSeconds'))
            return self.update({'startTimeSeconds': start, 'endTimeSeconds': end})

        def build(self, args=None):
            self.update(args)
            return Event(**self._args)

    class __MicroEventBuilder(_AbstractBuilder):
        def __init__(self):
            Builder._AbstractBuilder.__init__(self)

        def update_time(self, args):
            time = self._parse_or_now(args.get('timeSeconds'))
            return self.update({'timeSeconds': time})

        def build(self, args=None):
            self.update(args)
            return MicroEvent(**self._args)
