import click
import market.pylibrary.yenv as yenv
import requests
from requests.packages.urllib3.exceptions import InsecureRequestWarning
from timeline_events import EventSender, Event

from timeline_events_cli.settings import SETTINGS
from timeline_events_cli.builder import Builder

# FIXME(iandreyev) Disabled SNI warnings, until migrate on Ubuntu 16.04
requests.packages.urllib3.disable_warnings(InsecureRequestWarning)


@click.group()
@click.option('--env', default='production', type=click.Choice(['testing', 'production']), help='Timeline API server')
@click.option('--dry', is_flag=True, help='Dry run, without sending')
@click.pass_context
def cli(ctx, env, dry):
    """
    \b
    timeline-events --env testing event title="This a test event" tags=cli:test,cli:test2 source=localhost startTimeSeconds="2017-05-26 12:34:56+03:00" endTimeSeconds="$(date --rfc-3339=seconds)"
    timeline-events microevent --event parent_event tags=cli:test,cli:test2
    """
    ctx.obj['env'] = env
    ctx.obj['dry'] = dry


def _get_event_sender(ctx):
    return EventSender(yenv.resolve_settings(SETTINGS, ctx.obj['env'])['host'])


def build(args, micro=False):
    event_dict = {}
    for item in args:
        event_dict.update([item.split('=')])
    return Builder(event_dict, micro).build()


@cli.command()
@click.argument('fields', nargs=-1, type=click.UNPROCESSED)
@click.pass_context
def event(ctx, fields):
    """
    \b
    sends an event to timeline
    \b
    optional kwargs-style [FIELDS]:
        id: event id, default - random UUID,
        title: event title, default empty,
        tags: comma separated key:value tags, default empty,
        startTimeSeconds: start time in RFC-3389 format, default now(),
        endTimeSeconds: start time in RFC-3389 format, default now(),
        type: event type, cli by default,
        project: project name, default 'PAYSYS',
        source: event source, local fqdn by default,
        status: severity status, 'INFO' by default,
        author: event author, by default $USER,
        text: event text, defult empty
    """
    message = build(fields)
    sender = _get_event_sender(ctx)
    if ctx.obj['dry']:
        print(message)
    else:
        sender.send_events(message)


@cli.command()
@click.argument('fields', nargs=-1, type=click.UNPROCESSED)
@click.option('--event', nargs=1, required=True, help='Parent event id')
@click.pass_context
def microevent(ctx, fields, event):
    """
    \b
    sends a microevent to timeline
    \b
    --event id - optional parent event id
    \b
    optional kwargs-style [FIELDS]:
        tags: comma separated key:value tags, default empty,
        timeSeconds: event time in RFC-3389 format, default now(),
        type: event type, cli by default,
        project: project name, default 'PAYSYS',
        source: event source, local fqdn by default,
        status: severity status, 'INFO' by default,
        text: event text, defult empty
    """
    message = build(fields, micro=True)
    sender = _get_event_sender(ctx)
    if ctx.obj['dry']:
        print(message)
    elif event:
        event_message = Event(id=event, startTimeSeconds=message.timeSeconds, type=message.type)
        event_message.microEvents.extend([message])
        sender.send_events(event_message)
    else:
        raise NotImplementedError("Cannot send microevent without parent")
#       sender.send_micro_events(message)


def main():
    cli(obj={})
