# %%
import warnings
warnings.filterwarnings("ignore")

# %%
from privacy_office.lib.soft_request.main import soft_request, nirvana_host
from privacy_office.lib.nirvana_combine.main import combine_this

from yt.yson import loads as yson_loads
# import yt.yson
from yt.yson.convert import yson_to_json
# import yt.yson.convert

from datetime import datetime


# %%
def download_log_result_text(download_link, token_nirvana):
    text = soft_request(
        download_link,
        token_nirvana,
        status_whitelist=[201, 200, 403, 404],
        raw_text=True
    )[:-1]

    if text.startswith('Access denied for step result item '):
        return None
    elif text.startswith('The specified key does not exist.'):
        return None
    else:
        return text


def log_result_to_text(result_log, token_nirvana):
    if 'storagePath' in result_log:
        download_link = result_log['storagePath']
        return download_log_result_text(download_link, token_nirvana)
    else:
        return None


def get_output(bad_instance, token_nirvana):
    outputs = []

    r = soft_request(
        nirvana_host+'getBlockResults',
        token_nirvana,
        type='post',
        json={
            "jsonrpc": "2.0",
            "id": "1",
            "method": "getBlockResults",
            "params": {
                "workflowId": bad_instance["workflowId"],
                "workflowInstanceId": bad_instance["ownerWorkflowInstanceId"],
                "blocks": [
                    {
                        "guid": bad_instance['blockId'],
                        "code": bad_instance['blockCode']
                    }
                ]
            }
        }
    )

    if 'error' in r:
        # print('error!')
        return []

    if r['result'] == []:
        return []

    r = r['result'][0]['results']

    for result in r:
        # # debug
        # print(result)
        # # debug
        text = log_result_to_text(result, token_nirvana)
        if text:
            outputs.append(text)

    return outputs


# %%
def get_log(bad_instance, token_nirvana):
    logs = []

    r = soft_request(
        nirvana_host+'getBlockLogs',
        token_nirvana,
        type='post',
        json={
            "jsonrpc": "2.0",
            "id": "1",
            "method": "getBlockLogs",
            "params": {
                "workflowId": bad_instance["workflowId"],
                "workflowInstanceId": bad_instance["ownerWorkflowInstanceId"],
                "blocks": [
                    {
                        "guid": bad_instance['blockId'],
                        "code": bad_instance['blockCode']
                    }
                ]
            }
        }
    )

    if 'error' in r:
        # print('error!')
        return []

    if r['result'] == []:
        return []

    for log in r['result'][0]['logs']:
        if log['logName'][:12] == 'job_launcher':
            continue

        # debug
        print(log)
        # debug

        text = log_result_to_text(log, token_nirvana)
        if text:
            logs.append(text)

    return logs


# %%
def get_cube_results(bad_instance, token_nirvana):
    report = {
        'outputs': get_output(bad_instance, token_nirvana),
        'logs': get_log(bad_instance, token_nirvana)
    }
    return ','.join(report['outputs']+report['logs'])


# %%
def get_cube_tokens(bad_instance, token_nirvana):
    from privacy_office.lib.recognize_secrets.main import recognize_secrets
    return recognize_secrets(get_cube_results(bad_instance, token_nirvana))


# %%
def yson_text_to_json(text):
    return yson_to_json(
        yson_loads(
            bytes(
                text,
                'utf-8'
            )
        )
    )


# %%
def get_nirvana_secret_info(secret_name, token_nirvana):
    return soft_request(
        f'https://nirvana.yandex-team.ru/api/secret/get/nirvana/secret?name={secret_name}',
        token_nirvana,
        status_whitelist=[201, 200, 404],
        status_warn=[404]
    )


# %%
def get_cube_secrets(bad_instance, token_nirvana):
    options_all = yson_text_to_json(
        bad_instance['options']
    )
    options = [option for option in options_all if option['type'] == 'SECRET']

    out = []

    for option in options:
        r = get_nirvana_secret_info(option['value'], token_nirvana)
        if 'error' not in r:
            out.append(
                {
                    'secret_id': r['id'],
                    'secret_name': r['name'],
                    'secret_author': r['created-by']
                }
            )

    return out


# %%
def debug_generate_link(bad_instance):
    return f"https://nirvana.yandex-team.ru/flow/{bad_instance['workflowId']}/{bad_instance['ownerWorkflowInstanceId']}/graph/FlowchartBlockOperation/{bad_instance['blockId']}"


# %%
@combine_this
def main(in1, in2, in3, token1, token2, token3, param1, param2, param3):
    token_nirvana = token1
    if param1 == "REVERSE":
        print("REVERSE")
        in1.reverse()

    out = []
    out_crashed = []

    start_time = datetime.now()
    all_rows = len(in1)
    i = 0

    for bad_instance in in1:
        if i % 1000 == 0:
            print(f"{i}/{all_rows} {(datetime.now()-start_time).seconds}")
        i += 1

        if bad_instance['executionResult'] in [None, 'SKIPPED']:
            bad_instance['token_info'] = []
            bad_instance['secret_info'] = []
            continue

        try:
            token_info = get_cube_tokens(bad_instance, token_nirvana)
            secret_info = get_cube_secrets(bad_instance, token_nirvana)
            bad_instance['token_info'] = token_info
            bad_instance['secret_info'] = secret_info
            out.append(
                bad_instance
            )
        except Exception as e:
            print('🔥')
            print(e)
            print(bad_instance)

            out_crashed.append(
                bad_instance
            )

    return {
        "out1": out,
        "out2": {
            "tableName": f"//home/scarif/echo_chainsaw/incidents_cooked/{datetime.now().strftime('%Y-%m-%d')}"
        },
        "out3": out_crashed
    }
