# %%
# регулярки и реквесты
from privacy_office.lib.soft_request.main import soft_request, st_host, abc_host, staff_host
from privacy_office.lib.nirvana_combine.main import combine_this
import re

# нужно для слежением за временем и подсчёта времени разных функций
import time
from datetime import datetime, timedelta

# для выбора случайных примеров плохих тикетов
import random


# %%
def passwords():
    query = 'issues?query="пароль логин"'
    filtrs = ((r'пароль[:]?[ ]?[A-Za-z\d@$!#%*?&]{6,20}', True),)
    name = 'passwords'
    thresholds = {'watch': {'danger_rate': 0.03, 'danger_sum': 30, 'min_threshold': 1},
                  'check': {'danger_rate': 0.01, 'danger_sum': 20}}
    show_name = (
        'пароли (пароли даже от тестовых аккаунтов' +
        '((https://wiki.yandex-team.ru/security/awareness/skarif/policy/#autentifikacionnyedannyepolzovatelejj хранить в открытых тикетах запрещено))' +
        ')'
    )
    return({'query': query, 'filtrs': filtrs, 'name': name, 'thresholds': thresholds, 'show_name': show_name})


def phones():
    query = 'issues?query="телефон"'
    filtrs = ((r'\b[7|8][\s(-]{0,2}9[\d]{2}[\s)-]{0,2}[\d]{3}[\s-]?[\d]{2}[\s-]?[\d]{2}\b', True),)
    name = 'phones'
    thresholds = {'watch': {'danger_rate': 0.3, 'danger_sum': 900, 'min_threshold': 3},
                  'check': {'danger_rate': 0.25, 'danger_sum': 800}}
    show_name = 'номера телефонов'
    return({'query': query, 'filtrs': filtrs, 'name': name, 'thresholds': thresholds, 'show_name': show_name})


def mails():
    query = 'issues?query="mail" OR "email" OR "почта"'
    filtrs = ((r'([a-z0-9_\.\-\+]+)@(yandex|(?!yandex-team)[a-z0-9\-]+|((?!yandex)[a-z0-9\-\.])+)\.([a-z\.]{2,6})(?:[\n$\s>])', True),
              (r'drvrc\.com', False))
    name = 'mails'
    thresholds = {'watch': {'danger_rate': 0.5, 'danger_sum': 2000, 'min_threshold': 5},
                  'check': {'danger_rate': 0.45, 'danger_sum': 4000}}
    show_name = 'электронные почты'
    return({'query': query, 'filtrs': filtrs, 'name': name, 'thresholds': thresholds, 'show_name': show_name})


def documents():
    query = 'issues?query=("имя фамилия" OR "фио") AND ("водительское удостоверение" OR "снилс" OR "серия номер")'
    filtrs = (
        (r"([^\w\'\.-]|^)(\d\d[ ]?\d\d [\d]{6}|\d{3}-\d{3}-\d{3} \d\d)\b", True),  # всё, что очень похоже на документы
        (r"([^\w\'\.-]|^)\+\d\d ?(\(0\))? ?\d{4} \d{6}\b", False)
    )  # отбросить британские номера
    name = 'documents'
    show_name = 'документы (паспорт, в/у, СНИЛС, etc)'
    thresholds = {'watch': {'danger_rate': 0.003, 'danger_sum': 3, 'min_threshold': 1},
                  'check': {'danger_rate': 0.002, 'danger_sum': 10}}
    return({'query': query, 'filtrs': filtrs, 'name': name, 'thresholds': thresholds, 'show_name': show_name})


def contracts():
    query = 'issues?query=("Допник" OR "Договор" OR "Дополнительное соглашение")'
    filtrs = ((r"[ \n](договор(а|ов|ы)?|дополнительное соглашение|допник)[ .,\-\n].*[ \n](договор(а|ов|ы)?|дополнительное соглашение|допник)[ .,\-\n]", True),)
    name = 'contracts'
    show_name = 'документы (договора/контракты)'
    thresholds = {'watch': {'danger_rate': 0.03, 'danger_sum': 30, 'min_threshold': 1},
                  'check': {'danger_rate': 0.2, 'danger_sum': 400}}
    return({'query': query, 'filtrs': filtrs, 'name': name, 'thresholds': thresholds, 'show_name': show_name})


def get_handlers():
    return(passwords(),
           phones(),
           mails(),
           documents(),
           contracts())


def get_bad_names():
    return([handler['name'] for handler in get_handlers()])


# %%
def get_duty(token):
    today = datetime.now().strftime('%Y-%m-%d')
    # смотрю на такой большой отрезок, чтобы даже в случае длинных нерабочих отрезков множество было непусто
    next_month = (datetime.now() + timedelta(days=30)).strftime('%Y-%m-%d')
    return soft_request(
        abc_host + f"duty/shifts/?service__slug=privacy_office&date_from={today}&date_to={next_month}",
        token
    )['results'][0]['person']['login']


# %%
def watch_dog(token, handlers=get_handlers(), max_iter=100, verbose=True, debug=False):
    assert type(handlers) == tuple, "handlers must be a tuple of dicts!"
    assert type(handlers[0]) == dict, "handlers must be a tuple of dicts!"

    # bad_names = [handler['name'] for handler in handlers)]
    bad_names = get_bad_names()

    report = {'body': {}, 'meta': {'all_queues': 0, 'to_check_queues': 0, 'report_time': ''}}
    body = report['body']

    if debug:
        data = {}

    if verbose:
        start_time = time.time()

    out = {}

    for handler in handlers:

        query = handler['query']
        filtrs = handler['filtrs']
        bad_name = handler['name']
        threshold = handler['thresholds']['watch']

        if verbose:
            handler_time = time.time()
            print(f'\n handler = {bad_name}')

        # прокручиваем страницы от 1 до max_iter
        for n in range(1, 1 + max_iter):

            if verbose and n % 10 == 0:
                print(f'{n*100}/{max_iter*100} tickets are checked!')

            r = soft_request(
                st_host + query,
                token,
                params={
                    'perPage': 100,
                    'page': n
                }
            )
            for ticket in r:
                # description не всегда существует, так что
                ticket.setdefault('description', '')

                if debug:
                    data.update({ticket['key']: ticket})

                key = ticket['key']  # issue_key
                queue = ticket['queue']['key']
                if queue not in body:  # если впервые наткнулись на очередь
                    if not debug:
                        body.setdefault(
                            queue,
                            dict(
                                (
                                    i,
                                    {
                                        'total_bad': 0,
                                        'bad': {},
                                        'total': 0,
                                        'min': 1e11,
                                        'max': 0,
                                        'is_suspicious': False
                                    }
                                ) for i in bad_names
                            )
                        )
                    else:  # то же, но храню ещё хорошие тикеты
                        body.setdefault(
                            queue,
                            dict(
                                (
                                    i,
                                    {
                                        'total_bad': 0,
                                        'total_good': 0,
                                        'bad': {},
                                        'good': {},
                                        'total': 0,
                                        'min': 1e11,
                                        'max': 0,
                                        'is_suspicious': False
                                    }
                                ) for i in bad_names
                            )
                        )
                    body[queue]['to_check'] = 0

                body[queue][bad_name]['total'] += 1
                body[queue][bad_name]['min'] = min(body[queue][bad_name]['min'], int(key.split('-')[1]))
                body[queue][bad_name]['max'] = max(body[queue][bad_name]['max'], int(key.split('-')[1]))

                bad = True
                to_search = (ticket['summary'] + ' ' + ticket['description'] + '\n').lower()
                for filtr in filtrs:
                    if bool(re.search(filtr[0], to_search, flags=re.S)) != filtr[1]:
                        bad = False
                        break
                if bad:
                    body[queue][bad_name]['total_bad'] += 1
                    body[queue][bad_name]['bad'].update({key: 'st.yandex-team.ru/'+key})
                elif debug:
                    body[queue][bad_name]['total_good'] += 1
                    body[queue][bad_name]['good'].update({key: 'st.yandex-team.ru/'+key})

        # замер подозрительных очередей
        for queue in body:
            stats = body[queue][bad_name]

            stats_rate = stats['total_bad'] / (stats['max'] - stats['min'] + 1)
            stats_pred = stats['max'] * stats_rate
            bad_find = stats['total_bad']

            d_rate = threshold['danger_rate']
            d_sum = threshold['danger_sum']
            m_threshold = threshold['min_threshold']

            if (stats_rate >= d_rate or stats_pred >= d_sum) and bad_find >= m_threshold:
                stats['is_suspicious'] = True
                if body[queue]['to_check'] == 0:
                    report['meta']['to_check_queues'] += 1
                    body[queue]['to_check'] = 1

        if verbose:
            print(f'handler {bad_name} is done! {round(time.time() - handler_time, 2)} seconds')

    report['meta']['all_queues'] = len(body)
    report['meta']['report_time'] = datetime.now().strftime('%Y-%m-%d')

    if debug:
        out['debug'] = data

    if verbose:
        print(f'watch_dog is done! {round(time.time() - start_time, 2)} seconds')

    out = {'report': report}
    return out


# %%
def yt_table_watch(main_table, watch_report, verbose=False):
    main_table_queues = [row['queue'] for row in main_table]

    body = watch_report['report']['body']
    for queue in body:

        if verbose:
            print(f'\n\n👥 {queue}')

        can_update = False
        if queue not in main_table_queues:

            if verbose:
                print(f'new queue: {queue}')
                print('will be updated!')

            default_raw = dict(
                [
                    ('queue', queue),
                    ('status', None),
                    ('wait_warning', None),
                    ('issue_key', None),
                    ('last_checked', None),
                    ('check_after', None)
                ] +
                [
                    (f'{column}_{bad_name}', None) for column in ['is', 'examples', 'whitelist'] for bad_name in get_bad_names()
                ]
            )
            main_table.append(default_raw)
            main_table_queues.append(queue)
            queue_index = main_table_queues.index(queue)
            queue_row = main_table[queue_index]
            can_update = True
        else:

            queue_index = main_table_queues.index(queue)
            queue_row = main_table[queue_index]

            if verbose:
                print(f"old queue, last status = {queue_row['status']}, check after = {queue_row['check_after']}")

            if (
                queue_row['status'] not in ['s_dirt', 'whitelist'] and
                queue_row['wait_warning'] is not True and
                datetime.now() >= datetime.strptime(queue_row['check_after'], '%Y-%m-%d')
            ):
                can_update = True
                if verbose:
                    print('will be updated!')
            elif verbose:
                print('skipped')

        if can_update:
            if body[queue]['to_check'] == 0:

                if verbose:
                    print(f'{queue} -> s_clean')

                queue_row['status'] = 's_clean'
            else:

                if verbose:
                    print(f'{queue} -> s_dirt')

                queue_row['status'] = 's_dirt'

            queue_row['last_checked'] = datetime.now().strftime('%Y-%m-%d')
            queue_row['check_after'] = datetime.now().strftime('%Y-%m-%d')

    main_table[queue_index] = queue_row
    return main_table


# %%
def queue_check(queue_name, token, handlers=get_handlers(), max_tickets=100000, verbose=True, debug=False, ASC=False):

    if verbose:
        start_time = time.time()

    # костыль для поиска по ключу очереди https://st.yandex-team.ru/TOOLSUP-91224#60b7856948fd44111db5e968
    # не лечит кейсы, где название очереди строчное https://st.yandex-team.ru/TOOLSUP-119243
    queue_name = queue_name.lower()

    query_len = f'issues?query= Queue: "{queue_name}" "Sort by": Key DESC &perPage=1'
    brackets = '{}'
    if ASC:
        query = f'issues?query= Key: >= "{queue_name}-{brackets}" "Sort by": Key ASC &perPage=100'
    else:
        query = f'issues?query= Key: <= "{queue_name}-{brackets}" "Sort by": Key DESC &perPage=100'

    # узнаем длину очереди

    r = soft_request(
        st_host + query_len,
        token
    )

    if r != []:  # иногда это пустое множество, поэтому исключение
        queue_len = int(r[0]['key'].split('-')[-1])
    else:
        queue_len = 0

    if verbose:
        print(f'\n queue {queue_name}, ~{queue_len} tickets')
        start_time=time.time()

    # if debug:
    #     data = {}

    bad_names = [handler['name'] for handler in handlers]
    queue_report = dict(
        (
            i,
            {
                'total_bad': 0,
                'bad': {},
                'is_warning': False
            }
        ) for i in bad_names
    )
    queue_report.update(
        {
            'meta': {
                'total_checked': 0,
                'name': queue_name,
                'warning': 0
            }
        }
    )

    # Пропустить маленькую очередь
    if not debug and queue_len < 200:
        if verbose:
            print(f'{queue_name} is too small! Skipped!')
        return {'queue_report': queue_report}

    if ASC:
        n_key = 1
    else:
        n_key = queue_len

    while (
        ASC and (n_key <= queue_len and (n_key <= max_tickets or max_tickets <= -1)) or
        not ASC and (n_key >= 1 and (n_key >= queue_len - max_tickets + 1 or max_tickets <= -1))
    ):
        if debug:
            print(n_key)

        r = soft_request(
            st_host + query.format(n_key),
            token
        )

        for ticket in r:
            # description не всегда существует, так что
            ticket.setdefault('description', '')
            # summary тоже! Вы только посмотрите: MARKETSUP-8413
            ticket.setdefault('summary', '')

            # if debug:
            #     data.update({ticket['key']: ticket})

            queue_report['meta']['total_checked'] += 1

            key = ticket['key']
            to_search = (ticket['summary'] + ' ' + ticket['description'] + '\n').lower()
            for handler in handlers:
                filtrs = handler['filtrs']
                bad_name = handler['name']
                bad = True
                for filtr in filtrs:
                    if bool(re.search(filtr[0], to_search, flags=re.S)) != filtr[1]:
                        bad = False
                        break
                if bad:
                    queue_report[bad_name]['total_bad'] += 1
                    queue_report[bad_name]['bad'].update({key: 'st.yandex-team.ru/'+key})

        if r != []:  # иногда это пустое множество, поэтому исключение
            if ASC:
                n_key = int(r[-1]['key'].split('-')[-1]) + 1
            else:
                n_key = int(r[-1]['key'].split('-')[-1]) - 1
        else:
            break

    # замер подозрительных очередей
    for handler in handlers:

        bad_name = handler['name']
        stats = queue_report[bad_name]
        threshold = handler['thresholds']['check']

        if queue_report['meta']['total_checked'] != 0:
            stats_rate = stats['total_bad'] / (queue_report['meta']['total_checked'])
        else:
            stats_rate = 0
        stats_sum = stats['total_bad']

        d_rate = threshold['danger_rate']
        d_sum = threshold['danger_sum']

        if stats_rate >= d_rate or stats_sum >= d_sum:
            stats['is_warning'] = True
            queue_report['meta']['warning'] = 1

    if verbose:
        if max_tickets <= queue_len:
            result = 'stopped by max_tickets'
        else:
            result = 'checked completely'
        print(f'{queue_name} {result}')
        total_checked = queue_report['meta']['total_checked']
        print(f'~{total_checked} tickets for {round(time.time() - start_time, 2)} seconds')

    out = {'queue_report': queue_report}
    # if debug:
    #     out['debug'] = data

    return out


# %%
def yt_table_check(main_table, token, verbose=True, max_tickets=10000, **kwargs):

    if verbose:
        start_time = time.time()

    # триггер для неполной остановки, если проверка затянулись.
    stopped_by_timer = False

    for index in range(len(main_table)):
        row = main_table[index]
        # Если w_dirt, то тикет тоже надо проверить — если настало время check_after
        if row['status'] not in ['s_dirt', 'w_dirt']:
            continue

        if not datetime.now() >= datetime.strptime(row['check_after'], '%Y-%m-%d'):
            continue

        queue = row['queue']

        if verbose:
            print(f"\n queue: {queue}")

        # Если не подать данные о begin_time и sleep_time, обход будет идти до победного конца
        if all(key in kwargs for key in ['begin_time', 'sleep_time']):
            if time.time() - kwargs['begin_time'] >= kwargs['sleep_time']:
                stopped_by_timer = True
                break

        # Если тикет уже когда-то заводился
        issue_key = row["issue_key"]
        # это явно можно сделать элегантнее
        if not row['issue_key'] in [None, None]:
            query = f'issues/{issue_key}'
            r = soft_request(
                st_host + query,
                token
            )
            if r['status']['key'] != 'closed':
                if verbose:
                    print(f' ⁉️ this queue has unclosed ticket: {issue_key}')
                continue
            else:
                if verbose:
                    print(f' re-checking queue: {issue_key}')

        handlers_check = [handler for handler in get_handlers() if row[f"whitelist_{handler['name']}"] is not True]
        handlers_whitelisted = [handler for handler in get_handlers() if row[f"whitelist_{handler['name']}"] is True]
        queue_report = queue_check(queue, token=token, handlers=handlers_check, max_tickets=max_tickets, verbose=verbose)['queue_report']

        row['last_checked'] = datetime.now().strftime('%Y-%m-%d')
        if queue_report['meta']['warning'] == 0:
            if verbose:
                print('✅ good queue!')

            row['status'] = 'w_clean'
            row['last_checked'] = datetime.now().strftime('%Y-%m-%d')

            next_check = 21 + 21 * (queue_report['meta']['total_checked']/(max_tickets + 1))**(1/2)
            row['check_after'] = (datetime.now() + timedelta(days=next_check)).strftime('%Y-%m-%d')
        else:
            if verbose:
                print('❗️ warning! Bad queue!')

            row['status'] = 'w_dirt'
            row['wait_warning'] = True
            row['last_checked'] = datetime.now().strftime('%Y-%m-%d')

            next_check = 14 + 7 * (queue_report['meta']['total_checked']/(max_tickets + 1))**(1/2)
            row['check_after'] = (datetime.now() + timedelta(days=next_check)).strftime('%Y-%m-%d')

        bad_names = get_bad_names()
        bad_names_whitelisted = [handler['name'] for handler in handlers_whitelisted]
        for bad_name in bad_names:
            if bad_name in bad_names_whitelisted:
                row[f'is_{bad_name}'] = None
                row[f'examples_{bad_name}'] = None
                continue

            if queue_report[bad_name]['is_warning'] is not True:
                row[f'is_{bad_name}'] = None
                row[f'examples_{bad_name}'] = None
                continue

            row[f'is_{bad_name}'] = True
            # обновить examples_bad
            total_bad = queue_report[bad_name]['total_bad']
            bad_examples = random.sample(list(queue_report[bad_name]['bad'].values()), min(total_bad, 5))

            if bad_examples == []:
                row[f'examples_{bad_name}'] = None
            else:
                row[f'examples_{bad_name}'] = ','.join(bad_examples)

        if verbose:
            print(f"bad_status = {[row[f'is_{i}'] for i in bad_names]}")

        main_table[index] = row

    if verbose:
        if stopped_by_timer:
            print(f'yt_table_check is not done entirely. {round(time.time() - start_time, 2)} seconds')
        else:
            print(f'yt_table_check is done completely! {round(time.time() - start_time, 2)} seconds')

    return main_table

# %%

# С декабря 2021 баг с падежами не пофиксили https://st.yandex-team.ru/TOOLSUP-107166
# психанул — теперь называю людей по имени-фамилии
raw_description = '''**Подробнее о том, что здесь происходит:** https://clubs.at.yandex-team.ru/mag/71249

Привет, кто:{lead_username}! Я вижу в твоей очереди %%{queue}%% тикеты с ((https://wiki.yandex-team.ru/security/awareness/skarif/policy/ приватными данными)). \
Это значит, что у очереди неправильно выставлены настройки доступов. Сейчас я покажу тебе как всё исправить.

Я, не обладая доступами к закрытым специфичным группам, вижу тикеты, содержащие приватные данные в следующих категориях:
{problems}

**Что делать?**
1. Зайди на страницу настроек прав доступа своей очереди. ((https://st.yandex-team.ru/admin/queue/{queue}/permissions ссылка))
2. Ознакомься с **((https://wiki.yandex-team.ru/privacy_office/qhide/queue_hiding/ гайдом))** и проведи настройку очереди.
3. Отпишись в комментарии если...
* ... настройки очереди удалось поправить, что очередь прикрыта — для проверки модератора.
* ... тревога ложная, что все тикеты сработали ложно (шаблонные данные, B2B-телефоны, etc) — для добавления в вайтлист.
* ... есть вопросы, сложности — попроси помощи, отправь скриншот таблицы доступов — для консультации.
  **Вы восхитительны!**
'''


# %%
def ticket_creator(main_table, token, today_limit=3, verbose=True):
    qhide_duty = get_duty(token)

    warning_queues = [row for row in main_table if row['wait_warning'] is True]

    if verbose:
        print(f'total warning queues: {len(warning_queues)}')

    today = 0
    for row in warning_queues:
        today += 1
        if today > today_limit:
            break

        queue = row['queue']
        if verbose:
            print(f'\n😠 {queue}')

        # вытянем владельца
        query = f'queues/{queue}'
        r = soft_request(
            st_host + query,
            token,
            status_whitelist=[200, 201, 403]
        )
        # у st-api как-то криво реализован поиск владельца очереди, поэтому в случае ошибки 403 первый случай
        if 'errors' in r:
            lead_username = r['errorMessages'][0].split()[-1][:-2]
        else:
            lead_username = r['lead']['id']

        # узнаем его staff-имя
        query = 'persons'
        lead_name = soft_request(
            staff_host + query,
            token,
            params={
                '_pretty': 1,
                'login': lead_username
            }
        )['result'][0]['name']['first']['ru']

        if verbose:
            print(f'Владелец - {lead_name} {lead_username}')

        # генерация списка проблем
        problems = ''
        for handler in get_handlers():
            bad_name = handler['name']
            bad_show_name = handler['show_name']
            # превращаю в str чтобы None выдал нормальный split
            problems_list = str(row[f'examples_{bad_name}']).split(',')
            if problems_list == ['None']:
                pass
            else:
                problems += f'\n\n {bad_show_name} (до 5 примеров):'
                for example in problems_list:
                    problems += f'\n %%{example.split("/")[-1]}%%'

        summary = f'Приватные данные в очереди - {queue}'

        # https://st.yandex-team.ru/TOOLSUP-107166 дождаться исправления бага или выпилить lead_name
        description = raw_description.format(
            queue=queue,
            problems=problems,
            lead_username=lead_username,
            lead_name=lead_name
        )

        # создание тикета
        query = 'issues/'
        r = soft_request(
            st_host + query,
            token,
            json={
                "queue": 'QHIDE',
                "type": 'incident',
                "summary": summary,
                "description": description,
                "assignee": lead_username,
                "access": qhide_duty,
                "priority": 'critical',
                "iterationCount": 0,
                "deadline": (datetime.now() + timedelta(days=10)).strftime('%Y-%m-%d')
            },
            type='post'
        )
        new_issue_key = r['key']

        if row['issue_key'] is not None:
            issue_key = row['issue_key']

            if verbose:
                print(f'🔁 Повторное нарушение - {issue_key}')

            # линк к старому тикету, если повтор
            query = f'issues/{new_issue_key}/links'
            soft_request(
                st_host + query,
                token,
                json={
                    "relationship": 'duplicates',
                    "issue": issue_key
                },
                type='post'
            )

        row['issue_key'] = new_issue_key
        row['wait_warning'] = None

    return main_table


# %%
def debug_check_ticket(ticket, token, is_link=True):

    if type(ticket) == tuple or type(ticket) == list:
        for item in ticket:
            debug_check_ticket(item, is_link=is_link)
    else:
        if is_link:
            print(f'link: {ticket}')

            r = soft_request(f'https://st-api.yandex-team.ru/v2/issues/{ticket.split("/")[-1]}', token)
            r.setdefault('description', '')
            ticket = r['description']
        else:
            print(f'description: {ticket}')

        for handler in get_handlers():
            filtrs = handler['filtrs']
            bad_name = handler['name']

            print(f'checking via {bad_name}')

            bad = True
            for filtr in filtrs:
                if bool(re.search(filtr[0], ticket+'\n'.lower()), flags=re.S) != filtr[1]:
                    bad = False
                    break
                else:
                    print(re.findall(filtr[0], ticket+'\n'.lower()))
            if bad:
                print(f'❌ {bad_name} BAD')
            else:
                print(f'✅ {bad_name} GOOD')


# %%
@combine_this
def main(in1, in2, in3, token1, token2, token3, param1, param2, param3):
    params = in2
    sleep_time = params.get('sleep_time', 60*60*14)
    watch_iters = params.get('watch_iters', 100)
    check_tickets = params.get('check_tickets', 100000)
    tickets_limit = params.get('tickets_limit', 10)
    verbose = params.get('verbose', None)

    main_table = in1

    begin_time = time.time()

    verbose_arg = {}
    if verbose is not None:
        verbose_arg = {'verbose': verbose}

    main_table = yt_table_watch(main_table, watch_dog(token=token1, max_iter=watch_iters, **verbose_arg), **verbose_arg)

    main_table = yt_table_check(main_table, token=token1, max_tickets=check_tickets, **verbose_arg, begin_time=begin_time, sleep_time=sleep_time)

    main_table = ticket_creator(main_table, token=token1, today_limit=tickets_limit, **verbose_arg)

    if time.time() - begin_time <= sleep_time:
        print("Overfulfilling the plan!")

    print("It's time to sleep! 😴")
    print(f"{round(time.time() - begin_time, 2)} seconds")

    return {
        "out1": main_table
    }
