import argparse
import copy
import datetime
import requests
import time
from pathlib import Path

from ads.libs.yql import get_yql_client

# ===========================================================================================

# https://nirvana.yandex-team.ru/data/af455fa0-6fb1-4bb0-91d9-b0ea87d47d25
EXAMPLE_DATA_ID = "af455fa0-6fb1-4bb0-91d9-b0ea87d47d25"


HITMAN_CONFIG = {
    # 'token': (Path.home() / '.hitman/token').read_text().strip(),
    'requester': 'topilskiy-ak',
    'service': 'goods_category_coverage'
}


def hitman_process_starter(data_id=EXAMPLE_DATA_ID):
    """hitman process starter
    https://wiki.yandex-team.ru/SergejjMokin/Rabota-s-nirvanojj-i-xitmanom/Zapusk-xitman-processa-cherez-xitman-api/"""
    headers = {
        'Content-Type': 'application/json',
        'Authorization': 'OAuth ' + HITMAN_CONFIG['token']
        }

    hitman = 'https://hitman.yandex-team.ru/'
    start_proc = 'api/v1/execution/start/'
    service = HITMAN_CONFIG['service']
    start_proc_url = hitman + start_proc + service

    data = {}
    data['requester'] = 'topilskiy-ak'
    data['properties'] = {"INPUT_NIRVANA_STORED_DATA_ID": data_id}

    resp = requests.post(start_proc_url, headers=headers, json=data)
    print(resp.text)


# ===========================================================================================

DEFAULT_PARAMS = {
    '${global.force_reparse_us}': 'False',
    '${global.force_redo_aggregate}': 'False',
    '${global.disable_if_no_parsed_us}': 'False',
    '${global.disable_form_merged_sorted}': 'True',
    '${global.date}': '2022-07-01',
    '${global.outputdir_prime}': '//home/goods_quality/charts/category_coverage/prime',
    '${global.outputdir_parallel}': '//home/goods_quality/charts/category_coverage/parallel',
}


def get_base_scripts():
    return {
        'Category Coverage Prime': Path('../lib/prime.sql').read_text(),
        'Category Coverage Parallel': Path('../lib/parallel.sql').read_text(),
    }


def get_yql_queries(dates, base_scripts):
    queries = []
    for date in dates:
        params = copy.deepcopy(DEFAULT_PARAMS)
        params['${global.date}'] = date
        for base_title, script in base_scripts.items():
            title = 'YQL ' + base_title + ' [' + date + ']'
            for param, replacement in params.items():
                script = script.replace(param, replacement)
            query = {"title": title, "query": script, "syntax_version": 1}
            queries.append(query)
    return queries


def get_all_yql_queries(base_scripts, params_list):
    queries = []
    for base_script in base_scripts:
        for params in params_list:
            query = get_yql_queries(base_script, params)
            queries.append(query)
    return queries


def launch_yql_query(query, yql_client):
    request = yql_client.query(**query)
    request.run()


def get_dates_between(date_start, date_end, date_delta):
    start = datetime.datetime.fromisoformat(date_start)
    end = datetime.datetime.fromisoformat(date_end)
    dates = [start + datetime.timedelta(days=x) for x in range(0, (end - start).days + 1, date_delta)]
    return [d.strftime('%Y-%m-%d') for d in dates]


def recalculate_past_using_yql_scripts(
    date_start, date_end, date_delta,
    base_scripts, yql_client, is_dry_run
):
    dates = get_dates_between(date_start, date_end, date_delta)
    queries = get_yql_queries(dates, base_scripts)
    for i, query in enumerate(queries):
        print('\nLaunching Query ' + str(i) + '/' + str(len(queries)) + ': ' + query['title'])
        if not is_dry_run:
            launch_yql_query(query, yql_client)
            time.sleep(1)
    print("\nDone!")


# ===========================================================================================

def modify_base_scripts(base_scripts, args):
    scripts_type = args.type
    if scripts_type == 'prime':
        del base_scripts['Category Coverage Parallel']
    if scripts_type == 'parallel':
        del base_scripts['Category Coverage Prime']
    DEFAULT_PARAMS['${global.disable_if_no_parsed_us}'] = str(args.no_parse_us)
    DEFAULT_PARAMS['${global.force_redo_aggregate}'] = str(args.force_redo_aggregate)


# example: ./recalculate_past -s '2022-06-18' -e '2022-06-25'
def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('-s', '--start', default='2022-07-01')
    parser.add_argument('-e', '--end',   default='2022-07-01')
    parser.add_argument('-d', '--delta', default=1, type=int)
    parser.add_argument('-t', '--type',  default='all', choices=['all', 'prime', 'parallel'])

    parser.add_argument('--no-parse-us', action='store_true')
    parser.add_argument('--force-redo-aggregate', action='store_true')

    parser.add_argument('--dry-run', action='store_true')
    args = parser.parse_args()
    return args


def main():
    args = parse_args()

    yql_client = get_yql_client(token=None, db="hahn")  # token from ~/.yql/token

    base_scripts = get_base_scripts()
    modify_base_scripts(base_scripts, args)

    recalculate_past_using_yql_scripts(args.start, args.end, args.delta, base_scripts, yql_client, args.dry_run)


if __name__ == '__main__':
    main()
