from products.metrics.search_empty_serps.block import SearchEmptySerpsGraph, BinarySearch
import quality.webfresh.libraries.prs as prs

import argparse
import json
import typing as tp
import vh


def parse_arguments() -> argparse.Namespace:
    argument_parser = argparse.ArgumentParser()

    argument_parser.add_argument('--quota', required=False, default=None, help='nirvana quota to start instance')
    argument_parser.add_argument('--nirvana_token', required=True,
                                 help='get nirvana token: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=e27cff2994d34d8ea8e43333e0311d6d')

    argument_parser.add_argument('--testid', required=True, help='testid for formula for threshold experiment')
    argument_parser.add_argument('--target_value', type=float, required=False,
                                 help='baseline metric value, overrides empty serps metric from production when enabled')
    argument_parser.add_argument('--query', required=True, help='the number of one of the queries')
    argument_parser.add_argument('--cgi', required=True, help='cgi for models')

    argument_parser.add_argument('--begin', type=float, required=False, default=0, help='start value of threshold')
    argument_parser.add_argument('--end', type=float, required=False, default=1, help='end value of threshold')
    argument_parser.add_argument('--max_operations_count', type=int, required=False, default=5,
                                 help='the max number of operations to find threshold')
    argument_parser.add_argument('--max_percentage_eps', type=float, required=False, default=1.5,
                                 help='the max percentage difference that is acceptable')

    argument_parser.add_argument('--name', required=True, help='name of mlm process')
    argument_parser.add_argument('--max_running_jobs', type=int, required=True, help='max running jobs for mlm process')
    argument_parser.add_argument('--max_failed_rate', required=True, help='max failed rate for mlm process')

    argument_parser.add_argument('--parent_workflow_uid', required=True, help='uid of parent workflow')
    argument_parser.add_argument('--parent_workflow_url', required=True, help='url of parent workflow')

    argument_parser.add_argument('--output', required=True, help='result threshold')

    return argument_parser.parse_args()


def get_diff_percent_of_pv_empty_serp_from_launch_template(path_to_launch_template: str) -> float:
    with open(path_to_launch_template) as launch_template_file:
        launch_template = json.load(launch_template_file)
    return launch_template['launches'][0]['diffQueryGroups'][0]['metrics'][0]['diffPercent']


def get_diff_percent_of_pv_empty_serp_from_launch_template_no_baseline(path_to_launch_template: str,
                                                                       target_value: float) -> float:
    with open(path_to_launch_template) as launch_template_file:
        launch_template = json.load(launch_template_file)
    value = launch_template['launches'][0]['diffQueryGroups'][0]['metrics'][0]['value']
    return 100.0 * (value - target_value) / target_value


def build_graph(args: argparse.Namespace, threshold) -> vh.frontend.targets.File:
    graph = SearchEmptySerpsGraph()
    metrics_config = graph.get_metrics_config()
    test_cgi = graph.get_test_cgi(threshold)
    return graph.get_launch_template_file_target(
        metrics_config, args.testid, args.query, args.cgi, test_cgi, args.name, args.max_failed_rate,
        args.max_running_jobs
    )


def get_diff_percent_of_pv_empty_serp_for_threshold(args: argparse.Namespace, threshold: float) -> float:
    diff_percent_of_pv_empty_serp = None

    with vh.Graph():
        launch_template_file_target = build_graph(args, threshold)
        keeper = vh.run_async(
            oauth_token=args.nirvana_token, quota=args.quota,
            workflow_guid=args.parent_workflow_uid, description=args.parent_workflow_url,
            start=True, api_retries=5
        )
        keeper.get_total_completion_future()
        path_to_launch_template = keeper.download(launch_template_file_target)
        if args.target_value is not None:
            diff_percent_of_pv_empty_serp = get_diff_percent_of_pv_empty_serp_from_launch_template_no_baseline(path_to_launch_template,
                                                                                                                args.target_value)
        else:
            diff_percent_of_pv_empty_serp = get_diff_percent_of_pv_empty_serp_from_launch_template(path_to_launch_template)

    return diff_percent_of_pv_empty_serp


def main_binary_search(args: argparse.Namespace) -> tp.Tuple[float, float]:
    binary_search = BinarySearch(args.begin, args.end, args.max_percentage_eps, args.max_operations_count)
    diff_percent_of_pv_empty_serp = float('inf')
    threshold = float('inf')

    while not binary_search.is_stop(diff_percent_of_pv_empty_serp):
        prs.logger.info(f'{binary_search.operations_count} iteration: [{binary_search.begin}, {binary_search.end}]')
        threshold = binary_search.get_threshold()
        prs.logger.info(f'threshold: {threshold}')
        diff_percent_of_pv_empty_serp = get_diff_percent_of_pv_empty_serp_for_threshold(args, threshold)
        prs.logger.info(f'diff_percent_of_pv_empty_serp: {diff_percent_of_pv_empty_serp}')
        binary_search.next_step(threshold, diff_percent_of_pv_empty_serp)

    return threshold, diff_percent_of_pv_empty_serp


def write_threshold(args: argparse.Namespace, threshold: float, diff_percent_of_market_empty_serp: float) -> None:
    with open(args.output, 'w') as f:
        json.dump({'threshold': threshold, 'diff_percent_of_market_empty_serp': diff_percent_of_market_empty_serp}, f,
                  indent=4)


def main():
    args = parse_arguments()
    threshold, diff_percent_of_pv_empty_serp = main_binary_search(args)
    write_threshold(args, threshold, diff_percent_of_pv_empty_serp)
