# coding: utf-8

import logging
import argparse
import os

import yt.wrapper as yt

import irt.iron.options as iron_opts
from irt.bannerland.options import get_option as get_bl_opt

from bannerland.archive_workers.common import get_attr_name
from bannerland.archive_workers.aw_configs import get_worker_conf

logging.basicConfig(format="%(asctime)s\t[%(process)d]\t%(message)s", level=logging.INFO)


def run_worker(task_type, worker_name, worker_type, archive_dir, yt_cluster, ignore_depends):

    worker_inf = get_worker_conf(task_type, worker_type)['workers'][worker_name]
    kwargs = worker_inf.get('kwargs', {})

    default_pool = kwargs.get('yt_pool', 'bannerland-data')

    yt_client = yt.YtClient(yt_cluster, config={
        'token_path': iron_opts.get('yt_token_path'),
        'remote_temp_tables_directory': get_bl_opt('yt_temp_dir'),
        'spec_defaults': {
            'pool': os.getenv('YT_POOL', default_pool),
        },
    })

    kwargs['task_type'] = task_type
    if 'depend' in worker_inf:
        kwargs['depend_attrs'] = [get_attr_name(d) for d in worker_inf['depend']]

    worker = worker_inf['class'](attr_name=get_attr_name(worker_name), yt_client=yt_client, ignore_depends=ignore_depends, **kwargs)

    logging.warning('run worker %s for dir %s ...', worker_name, archive_dir)
    worker.process(archive_dir)

    logging.warning('done!')


# лочим + определяем директорию
def main_run(task_type, worker_name, worker_type, yt_cluster, ignore_depends):
    conf = get_worker_conf(task_type, worker_type)
    lock_table = conf['lock_dir'] + '/' + worker_name
    lock_client = yt.YtClient(yt_cluster, config={'token_path': iron_opts.get('yt_token_path')})
    lock_client.create('table', lock_table, recursive=True, ignore_existing=True)
    logging.warning('main_run {}, type = {}, cluster {}'.format(worker_name, worker_type, yt_cluster))

    with lock_client.Transaction():
        try:
            lock_client.lock(lock_table)
        except:
            logging.warning("Can't get lock for %s, worker already running", lock_table)
            return False

        run_worker(task_type, worker_name, worker_type, conf['archive_dir'], yt_cluster=yt_cluster, ignore_depends=ignore_depends)


if __name__ == '__main__':
    yt.config['proxy']['url'] = 'hahn'
    yt.config['token_path'] = iron_opts.get('yt_token_path')

    argparser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    argparser.add_argument('worker_name')
    argparser.add_argument('--type', choices=['pocket', 'full_state'], default='pocket')
    argparser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    argparser.add_argument('-c', '--yt-cluster', choices=['hahn', 'arnold'], default='hahn')
    argparser.add_argument('-i', '--ignore-depends', action='store_true', default=False)
    args = argparser.parse_args()

    worker_name = args.worker_name
    worker_type = args.type

    yt_cluster = args.yt_cluster
    ignore_depends = args.ignore_depends
    main_run(args.task_type, worker_name, worker_type, yt_cluster, ignore_depends)
