import datetime
import dateutil.tz
import dateutil.parser
import time
import isodate
import re
import enum

import yt.wrapper as yt
import sandbox.projects.common.solomon as solomon

import irt.logging

logger = irt.logging.getLogger(irt.logging.BANNERLAND_PROJECT, __name__)

DT_FORMAT = '%Y-%m-%d_%H:%M:%S'


def get_backup_dir_top_path(release_type):
    return f'//home/bannerland/{release_type}/backups'


def get_backup_config_path(release_type):
    return f'//home/bannerland/{release_type}/yt_tables_backup_config'


def get_yt_folder_size_recursive(folder_path, yt_client):
    return sum(x.attributes['resource_usage']['disk_space'] for x in yt_client.search(
        folder_path,
        node_type=['table', 'file', 'document'],
        attributes=['resource_usage']
    ))


def get_backup_config(yt_token, release_type):
    yt_client = yt.YtClient('markov', token=yt_token)

    # You can get old state of the table, if `enable_dynamic_store_read` attribute is not equal to `True`
    backup_config = yt_client.read_table(get_backup_config_path(release_type))
    return list(backup_config)


class Action(enum.Enum):
    RENAME_LAST = 0
    MAKE_BACKUP = 1
    SKIP = 2


def decide_action(now_dt, source_object_modification_dt, last_backup_dt, backup_period):
    if last_backup_dt is None or now_dt > last_backup_dt + backup_period:
        if last_backup_dt is not None and source_object_modification_dt < last_backup_dt:
            return Action.RENAME_LAST
        else:
            return Action.MAKE_BACKUP
    else:
        return Action.SKIP


def backup(backup_config, yt_token, release_type):
    sensors = []
    backup_dir_top = get_backup_dir_top_path(release_type)
    now_dt = datetime.datetime.now(dateutil.tz.gettz('Europe/Moscow'))
    backup_object_name = now_dt.strftime(DT_FORMAT)
    backup_time_attr = f"{now_dt}"

    logger.info('Start backuping YT tables')
    for config_row in backup_config:
        logger.info('Get input data')
        source_path = config_row['source_path']
        backup_dir = yt.ypath_join(backup_dir_top, config_row['backup_subdir'])
        src_cluster = config_row['src_cluster']
        dst_cluster = config_row['dst_cluster']
        backup_period = isodate.parse_duration(config_row['backup_period'])
        backup_latest_regex = config_row.get('backup_latest_regex')
        max_tables = config_row.get('max_tables')

        logger.info('Check clusters')
        if src_cluster != dst_cluster:
            raise RuntimeError(f'[UNIMPLEMENTED] Sorry, copying table `{source_path}` from cluster `{src_cluster}` to cluster `{dst_cluster}` is not implemented')
        yt_client = yt.YtClient(src_cluster, token=yt_token)

        def append_sensor(name, value):
            sensors.append({
                'ts': now_dt.timestamp(),
                'value': value,
                'labels': {
                    'sensor': name,
                    'table': src_cluster + '.' + source_path + ' -> ' + dst_cluster,
                },
            })

        logger.info('Calculate object path (folder or table) to backup')
        if backup_latest_regex is not None:
            backup_latest_compiled = re.compile(backup_latest_regex)
            found = list(yt_client.list(source_path, attributes=['modification_time']))
            found = list(filter(lambda x: backup_latest_compiled.match(x), found))
            found.sort(key=lambda x: x.attributes['modification_time'])
            found.reverse()
            if len(found) == 0:
                raise RuntimeError(f'There is no tables in folder `{source_path}` that match regex `{backup_latest_regex}`')
            source_object = yt.ypath_join(source_path, found[0])
        else:
            source_object = source_path
        source_object_modification_dt = yt_client.get_attribute(source_object, 'modification_time')
        source_object_modification_dt = dateutil.parser.parse(source_object_modification_dt)

        logger.info('Get destination data')
        yt_client.create('map_node', backup_dir, ignore_existing=True, recursive=True)
        destination_object = yt.ypath_join(backup_dir, backup_object_name)
        last_backup_dt = yt_client.get_attribute(backup_dir, 'last_backup_dt', None)
        if last_backup_dt is not None:
            last_backup_dt = dateutil.parser.parse(last_backup_dt)

        action = decide_action(now_dt, source_object_modification_dt, last_backup_dt, backup_period)
        if action == Action.MAKE_BACKUP:
            if last_backup_dt is not None:
                append_sensor('actual_period_seconds', (now_dt - last_backup_dt).total_seconds())

            logger.info('Start backup `%s`, using `%s`', source_path, source_object)
            with yt_client.Transaction():
                if max_tables:
                    table_names = yt_client.list(backup_dir, sort=True, absolute=True)
                    while len(table_names) >= max_tables:
                        min_table_path = table_names[0]
                        logger.info('Tables count limit occurred, remove `%s`', min_table_path)
                        start = time.time()
                        yt_client.remove(min_table_path)
                        append_sensor('remove_duration_seconds', time.time() - start)
                        table_names = table_names[1:]

                start = time.time()
                yt_client.copy(
                    source_object,
                    destination_object,
                    recursive=True,
                    preserve_creation_time=True,
                    preserve_modification_time=True,
                )
                yt_client.set_attribute(destination_object, 'backup_source_object', source_object)
                yt_client.set_attribute(destination_object, 'backup_source_path', source_path)
                append_sensor('copy_duration_seconds', time.time() - start)
            logger.info('Backup in path `%s` using `%s` OK, copied into `%s`', source_path, source_object, destination_object)

            yt_client.set_attribute(backup_dir, 'last_backup_dt', backup_time_attr)
        elif action == Action.RENAME_LAST:
            if last_backup_dt is not None:
                append_sensor('actual_period_seconds', (now_dt - last_backup_dt).total_seconds())

            logger.info('Object `%s` is already backuped, ignoring', source_path)

            yt_client.set_attribute(backup_dir, 'last_backup_dt', backup_time_attr)
        elif action == Action.SKIP:
            logger.info('Path `%s` is skipped', source_path)

    logger.info('Measure total backups size')
    total_backup_size = 0
    for dst_cluster in set(config_row['dst_cluster'] for config_row in backup_config):
        yt_client = yt.YtClient(dst_cluster, token=yt_token)
        total_backup_size += get_yt_folder_size_recursive(backup_dir_top, yt_client)
    sensors.append({
        'ts': now_dt.timestamp(),
        'value': total_backup_size,
        'labels': {
            'sensor': 'backups_size',
        },
    })
    return sensors


def run(yt_token, solomon_token, release_type):
    backup_config = get_backup_config(yt_token, release_type)
    sensors = backup(backup_config, yt_token, release_type)

    logger.info('Push sensors to solomon')
    params = {
        'project': 'bannerland',
        'cluster': release_type,
        'service': 'backup_yt_tables',
    }
    solomon.push_to_solomon_v2(solomon_token, params, sensors, common_labels=())
    logger.info('Pushed %d sensors to solomon', len(sensors))
