#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
import argparse
import json
import datetime
import ylock
from time import sleep

from library.python.svn_version import svn_revision
import yt.wrapper as yt

import irt.common.yt
from irt.bannerland.options import get_option as get_bl_opt, get_cypress_config

from bannerland.make_banners import get_maker_cls
from bannerland.validate_banners import validate_pocket
import bannerland.yt_utils


logging.basicConfig(level=logging.INFO)


class PocketMaker(object):
    def __init__(self, task_type, strict=False, sandbox_task_id=None):
        self.task_type = task_type
        self.strict = strict
        self.sandbox_task_id = sandbox_task_id
        self.yt_client = bannerland.yt_utils.get_yt_client()
        self.cypress_conf = get_cypress_config(task_type)
        self.try_attr = 'make_pocket.try_number'
        self.pocket_name_format = get_bl_opt('bannerland_pocket_name_format')

    def run_make_banners(self, yt_work_dir, input_tao_table, start_step=None):
        logging.info('run make_banners ...')
        catalogia_spec = {
            'lib': {'type': 'local', 'dir': '.'},
            'dicts': {'type': 'local', 'dir': '.'},
        }
        conf_file = './catalogia_resources.json'
        with open(conf_file) as fh:
            catalogia_resources = json.load(fh)
            # валидируем только perllibs, а gendicts берём дефолтные (самые свежие из SB)
            for res in catalogia_resources:
                if res['name'] == 'perllibs':
                    catalogia_spec['perllibs'] = res

        maker_cls = get_maker_cls(self.task_type)
        maker = maker_cls(
            yt_dir=yt_work_dir, yt_config=bannerland.yt_utils.get_yt_config(),
            input_tao_table=input_tao_table, catalogia_spec=catalogia_spec,
        )
        run_kwargs = {}
        if start_step is not None:
            run_kwargs['start'] = start_step
        maker.run(**run_kwargs)
        validate_pocket(self.task_type, yt_work_dir, yt_client=maker.yt_client)
        logging.info('make_banners OK!')

    def move_make_banners_dir_in_bad_archive(self, attr_node, bad_pocket, bad_run_pocker_dir):
        yt_client = self.yt_client
        with yt_client.Transaction():
            prev_crash = yt_client.get_attribute(attr_node, 'last_crash', False)
            if prev_crash:
                if yt_client.exists(prev_crash):
                    logging.info('remove last crash: %s', prev_crash)
                    yt_client.remove(prev_crash, recursive=True)

            tf = self.pocket_name_format
            target_name = yt_client.get_attribute(bad_pocket, 'start_time', datetime.datetime.now().strftime(tf))
            target_path = bad_run_pocker_dir + '/' + target_name
            logging.info('move %s to %s', bad_pocket, target_path)
            yt_client.move(bad_pocket, target_path)
            yt_client.set_attribute(attr_node, 'last_crash', target_path)

    def is_pocket_continued(self, cur_mb_dir):
        yt_client = self.yt_client

        if not yt_client.exists(cur_mb_dir):
            # previous pocket generated ok
            return False

        if yt_client.get_attribute(cur_mb_dir, 'make_pocket.force_continue', None):
            # set manually
            return True

        old_svn_rev = yt_client.get_attribute(cur_mb_dir, 'svn_revision', 0)
        new_svn_rev = svn_revision()

        if old_svn_rev != new_svn_rev:
            return False

        try_no = yt_client.get_attribute(cur_mb_dir, self.try_attr, 0)
        if try_no >= 2:
            return False

        return True

    def make_pocket(self):
        task_type = self.task_type
        yt_client = self.yt_client
        sandbox_task_id = self.sandbox_task_id
        logging.warning('TASK_TYPE: %s', task_type)
        conf = self.cypress_conf
        mb_dir = conf.get_path('make_banners_dir')
        cur_mb_dir = yt.ypath_join(conf.get_path('make_banners_current'), 'blrt')
        archive_dir = conf.get_path('make_banners_archive')
        bad_run_archive_dir = conf.get_path('make_banners_badrun')
        merged_tao_table = mb_dir + '/new_tasks_and_offers'

        # определяем, будем ли продолжать старый карман или генерить новый

        continue_pocket = self.is_pocket_continued(cur_mb_dir)
        if yt_client.exists(cur_mb_dir) and not continue_pocket:
            self.move_make_banners_dir_in_bad_archive(mb_dir, cur_mb_dir, bad_run_archive_dir)

        run_kwargs = {}
        if continue_pocket:
            run_kwargs['start_step'] = 'continue'
            try_no = yt_client.get_attribute(cur_mb_dir, self.try_attr, 0) + 1
            logging.warning('will CONTINUE pocket generation, try %d', try_no)
            yt_client.set_attribute(cur_mb_dir, self.try_attr, try_no)
            if sandbox_task_id is not None:
                sandbox_task_ids = yt_client.get_attribute(cur_mb_dir, "sandbox_task_ids", [])
                sandbox_task_ids.append(sandbox_task_id)
                yt_client.set_attribute(cur_mb_dir, "sandbox_task_ids", sandbox_task_ids)
        else:
            # Создаем руками, чтобы замечать падения до создания этой директории в run_make_banners
            logging.warning('will make NEW pocket')
            yt_client.create('map_node', path=cur_mb_dir, ignore_existing=True, recursive=True)

            run_kwargs['start_step'] = 'filter_offer_duplicates'

            last_imported_blrt_pocket = yt_client.get_attribute(mb_dir, 'last_imported_blrt_pocket', '')
            blrt_import_dir = conf.blrt_import_dir[conf.get_env_mode()]
            valid_pocket_list = list(filter(lambda x: x > last_imported_blrt_pocket, yt_client.list(blrt_import_dir)))
            if valid_pocket_list:
                current_pocket = min(valid_pocket_list)
                current_pocket_dir = yt.ypath_join(blrt_import_dir, current_pocket)
                logging.info('copy %s to %s', current_pocket_dir, cur_mb_dir)
                yt_client.copy(current_pocket_dir, cur_mb_dir, recursive=True, force=True)
                yt_client.set_attribute(cur_mb_dir, 'last_imported_blrt_pocket', current_pocket)

                tf = self.pocket_name_format
                start_time = datetime.datetime.now().strftime(tf)
                yt_client.set_attribute(cur_mb_dir, 'start_time', start_time)
            else:
                logging.info('no new blrt pockets; do nothing')
                logging.info('removing current dir (if empty): %s', cur_mb_dir)
                yt_client.remove(cur_mb_dir, recursive=False)
                return

            if sandbox_task_id is not None:
                yt_client.set_attribute(cur_mb_dir, "sandbox_task_ids", [sandbox_task_id])

        self.run_make_banners(yt_work_dir=cur_mb_dir, input_tao_table=merged_tao_table, **run_kwargs)

        ylock_manager = ylock.create_manager(backend='yt',
                                             prefix=conf.get_path('lock_dir'),
                                             token=bannerland.yt_utils.get_yt_token())

        lock_name = 'move_pocket_to_archive'
        with ylock_manager.lock(lock_name, block=True) as acquired:
            if not acquired:
                raise RuntimeError('Failed to acquire lock: {}'.format(lock_name))

            pocket_timestamp = datetime.datetime.now().strftime(self.pocket_name_format)
            while yt_client.exists(yt.ypath_join(archive_dir, pocket_timestamp)):
                sleep(1)
                pocket_timestamp = datetime.datetime.now().strftime(self.pocket_name_format)

            with yt_client.Transaction():
                yt_client.set_attribute(cur_mb_dir, 'make_banners_mode', 'blrt')

                last_imported_blrt_pocket = yt_client.get_attribute(cur_mb_dir, 'last_imported_blrt_pocket', '')
                yt_client.set_attribute(mb_dir, 'last_imported_blrt_pocket', last_imported_blrt_pocket)

                yt_client.move(cur_mb_dir, yt.ypath_join(archive_dir, pocket_timestamp))
                yt_client.set_attribute(mb_dir, 'last_crash', False)

                logging.info('Set TTL=1d for %s', current_pocket_dir)
                irt.common.yt.set_expiration_time(current_pocket_dir, datetime.timedelta(days=1), yt_client)

    def get_tao_tables(self):
        yt_client = self.yt_client
        conf = self.cypress_conf
        mb_dir = conf.get_path('make_banners_dir')
        tao_dir = conf.get_path('tao_current')

        last_done_table = yt_client.get_attribute(mb_dir, 'last_tao_table', None)
        table_names = yt_client.list(tao_dir, sort=True)

        if last_done_table:
            logging.info('last generated merged table: %s, process only later tables', last_done_table)
            table_names = [t for t in table_names if t > last_done_table]

        # не начинаем генерацию, пока не закешировали аватарки
        last_cached_table = yt_client.get_attribute(tao_dir, 'last_avatars_put_table')
        table_names = [t for t in table_names if t <= last_cached_table]

        return [yt.ypath_join(tao_dir, t) for t in table_names]


def do_dry_run(task_type):
    catalogia_spec = {
        'lib': {'type': 'local', 'dir': '.'},
        'dicts': {'type': 'local', 'dir': '.'},
    }
    maker_cls = get_maker_cls(task_type)
    maker = maker_cls(yt_dir='', yt_config=bannerland.yt_utils.get_yt_config(), input_tao_table=None, catalogia_spec=catalogia_spec)
    catalogia_resources = maker.bmyt.catalogia_resources
    with open('catalogia_resources.json', 'w') as fh:
        json.dump(catalogia_resources, fh)
    with open('_svn_revision.info', 'w') as fh:
        fh.write(str(svn_revision()) + "\n")


if __name__ == '__main__':
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--sandbox-task-id', default=None)
    parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    parser.add_argument('--make-banners-mode', choices=['yt', 'blrt'], default='blrt')
    parser.add_argument('--strict', action='store_true', help='die if not data')
    parser.add_argument('--dry-run', action='store_true')
    args = parser.parse_args()
    if args.dry_run:
        do_dry_run(args.task_type)
    else:
        pocket_maker = PocketMaker(args.task_type, strict=args.strict, sandbox_task_id=args.sandbox_task_id)
        pocket_maker.make_pocket()
