# coding: utf-8

import argparse
import logging

import yt.wrapper as yt

import irt.iron.options as iron_opts

from irt.bannerland.options import get_option as get_bl_opt, get_cypress_config

from bannerland.prepare_input import cache_avatars_for_offer_images
from bannerland.client.avatars import AvatarsClient

from irt.monitoring.solomon.sensors import SolomonAgentSensorsClient
from irt.utils import from_json


def main(task_type):
    logging.basicConfig(level=logging.INFO, format="%(asctime)s\t[%(process)d]\t%(levelname)s\t%(message)s")
    yt.config['proxy']['url'] = 'hahn'
    yt.config['token_path'] = iron_opts.get('yt_token_path')
    yt.config['spec_defaults'] = {
        'pool': 'bannerland-prod',
    }
    yt.config['remote_temp_tables_directory'] = get_bl_opt('yt_temp_dir')
    config = from_json('config_json')[task_type]

    cypress_conf = get_cypress_config(task_type)
    lock_table = yt.ypath_join(cypress_conf.get_path('lock_dir'), 'cache_avatars')
    lock_client = yt.YtClient('hahn', config={'token_path': iron_opts.get('yt_token_path')})
    markov_lookup_client = yt.YtClient('markov', config={'token_path': iron_opts.get('yt_token_path')})
    lock_client.create("table", lock_table, ignore_existing=True, recursive=True)
    hahn_map_reduce_client = yt
    solomon_client = SolomonAgentSensorsClient()

    with lock_client.Transaction():
        try:
            lock_client.lock(lock_table, mode="exclusive")
        except yt.YtHttpResponseError as err:
            logging.error("Fail take yt lock. script is already start. YT msg: %s", err.error)
            return

        put_attr_name = 'last_avatars_put_table'
        arc_dir = cypress_conf.get_path('tao_current')
        last_processed = hahn_map_reduce_client.get_attribute(arc_dir, put_attr_name, None)

        tao_tables = hahn_map_reduce_client.list(arc_dir, sort=True)
        max_tables_to_process = config['max_tables_to_process']
        if last_processed is not None:
            tao_tables = [t for t in tao_tables if t > last_processed]
            tao_tables = tao_tables[:max_tables_to_process]
        else:
            tao_tables = tao_tables[-max_tables_to_process:]
        logging.info('tables to process: %s', tao_tables)

        avatars_client = AvatarsClient.init_by_task_type(
            task_type=task_type,
            lookup_client=markov_lookup_client,
            map_reduce_client=hahn_map_reduce_client,
            avatars_process_count=config['avatars_process_count'],
            avatars_retries=config['avatars_retries'],
            put_images_pack_size=config['put_images_pack_size'],
            init_process_pool=True,
            solomon_client=solomon_client,
        )
        avatars_client.cache.ensure_mounted()
        tao_paths = [yt.ypath_join(arc_dir, t) for t in tao_tables]

        def push_cache_avatars_sensors(sensors):
            for name, val in sensors.items():
                solomon_client.push_single_sensor(
                    cluster='yt_hahn',
                    service='bannerland_yt',
                    sensor='{}.prepare_input.cache_avatars.{}'.format(task_type, name),
                    value=val,
                )

        def push_stats_sensors(stats):
            # (осталось проаватарить) = cache_miss - (сколько уже отправили в функцию аватаризации)
            sensors = {'queue_size': stats['cache_miss'] - stats['function_input']}
            push_cache_avatars_sensors(sensors)

        stats = cache_avatars_for_offer_images(tao_paths, avatars_client=avatars_client, yt_client=hahn_map_reduce_client, stats_hook=push_stats_sensors)

        hahn_map_reduce_client.set_attribute(arc_dir, put_attr_name, tao_tables[-1])
        logging.info('set last processed table: %s', tao_tables[-1])

        sensors = {
            'hit_count': stats['cache_hit'],
            'miss_count': stats['cache_miss'],
        }
        push_cache_avatars_sensors(sensors)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--task-type', choices=['dyn', 'perf'], default='perf')
    args = parser.parse_args()
    main(args.task_type)
