import argparse
import logging

import bannerland.dir_worker.dir_worker
import irt.bannerland.options
import yt.wrapper as yt

import irt.iron.options as iron_opts

log = logging.getLogger(__name__)


def setup_logging(debug=False):
    level = logging.DEBUG if debug else logging.INFO
    logging.basicConfig(level=level, format="%(asctime)s\t[%(process)d]\t%(levelname)s\t%(message)s")


def parse_arguments():
    argparser = argparse.ArgumentParser()
    argparser.add_argument('--task-type', choices=['dyn', 'perf'], help='subj', required=True)
    argparser.add_argument('--type', choices=['full_state', 'make_banners'], help='Cleaner configuration to run', required=True)
    argparser.add_argument('--dry-run', help='Do not delete anything, just print logs', action="store_true")
    argparser.add_argument('--debug', help='Print more logs', action="store_true")
    return argparser.parse_args()


def get_cleaner_config(cluster, run_type, is_prod, task_type):
    if cluster != 'hahn' or not is_prod:
        return irt.bannerland.options.get_option('cleaner_strict_config')
    elif run_type == 'make_banners':
        return irt.bannerland.options.get_option('make_banners_cleaner_config')
    elif run_type == 'full_state':
        return irt.bannerland.options.get_option('full_state_cleaner_config')
    else:
        raise ValueError('bad run_type arg={}'.format(run_type))


def get_cypress_name_by_run_type(run_type):
    d = {
        'make_banners': 'make_banners_archive',
        'full_state': 'full_state_archive',
    }
    return d[run_type]


def run_clear_on_cluster(cluster, args):
    cypress_config = irt.bannerland.options.get_cypress_config(args.task_type)
    is_prod = bool(cypress_config.mode == 'prod')

    yt.config['proxy']['url'] = cluster
    cleaner_config = get_cleaner_config(cluster, args.type, is_prod, args.task_type)
    cypress_coufig_option_name = get_cypress_name_by_run_type(args.type)
    clear_dir = cypress_config.get_path(cypress_coufig_option_name)

    log.info("Cleaning %s on %s", args.type, cluster)
    worker = bannerland.dir_worker.dir_worker.YtCleaner(
        yt_dir=clear_dir,
        cleaner_config=cleaner_config,
    )
    worker.run(args.dry_run)


def main():
    args = parse_arguments()
    setup_logging(args.debug)

    yt.config['token_path'] = iron_opts.get('yt_token_path')

    clusters = []
    if args.type == 'full_state':
        clusters = ['arnold', 'hahn']
    elif args.type == "make_banners":
        clusters = ['hahn']

    for cluster in clusters:
        run_clear_on_cluster(cluster, args)


if __name__ == '__main__':
    main()
