import yt.wrapper as yt

import irt.iron.options as iron_opts

from bannerland.dir_worker.dir_worker import YtDirWorker
from irt.bannerland.options import get_cypress_config
from bannerland.yql.tools import get_client as get_yql_client
from irt.monitoring.solomon.sensors import SolomonAgentSensorsClient
from irt.bannerland.proto.common import extensions_pb2
from irt.bannerland.proto.offer import offer_pb2 as offer_proto
from irt.common import proto_utils
import six.moves.urllib.parse


OFFER_SOURCE_MAPPING = proto_utils.get_proto_enum_value_mapping(offer_proto.EOfferSource, extensions_pb2.StringRepresentation)
FEED_TYPE_MAPPING = proto_utils.get_proto_enum_value_mapping(offer_proto.EDataType, extensions_pb2.StringRepresentation)
FS_NAME_DATE_FORMAT = '%Y-%m-%d_%H:%M:%S'


def convert_offer_source_str(offer_source):
    return OFFER_SOURCE_MAPPING.get(offer_source)


def convert_task_feed_type_str(feed_type):
    return FEED_TYPE_MAPPING.get(feed_type)


def get_attr_name(worker_name):
    return 'archive_worker_' + worker_name


def get_order_id_from_task_id(task_id):
    order_id, _ = task_id.split('_', 1)
    return int(order_id)


def get_offer_source_from_feed_url(feed_url):
    if feed_url is None:
        return offer_proto.EOfferSource.OFFER_SOURCE_FEED

    parsed = six.moves.urllib.parse.urlparse(feed_url)
    domain = parsed.netloc
    if 'market.feed' == domain and 'site' in six.moves.urllib.parse.parse_qs(parsed.query):
        return offer_proto.EOfferSource.OFFER_SOURCE_SITE

    return offer_proto.EOfferSource.OFFER_SOURCE_FEED


class BLYTWorker(YtDirWorker):
    def __init__(self, task_type, yt_pool='bannerland-data', yql_client=None, use_monitorings=True, **kwargs):
        if not kwargs.get('yt_client'):
            kwargs['yt_client'] = self.get_yt_client(yt_pool)
        super(BLYTWorker, self).__init__(**kwargs)
        self.task_type = task_type
        self.dt_format = FS_NAME_DATE_FORMAT
        self.yt_pool = yt_pool
        self.yt_cluster = self.yt_client.config['proxy']['url'].split('.')[0]
        self.yql_client = yql_client
        if self.yql_client is None:
            self.yql_client = get_yql_client(self.yt_cluster)
        self.use_monitorings = use_monitorings
        if self.use_monitorings:
            if 'hahn' in self.yt_cluster:
                self.solomon_cluster = 'yt_hahn'
            elif 'arnold' in self.yt_cluster:
                self.solomon_cluster = 'yt_arnold'
            else:
                raise ValueError("can't get solomon cluster for yt cluster %s", self.yt_cluster)

    def do_work_wrapper(self, node):
        archive_worker_res = 0
        try:
            super(BLYTWorker, self).do_work_wrapper(node)
            archive_worker_res = 1
        except Exception:
            raise
        finally:
            if self.use_monitorings:
                SolomonAgentSensorsClient().push_single_sensor(
                    cluster=self.solomon_cluster,
                    service='bannerland_yt',
                    sensor='archive_worker_ok',
                    labels={
                        'task_type': self.task_type,
                        'archive_worker': self.attr_name,
                        'cypress_mode': self.get_cypress_config().mode,
                    },
                    value=archive_worker_res,
                )

    def get_cypress_config(self):
        return get_cypress_config(self.task_type)

    def get_yt_client(self, yt_pool):
        return yt.YtClient(
            proxy=iron_opts.get('yt_default_proxy'),
            config={
                'token_path': iron_opts.get('yt_token_path'),
                'spec_defaults': {
                    'pool': yt_pool,
                },
            },
        )
