# coding: utf-8

import datetime
import logging
import json

import yt.wrapper as yt

from bannerland.yql.tools import do_yql
from irt.bannerland.options import get_option as get_bl_opt

import bannerland.archive_workers.common as aw_common
import irt.bannerland.switcher_filter
import irt.common.yt


def get_timestamp():
    return datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')


def add_values(main, add):
    for k, v in add.items():
        if isinstance(v, dict):
            main.setdefault(k, {})
            add_values(main[k], v)
        else:
            main[k] = main.get(k, 0) + v


class FunnelWorker(aw_common.BLYTWorker):
    def __init__(self, attr_name, info_dyntable, **kwargs):
        super(FunnelWorker, self).__init__(attr_name=attr_name, **kwargs)
        self.info_dyntable = info_dyntable
        if self.task_type == 'perf':
            self.funnel_str_conf = get_bl_opt('base_perf_funnel_str')
        elif self.task_type == 'dyn':
            self.funnel_str_conf = get_bl_opt('dyn_funnel_str')
        else:
            raise ValueError('not supported task_type = {}'.format(self.task_type))

        yt_root = self.get_cypress_config().root
        self.ignore_grut_orders = irt.common.yt.get_attribute(yt_root, 'ignore_grut_orders', self.yt_client, default=0)

    def get_banners_count(self, input_table):
        with self.yt_client.TempTable() as tmp:
            yql_query = """
                PRAGMA yson.DisableStrict = '1';

                INSERT INTO `{output}` WITH TRUNCATE
                SELECT
                    task_id,
                    offer_source,
                    COUNT(DISTINCT BannerID) as banners
                FROM
                    `{input}`
                GROUP BY
                    task_id,
                    Yson::LookupString(BLBannerDetails, "offer_source") ?? "feed" as offer_source
            """.format(input=input_table, output=tmp)
            banners_count = {}
            do_yql(self.yql_client, yql_query)
            for row in self.yt_client.read_table(tmp):
                task_counts = banners_count.setdefault(row['task_id'], {})
                task_counts[row['offer_source']] = row['banners']
        return banners_count

    def get_funnel_for_source(self, pocket_dir):
        funnel_for_source = {}
        task_feed_types = {}

        tasks_table = yt.TablePath(pocket_dir + '/tasks', columns=['task_id', 'export_offers_info', 'task_inf'])
        tid_to_order_shop = {}
        for row in self.yt_client.read_table(tasks_table):
            tid = row['task_id']
            task_inf_json = json.loads(row['task_inf'])
            order_id = int(task_inf_json['OrderID'])
            export_offers_info = json.loads(row['export_offers_info'])
            tid_to_order_shop[tid] = (order_id, task_inf_json['ShopID'])
            funnel_for_source[(tid, order_id, task_inf_json['ShopID'])] = export_offers_info.get('funnel_for_source', {})
            if 'LastValidFeedType' in task_inf_json['Resource']:
                task_feed_types[tid] = task_inf_json['Resource']['LastValidFeedType']

        def filter_funnel(row):
            if row['ns'] == 'funnel_for_source':
                yield row

        def reduce_funnel(key, rows):
            total = {}
            for row in rows:
                task_id = row['task_id']
                add_values(total, json.loads(row['log']))

            yield {'task_id': task_id, 'funnel_for_source': total}

        with self.yt_client.TempTable() as tmp:
            self.yt_client.run_map_reduce(
                filter_funnel,
                reduce_funnel,
                yt.ypath_join(pocket_dir, 'log_merged'),
                tmp,
                reduce_by=['task_id'],
            )
            for row in self.yt_client.read_table(tmp):
                if row['task_id'] not in tid_to_order_shop:
                    logging.error('No task id %s in task table', row['task_id'])
                    continue
                order_id, shop_id = tid_to_order_shop[row['task_id']]
                task_funnel = funnel_for_source[(row['task_id'], order_id, shop_id)]
                add_values(task_funnel, row['funnel_for_source'])

        steps = ['add_avatars', 'end']
        for step in steps:
            if step == 'end':
                table_name ='generated_banners.final'
                funnel_name = 'banners_in_pocket'
            else:
                table_name = 'make_banners.{}.done'.format(step)
                funnel_name = 'banners_after_{}'.format(step)
            input_table = yt.ypath_join(pocket_dir, table_name)
            for tid, counts in self.get_banners_count(input_table).items():
                if tid not in tid_to_order_shop:
                    logging.error('No task id %s in task table', tid)
                    continue
                order_id, shop_id = tid_to_order_shop[tid]
                task_funnel = funnel_for_source[(tid, order_id, shop_id)]
                for offer_source, count in counts.items():
                    add_values(task_funnel, {offer_source: {funnel_name: count}})

        # temp fix for empty key
        for task_funnel in funnel_for_source.values():
            if '' in task_funnel:
                add_values(task_funnel, {'feed': task_funnel.pop('')})

        return funnel_for_source, task_feed_types

    def get_phrase_src_count(self, pocket_dir):
        phrase_src_count = {}
        with self.yt_client.TempTable() as tmp:
            query = """
                insert into `{output_table}` with truncate
                SELECT
                    COUNT(1) as count,
                    bl_phrase_template_type,
                    OrderID,
                    ShopId,
                    task_id
                FROM `{generated_banners_table}`
                GROUP BY task_id, OrderID, ShopId, bl_phrase_template_type
            """.format(generated_banners_table=pocket_dir + '/generated_banners.final', output_table=tmp)
            do_yql(self.yql_client, query, yt_pool=self.yt_pool)
            for row in self.yt_client.read_table(tmp):
                tid = (row['task_id'], row['OrderID'], row.get('ShopId'))
                if tid not in phrase_src_count:
                    phrase_src_count[tid] = {}

                letter = row['bl_phrase_template_type']
                phrase_src_count[tid][letter] = row['count']

        return phrase_src_count

    def is_for_grut(self, shop_id, order_id, offer_source, feed_type):
        irt.bannerland.switcher_filter.is_for_grut(order_id, shop_id, self.task_type, offer_source, feed_type)

    def get_funnel_rows(self, pocket_dir):
        pocket = yt.ypath_split(pocket_dir)[1]

        funnel_for_source, task_feed_types = self.get_funnel_for_source(pocket_dir)
        phrase_src_count = self.get_phrase_src_count(pocket_dir)

        update_time = get_timestamp()

        def get_task_source(tid, funnel_for_source):
            source = 'feed'
            if tid in funnel_for_source:
                task_funnel = funnel_for_source[tid]
                if 'site' in task_funnel:
                    source = 'site'
            return aw_common.convert_offer_source_str(source)

        def get_task_feed_type(task_id, task_feed_types):
            if task_id in task_feed_types:
                return aw_common.convert_task_feed_type_str(task_feed_types[task_id])

        bs_rows = []
        for tid_order_shop, tid_phrase_src_counts in phrase_src_count.items():
            task_id, order_id, shop_id = tid_order_shop
            offer_source = get_task_source(tid_order_shop, funnel_for_source)
            feed_type = get_task_feed_type(task_id, task_feed_types)
            if not (self.ignore_grut_orders and self.is_for_grut(order_id=order_id, shop_id=shop_id, offer_source=offer_source, feed_type=feed_type)):
                bs_rows.append({
                    'task_id': task_id,
                    'property': 'phrase_src_count',
                    'value': tid_phrase_src_counts,
                    'pocket': pocket,
                    'update_time': update_time,
                })

        funnel_rows = []
        for tid_order_shop, task_funnel in funnel_for_source.items():
            task_id, order_id, shop_id = tid_order_shop
            offer_source = get_task_source(tid_order_shop, funnel_for_source)
            feed_type = get_task_feed_type(task_id, task_feed_types)
            if not (self.ignore_grut_orders and self.is_for_grut(order_id=order_id, shop_id=shop_id, offer_source=offer_source, feed_type=feed_type)):
                funnel_rows.append({
                    'task_id': task_id,
                    'property': 'funnel_str',
                    'value': self.format_funnel_str(
                        funnel_for_source=task_funnel,
                        phrase_src_count=phrase_src_count.get(tid_order_shop, {}),
                    ),
                    'pocket': pocket,
                    'update_time': update_time,
                })

        return funnel_rows + bs_rows

    def save_new_tasks(self, pocket_dir):
        yql_query = '''
            INSERT INTO `{pocket_dir}/tasks.new` WITH TRUNCATE
            SELECT task_id
            from `{pocket_dir}/tasks.final` as o
            LEFT ONLY JOIN `{info_dyntable}`
            USING (task_id)
        '''.format(
            pocket_dir=pocket_dir,
            info_dyntable=self.info_dyntable,
        )
        do_yql(self.yql_client, yql_query, yt_pool=self.yt_pool)

    def do_work(self, pocket_dir):
        self.save_new_tasks(pocket_dir)

        logging.warning('do_work for %s', pocket_dir)
        rows_to_insert = self.get_funnel_rows(pocket_dir)
        pack_size = 1000
        while rows_to_insert:
            pack_to_insert = rows_to_insert[:pack_size]
            rows_to_insert = rows_to_insert[pack_size:]
            self.yt_client.insert_rows(self.info_dyntable, pack_to_insert)

    def format_funnel_str(self, funnel_for_source, phrase_src_count):
        funnel_info = list()
        funnel_info.append('funnel:')

        offer_sources = sorted(funnel_for_source.keys())
        funnel_info.append('- offer sources: ' + ', '.join(offer_sources))

        last_funnel_key = self.funnel_str_conf['funnel_for_source_order'][-1]
        last_funnel_nonzero = any(src_funnel.get(last_funnel_key) for src_funnel in funnel_for_source.values())

        for funnel_key in self.funnel_str_conf['funnel_for_source_order']:
            count = {src: funnel_for_source[src].get(funnel_key, 0) for src in offer_sources}
            val = sum(count.values())
            if val == 0 and last_funnel_nonzero:
                val = '?'   # могут возникать нули при добавлении в воронку новых полей

            active_sources = [src for src in offer_sources if count[src]]
            if len(offer_sources) > 1 and active_sources:
                val_by_source = ' (' + ', '.join(['{}: {}'.format(src, count[src]) for src in active_sources]) + ')'
            else:
                val_by_source = ''
            funnel_info.append('* {val} {key}{val_by_source}'.format(
                key=funnel_key,
                val=val,
                val_by_source=val_by_source,
            ))

        key_readable_mapping = self.funnel_str_conf['phrase_src_count_key_mapping']
        funnel_info.append('phrase_src_count:')
        letters = set(phrase_src_count.keys()) | set(key_readable_mapping.keys())
        for letter in sorted(letters):
            key = key_readable_mapping.get(letter, 'phrases_letter_{}_count'.format(letter))
            funnel_info.append('- {key}: {val}'.format(
                key=key,
                val=str(phrase_src_count.get(letter, 0))
            ))

        return '\n'.join(funnel_info)
