# coding: utf-8

import datetime
import time

import yt.wrapper as yt

import irt.bannerland.options
import irt.logging
from irt.monitoring.solomon import sensors as solomon_sensors
from bm import yt_tools

from bannerland.archive_workers import transfer

logger = irt.logging.getLogger(irt.logging.BANNERLAND_PROJECT, __name__)


class PublishWorker(transfer.TransferWorker):
    class Mode(object):
        full = 1
        publish_local = 2
        transfer_and_publish = 3

    def __init__(self, publish_mode=Mode.full, **kwargs):
        super(PublishWorker, self).__init__(**kwargs)
        self.publish_mode = publish_mode

    def publish(self, src_path, target_dir, version=None, version_attr=None, copy_attrs=(), link_dir=None, sensor_name=None):
        """Copy src_path (table or map_node) to target_dir/version on all clusters."""
        if self.publish_mode != PublishWorker.Mode.publish_local:
            self.transfer_node(src_path, copy_attrs=copy_attrs)

        yt_client = self.yt_client
        if version is None:
            version = yt_tools.get_attribute(src_path, version_attr, yt_client=yt_client)
        target_path = yt.ypath_join(target_dir, version)
        link_path = yt.ypath_join(link_dir, 'last_published') if (link_dir is not None) else None
        self.copy(src_path, target_path=target_path, link_path=link_path)

        if sensor_name is not None:
            self.monitor(src_path, sensor_name=sensor_name)

    def get_publish_clients(self):
        if self.publish_mode == PublishWorker.Mode.full:
            return [self.yt_client] + [r['client'] for r in self.get_replicas()]
        elif self.publish_mode == PublishWorker.Mode.publish_local:
            return [self.yt_client]
        elif self.publish_mode == PublishWorker.Mode.transfer_and_publish:
            return [r['client'] for r in self.get_replicas()]
        else:
            raise ValueError("Unknown publish_mode")

    def set_dir_worker_attr_done(self, node, start_time):
        """Set worker attr on main cluster and all replicas."""
        timings = self.get_timings(start_time)
        for yt_client in self.get_publish_clients() + [self.yt_client]:
            yt_tools.set_attribute(node, self.attr_name, timings, yt_client=yt_client)

    def copy(self, src_path, target_path, link_path=None):
        publish_clients = self.get_publish_clients()

        # поскольку у нас нет междатацентровой репликации, но крайне желательно иметь везде одинаковые данные,
        # проверим ещё раз, что клиенты работают и всё хорошо
        paths = [target_path]
        if link_path is not None:
            paths.append(link_path)
        for yt_client in publish_clients:
            for path in paths:
                yt_client.create('map_node', yt.ypath_dirname(path), recursive=True, ignore_existing=True)

        # есть два варианта - делать линк, или копировать + ставить атрибут
        # делаем то и другое: линк для удобства, использования через yql и т.п; копируем в экспортную директорию для прода
        for yt_client in publish_clients:
            with yt_client.Transaction():
                yt_client.copy(src_path, target_path, force=True)
                if link_path is not None:
                    yt_client.link(src_path, link_path, force=True)
            logger.info('published full state: %s => %s (at %s)', src_path, target_path, yt_client.config['proxy']['url'])

    def monitor(self, fs_path, sensor_name):
        """Send to solomon timestamp of published node."""
        fs_time = datetime.datetime.strptime(fs_path.split('/')[-2], irt.bannerland.options.get_option('bannerland_pocket_name_format'))
        fs_time += datetime.timedelta(0, time.timezone)
        yt_cluster = self.yt_client.config['proxy']['url'].split('.')[0]
        solomon_client = solomon_sensors.SolomonAgentSensorsClient()
        solomon_client.push_single_sensor(
            cluster="yt_{}".format(yt_cluster),
            service="bannerland_yt",
            sensor=sensor_name,
            value=1,
            ts_datetime=str(fs_time),
        )
